/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.protocol.datatransfer.sasl;

import com.facebook.presto.hadoop.$internal.com.google.common.base.Charsets;
import com.facebook.presto.hadoop.$internal.com.google.common.collect.Lists;
import com.facebook.presto.hadoop.$internal.org.apache.commons.codec.binary.Base64;
import com.facebook.presto.hadoop.$internal.org.slf4j.Logger;
import com.facebook.presto.hadoop.$internal.org.slf4j.LoggerFactory;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Map;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.sasl.AuthorizeCallback;
import javax.security.sasl.RealmCallback;
import javax.security.sasl.SaslException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.crypto.CipherOption;
import org.apache.hadoop.hdfs.net.Peer;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.datatransfer.IOStreamPair;
import org.apache.hadoop.hdfs.protocol.datatransfer.InvalidEncryptionKeyException;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.DataTransferSaslUtil;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.InvalidMagicNumberException;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.SaslParticipant;
import org.apache.hadoop.hdfs.protocol.proto.DataTransferProtos;
import org.apache.hadoop.hdfs.security.token.block.BlockPoolTokenSecretManager;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.server.datanode.DNConf;
import org.apache.hadoop.security.SaslPropertiesResolver;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;

@InterfaceAudience.Private
public class SaslDataTransferServer {
    private static final Logger LOG = LoggerFactory.getLogger(SaslDataTransferServer.class);
    private final BlockPoolTokenSecretManager blockPoolTokenSecretManager;
    private final DNConf dnConf;

    public SaslDataTransferServer(DNConf dnConf, BlockPoolTokenSecretManager blockPoolTokenSecretManager) {
        this.blockPoolTokenSecretManager = blockPoolTokenSecretManager;
        this.dnConf = dnConf;
    }

    public IOStreamPair receive(Peer peer, OutputStream underlyingOut, InputStream underlyingIn, int xferPort, DatanodeID datanodeId) throws IOException {
        if (this.dnConf.getEncryptDataTransfer()) {
            LOG.debug("SASL server doing encrypted handshake for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return this.getEncryptedStreams(peer, underlyingOut, underlyingIn);
        }
        if (!UserGroupInformation.isSecurityEnabled()) {
            LOG.debug("SASL server skipping handshake in unsecured configuration for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        if (SecurityUtil.isPrivilegedPort(xferPort)) {
            LOG.debug("SASL server skipping handshake in secured configuration for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        if (this.dnConf.getSaslPropsResolver() != null) {
            LOG.debug("SASL server doing general handshake for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return this.getSaslStreams(peer, underlyingOut, underlyingIn);
        }
        if (this.dnConf.getIgnoreSecurePortsForTesting()) {
            LOG.debug("SASL server skipping handshake in secured configuration with no SASL protection configured for peer = {}, datanodeId = {}", (Object)peer, (Object)datanodeId);
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        throw new IOException(String.format("Cannot create a secured connection if DataNode listens on unprivileged port (%d) and no protection is defined in configuration property %s.", datanodeId.getXferPort(), "dfs.data.transfer.protection"));
    }

    private IOStreamPair getEncryptedStreams(Peer peer, OutputStream underlyingOut, InputStream underlyingIn) throws IOException {
        if (peer.hasSecureChannel() || this.dnConf.getTrustedChannelResolver().isTrusted(DataTransferSaslUtil.getPeerAddress(peer))) {
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        Map<String, String> saslProps = DataTransferSaslUtil.createSaslPropertiesForEncryption(this.dnConf.getEncryptionAlgorithm());
        if (LOG.isDebugEnabled()) {
            LOG.debug("Server using encryption algorithm " + this.dnConf.getEncryptionAlgorithm());
        }
        SaslServerCallbackHandler callbackHandler = new SaslServerCallbackHandler(new PasswordFunction(){

            @Override
            public char[] apply(String userName) throws IOException {
                return DataTransferSaslUtil.encryptionKeyToPassword(SaslDataTransferServer.this.getEncryptionKeyFromUserName(userName));
            }
        });
        return this.doSaslHandshake(underlyingOut, underlyingIn, saslProps, callbackHandler);
    }

    private byte[] getEncryptionKeyFromUserName(String userName) throws IOException {
        String[] nameComponents = userName.split(" ");
        if (nameComponents.length != 3) {
            throw new IOException("Provided name '" + userName + "' has " + nameComponents.length + " components instead of the expected 3.");
        }
        int keyId = Integer.parseInt(nameComponents[0]);
        String blockPoolId = nameComponents[1];
        byte[] nonce = Base64.decodeBase64(nameComponents[2]);
        return this.blockPoolTokenSecretManager.retrieveDataEncryptionKey(keyId, blockPoolId, nonce);
    }

    private IOStreamPair getSaslStreams(Peer peer, OutputStream underlyingOut, InputStream underlyingIn) throws IOException {
        if (peer.hasSecureChannel() || this.dnConf.getTrustedChannelResolver().isTrusted(DataTransferSaslUtil.getPeerAddress(peer))) {
            return new IOStreamPair(underlyingIn, underlyingOut);
        }
        SaslPropertiesResolver saslPropsResolver = this.dnConf.getSaslPropsResolver();
        Map<String, String> saslProps = saslPropsResolver.getServerProperties(DataTransferSaslUtil.getPeerAddress(peer));
        SaslServerCallbackHandler callbackHandler = new SaslServerCallbackHandler(new PasswordFunction(){

            @Override
            public char[] apply(String userName) throws IOException {
                return SaslDataTransferServer.this.buildServerPassword(userName);
            }
        });
        return this.doSaslHandshake(underlyingOut, underlyingIn, saslProps, callbackHandler);
    }

    private char[] buildServerPassword(String userName) throws IOException {
        BlockTokenIdentifier identifier = this.deserializeIdentifier(userName);
        byte[] tokenPassword = this.blockPoolTokenSecretManager.retrievePassword(identifier);
        return new String(Base64.encodeBase64(tokenPassword, false), Charsets.UTF_8).toCharArray();
    }

    private BlockTokenIdentifier deserializeIdentifier(String str) throws IOException {
        BlockTokenIdentifier identifier = new BlockTokenIdentifier();
        identifier.readFields(new DataInputStream(new ByteArrayInputStream(Base64.decodeBase64(str))));
        return identifier;
    }

    private IOStreamPair doSaslHandshake(OutputStream underlyingOut, InputStream underlyingIn, Map<String, String> saslProps, CallbackHandler callbackHandler) throws IOException {
        DataInputStream in = new DataInputStream(underlyingIn);
        DataOutputStream out = new DataOutputStream(underlyingOut);
        SaslParticipant sasl = SaslParticipant.createServerSaslParticipant(saslProps, callbackHandler);
        int magicNumber = in.readInt();
        if (magicNumber != -559038737) {
            throw new InvalidMagicNumberException(magicNumber, this.dnConf.getEncryptDataTransfer());
        }
        try {
            byte[] remoteResponse = DataTransferSaslUtil.readSaslMessage(in);
            byte[] localResponse = sasl.evaluateChallengeOrResponse(remoteResponse);
            DataTransferSaslUtil.sendSaslMessage(out, localResponse);
            ArrayList<CipherOption> cipherOptions = Lists.newArrayList();
            remoteResponse = DataTransferSaslUtil.readSaslMessageAndNegotiationCipherOptions(in, cipherOptions);
            localResponse = sasl.evaluateChallengeOrResponse(remoteResponse);
            DataTransferSaslUtil.checkSaslComplete(sasl, saslProps);
            CipherOption cipherOption = null;
            if (sasl.isNegotiatedQopPrivacy() && (cipherOption = DataTransferSaslUtil.negotiateCipherOption(this.dnConf.getConf(), cipherOptions)) != null && LOG.isDebugEnabled()) {
                LOG.debug("Server using cipher suite " + cipherOption.getCipherSuite().getName());
            }
            DataTransferSaslUtil.sendSaslMessageAndNegotiatedCipherOption(out, localResponse, DataTransferSaslUtil.wrap(cipherOption, sasl));
            return cipherOption != null ? DataTransferSaslUtil.createStreamPair(this.dnConf.getConf(), cipherOption, underlyingOut, underlyingIn, true) : sasl.createStreamPair(out, in);
        }
        catch (IOException ioe) {
            if (ioe instanceof SaslException && ioe.getCause() != null && ioe.getCause() instanceof InvalidEncryptionKeyException) {
                SaslDataTransferServer.sendInvalidKeySaslErrorMessage(out, ioe.getCause().getMessage());
            } else {
                DataTransferSaslUtil.sendGenericSaslErrorMessage(out, ioe.getMessage());
            }
            throw ioe;
        }
    }

    private static void sendInvalidKeySaslErrorMessage(DataOutputStream out, String message) throws IOException {
        DataTransferSaslUtil.sendSaslMessage(out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR_UNKNOWN_KEY, null, message);
    }

    private static final class SaslServerCallbackHandler
    implements CallbackHandler {
        private final PasswordFunction passwordFunction;

        public SaslServerCallbackHandler(PasswordFunction passwordFunction) {
            this.passwordFunction = passwordFunction;
        }

        @Override
        public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
            NameCallback nc = null;
            PasswordCallback pc = null;
            AuthorizeCallback ac = null;
            for (Callback callback : callbacks) {
                if (callback instanceof AuthorizeCallback) {
                    ac = (AuthorizeCallback)callback;
                    continue;
                }
                if (callback instanceof PasswordCallback) {
                    pc = (PasswordCallback)callback;
                    continue;
                }
                if (callback instanceof NameCallback) {
                    nc = (NameCallback)callback;
                    continue;
                }
                if (callback instanceof RealmCallback) continue;
                throw new UnsupportedCallbackException(callback, "Unrecognized SASL DIGEST-MD5 Callback: " + callback);
            }
            if (pc != null) {
                pc.setPassword(this.passwordFunction.apply(nc.getDefaultName()));
            }
            if (ac != null) {
                ac.setAuthorized(true);
                ac.setAuthorizedID(ac.getAuthorizationID());
            }
        }
    }

    private static interface PasswordFunction {
        public char[] apply(String var1) throws IOException;
    }
}

