/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.hive;

import com.google.common.collect.ImmutableList;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

public class RetryDriver {
    private static final Logger log = Logger.get(RetryDriver.class);
    private static final int DEFAULT_RETRY_ATTEMPTS = 10;
    private static final Duration DEFAULT_SLEEP_TIME = Duration.valueOf((String)"1s");
    private static final Duration DEFAULT_MAX_RETRY_TIME = Duration.valueOf((String)"30s");
    private static final double DEFAULT_SCALE_FACTOR = 2.0;
    private final int maxAttempts;
    private final Duration minSleepTime;
    private final Duration maxSleepTime;
    private final double scaleFactor;
    private final Duration maxRetryTime;
    private final Function<Exception, Exception> exceptionMapper;
    private final List<Class<? extends Exception>> exceptionWhiteList;
    private final Optional<Runnable> retryRunnable;

    private RetryDriver(int maxAttempts, Duration minSleepTime, Duration maxSleepTime, double scaleFactor, Duration maxRetryTime, Function<Exception, Exception> exceptionMapper, List<Class<? extends Exception>> exceptionWhiteList, Optional<Runnable> retryRunnable) {
        this.maxAttempts = maxAttempts;
        this.minSleepTime = minSleepTime;
        this.maxSleepTime = maxSleepTime;
        this.scaleFactor = scaleFactor;
        this.maxRetryTime = maxRetryTime;
        this.exceptionMapper = exceptionMapper;
        this.exceptionWhiteList = exceptionWhiteList;
        this.retryRunnable = retryRunnable;
    }

    private RetryDriver() {
        this(10, DEFAULT_SLEEP_TIME, DEFAULT_SLEEP_TIME, 2.0, DEFAULT_MAX_RETRY_TIME, Function.identity(), (List<Class<? extends Exception>>)ImmutableList.of(), Optional.empty());
    }

    public static RetryDriver retry() {
        return new RetryDriver();
    }

    public final RetryDriver maxAttempts(int maxAttempts) {
        return new RetryDriver(maxAttempts, this.minSleepTime, this.maxSleepTime, this.scaleFactor, this.maxRetryTime, this.exceptionMapper, this.exceptionWhiteList, this.retryRunnable);
    }

    public final RetryDriver exponentialBackoff(Duration minSleepTime, Duration maxSleepTime, Duration maxRetryTime, double scaleFactor) {
        return new RetryDriver(this.maxAttempts, minSleepTime, maxSleepTime, scaleFactor, maxRetryTime, this.exceptionMapper, this.exceptionWhiteList, this.retryRunnable);
    }

    public final RetryDriver onRetry(Runnable retryRunnable) {
        return new RetryDriver(this.maxAttempts, this.minSleepTime, this.maxSleepTime, this.scaleFactor, this.maxRetryTime, this.exceptionMapper, this.exceptionWhiteList, Optional.ofNullable(retryRunnable));
    }

    public final RetryDriver exceptionMapper(Function<Exception, Exception> exceptionMapper) {
        return new RetryDriver(this.maxAttempts, this.minSleepTime, this.maxSleepTime, this.scaleFactor, this.maxRetryTime, exceptionMapper, this.exceptionWhiteList, this.retryRunnable);
    }

    @SafeVarargs
    public final RetryDriver stopOn(Class<? extends Exception> ... classes) {
        Objects.requireNonNull(classes, "classes is null");
        ImmutableList exceptions = ImmutableList.builder().addAll(this.exceptionWhiteList).addAll(Arrays.asList(classes)).build();
        return new RetryDriver(this.maxAttempts, this.minSleepTime, this.maxSleepTime, this.scaleFactor, this.maxRetryTime, this.exceptionMapper, (List<Class<? extends Exception>>)exceptions, this.retryRunnable);
    }

    public RetryDriver stopOnIllegalExceptions() {
        return this.stopOn(NullPointerException.class, IllegalStateException.class, IllegalArgumentException.class);
    }

    public <V> V run(String callableName, Callable<V> callable) throws Exception {
        Objects.requireNonNull(callableName, "callableName is null");
        Objects.requireNonNull(callable, "callable is null");
        ArrayList<Throwable> suppressedExceptions = new ArrayList<Throwable>();
        long startTime = System.nanoTime();
        int attempt = 0;
        while (true) {
            if (++attempt > 1) {
                this.retryRunnable.ifPresent(Runnable::run);
            }
            try {
                return callable.call();
            }
            catch (Exception e) {
                e = this.exceptionMapper.apply(e);
                for (Class<? extends Exception> clazz : this.exceptionWhiteList) {
                    if (!clazz.isInstance(e)) continue;
                    RetryDriver.addSuppressed(e, suppressedExceptions);
                    throw e;
                }
                if (attempt >= this.maxAttempts || Duration.nanosSince((long)startTime).compareTo(this.maxRetryTime) >= 0) {
                    RetryDriver.addSuppressed(e, suppressedExceptions);
                    throw e;
                }
                log.debug("Failed on executing %s with attempt %d, will retry. Exception: %s", new Object[]{callableName, attempt, e.getMessage()});
                suppressedExceptions.add(e);
                int delayInMs = (int)Math.min((double)this.minSleepTime.toMillis() * Math.pow(this.scaleFactor, attempt - 1), (double)this.maxSleepTime.toMillis());
                int jitter = ThreadLocalRandom.current().nextInt(Math.max(1, (int)((double)delayInMs * 0.1)));
                try {
                    TimeUnit.MILLISECONDS.sleep(delayInMs + jitter);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    RuntimeException exception = new RuntimeException(ie);
                    RetryDriver.addSuppressed(exception, suppressedExceptions);
                    throw exception;
                }
            }
        }
    }

    private static void addSuppressed(Exception exception, List<Throwable> suppressedExceptions) {
        for (Throwable suppressedException : suppressedExceptions) {
            if (exception == suppressedException) continue;
            exception.addSuppressed(suppressedException);
        }
    }
}

