/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.orc;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.primitives.Longs;
import io.airlift.units.DataSize;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.OptionalInt;
import java.util.Set;
import java.util.stream.Collectors;

public class DictionaryCompressionOptimizer {
    private static final double DICTIONARY_MIN_COMPRESSION_RATIO = 1.25;
    static final DataSize DICTIONARY_MEMORY_MAX_RANGE = new DataSize(4.0, DataSize.Unit.MEGABYTE);
    static final DataSize DIRECT_COLUMN_SIZE_RANGE = new DataSize(4.0, DataSize.Unit.MEGABYTE);
    private final Set<DictionaryColumnManager> allWriters;
    private final Set<DictionaryColumnManager> directConversionCandidates = new HashSet<DictionaryColumnManager>();
    private final int stripeMinBytes;
    private final int stripeMaxBytes;
    private final int stripeMaxRowCount;
    private final int dictionaryMemoryMaxBytesLow;
    private final int dictionaryMemoryMaxBytesHigh;
    private int dictionaryMemoryBytes;

    public DictionaryCompressionOptimizer(Set<? extends DictionaryColumn> writers, int stripeMinBytes, int stripeMaxBytes, int stripeMaxRowCount, int dictionaryMemoryMaxBytes) {
        Objects.requireNonNull(writers, "writers is null");
        this.allWriters = ImmutableSet.copyOf((Collection)writers.stream().map(DictionaryColumnManager::new).collect(Collectors.toSet()));
        Preconditions.checkArgument((stripeMinBytes >= 0 ? 1 : 0) != 0, (Object)"stripeMinBytes is negative");
        this.stripeMinBytes = stripeMinBytes;
        Preconditions.checkArgument((stripeMaxBytes >= stripeMinBytes ? 1 : 0) != 0, (Object)"stripeMaxBytes is less than stripeMinBytes");
        this.stripeMaxBytes = stripeMaxBytes;
        Preconditions.checkArgument((stripeMaxRowCount >= 0 ? 1 : 0) != 0, (Object)"stripeMaxRowCount is negative");
        this.stripeMaxRowCount = stripeMaxRowCount;
        Preconditions.checkArgument((dictionaryMemoryMaxBytes >= 0 ? 1 : 0) != 0, (Object)"dictionaryMemoryMaxBytes is negative");
        this.dictionaryMemoryMaxBytesHigh = dictionaryMemoryMaxBytes;
        this.dictionaryMemoryMaxBytesLow = (int)Math.max((long)dictionaryMemoryMaxBytes - DICTIONARY_MEMORY_MAX_RANGE.toBytes(), 0L);
        this.directConversionCandidates.addAll(this.allWriters);
    }

    public int getDictionaryMemoryBytes() {
        return this.dictionaryMemoryBytes;
    }

    public boolean isFull(long bufferedBytes) {
        if (bufferedBytes > (long)this.stripeMinBytes) {
            return this.dictionaryMemoryBytes > this.dictionaryMemoryMaxBytesLow;
        }
        return this.dictionaryMemoryBytes > this.dictionaryMemoryMaxBytesHigh;
    }

    public void reset() {
        this.directConversionCandidates.clear();
        this.directConversionCandidates.addAll(this.allWriters);
        this.dictionaryMemoryBytes = 0;
        this.allWriters.forEach(DictionaryColumnManager::reset);
    }

    public void finalOptimize(int bufferedBytes) {
        this.convertLowCompressionStreams(bufferedBytes);
    }

    public void optimize(int bufferedBytes, int stripeRowCount) {
        this.dictionaryMemoryBytes = this.allWriters.stream().filter(writer -> !writer.isDirectEncoded()).mapToInt(DictionaryColumnManager::getDictionaryBytes).sum();
        this.allWriters.stream().filter(writer -> !writer.isDirectEncoded()).forEach(column -> column.updateHistory(stripeRowCount));
        if (this.dictionaryMemoryBytes <= this.dictionaryMemoryMaxBytesLow) {
            return;
        }
        bufferedBytes = this.convertLowCompressionStreams(bufferedBytes);
        if (this.dictionaryMemoryBytes <= this.dictionaryMemoryMaxBytesLow || bufferedBytes >= this.stripeMaxBytes) {
            return;
        }
        int nonDictionaryBufferedBytes = bufferedBytes;
        for (DictionaryColumnManager dictionaryWriter : this.allWriters) {
            if (dictionaryWriter.isDirectEncoded()) continue;
            nonDictionaryBufferedBytes = (int)((long)nonDictionaryBufferedBytes - dictionaryWriter.getBufferedBytes());
        }
        while (!this.directConversionCandidates.isEmpty() && this.dictionaryMemoryBytes > this.dictionaryMemoryMaxBytesHigh && bufferedBytes < this.stripeMaxBytes) {
            DictionaryCompressionProjection projection = this.selectDictionaryColumnToConvert(nonDictionaryBufferedBytes, stripeRowCount);
            int selectDictionaryColumnBufferedBytes = Math.toIntExact(projection.getColumnToConvert().getBufferedBytes());
            OptionalInt directBytes = this.tryConvertToDirect(projection.getColumnToConvert(), this.getMaxDirectBytes(bufferedBytes));
            if (!directBytes.isPresent()) continue;
            bufferedBytes = bufferedBytes + directBytes.getAsInt() - selectDictionaryColumnBufferedBytes;
            nonDictionaryBufferedBytes += directBytes.getAsInt();
        }
        if (bufferedBytes >= this.stripeMaxBytes) {
            return;
        }
        if (bufferedBytes >= this.stripeMinBytes) {
            double currentCompressionRatio = this.currentCompressionRatio(nonDictionaryBufferedBytes);
            while (!this.directConversionCandidates.isEmpty() && bufferedBytes < this.stripeMaxBytes) {
                DictionaryCompressionProjection projection = this.selectDictionaryColumnToConvert(nonDictionaryBufferedBytes, stripeRowCount);
                if (projection.getPredictedFileCompressionRatio() < currentCompressionRatio) {
                    return;
                }
                int selectDictionaryColumnBufferedBytes = Math.toIntExact(projection.getColumnToConvert().getBufferedBytes());
                OptionalInt directBytes = this.tryConvertToDirect(projection.getColumnToConvert(), this.getMaxDirectBytes(bufferedBytes));
                if (!directBytes.isPresent()) continue;
                bufferedBytes = bufferedBytes + directBytes.getAsInt() - selectDictionaryColumnBufferedBytes;
                nonDictionaryBufferedBytes += directBytes.getAsInt();
            }
        }
    }

    private int convertLowCompressionStreams(int bufferedBytes) {
        for (DictionaryColumnManager dictionaryWriter : ImmutableList.copyOf(this.directConversionCandidates)) {
            if (!(dictionaryWriter.getCompressionRatio() < 1.25)) continue;
            int columnBufferedBytes = Math.toIntExact(dictionaryWriter.getBufferedBytes());
            OptionalInt directBytes = this.tryConvertToDirect(dictionaryWriter, this.getMaxDirectBytes(bufferedBytes));
            if (!directBytes.isPresent() || (bufferedBytes = bufferedBytes + directBytes.getAsInt() - columnBufferedBytes) < this.stripeMaxBytes) continue;
            return bufferedBytes;
        }
        return bufferedBytes;
    }

    private OptionalInt tryConvertToDirect(DictionaryColumnManager dictionaryWriter, int maxDirectBytes) {
        int dictionaryBytes = dictionaryWriter.getDictionaryBytes();
        OptionalInt directBytes = dictionaryWriter.tryConvertToDirect(maxDirectBytes);
        if (directBytes.isPresent()) {
            this.dictionaryMemoryBytes -= dictionaryBytes;
        }
        this.directConversionCandidates.remove(dictionaryWriter);
        return directBytes;
    }

    private double currentCompressionRatio(int totalNonDictionaryBytes) {
        long uncompressedBytes = totalNonDictionaryBytes;
        long compressedBytes = totalNonDictionaryBytes;
        for (DictionaryColumnManager column : this.allWriters) {
            if (column.isDirectEncoded()) continue;
            uncompressedBytes += column.getRawBytes();
            compressedBytes += (long)column.getDictionaryBytes();
        }
        return 1.0 * (double)uncompressedBytes / (double)compressedBytes;
    }

    private DictionaryCompressionProjection selectDictionaryColumnToConvert(int totalNonDictionaryBytes, int stripeRowCount) {
        Preconditions.checkState((!this.directConversionCandidates.isEmpty() ? 1 : 0) != 0);
        int totalNonDictionaryBytesPerRow = totalNonDictionaryBytes / stripeRowCount;
        long totalDictionaryRawBytes = 0L;
        long totalDictionaryBytes = 0L;
        long totalDictionaryIndexBytes = 0L;
        long totalDictionaryRawBytesPerRow = 0L;
        long totalDictionaryBytesPerNewRow = 0L;
        long totalDictionaryIndexBytesPerRow = 0L;
        for (DictionaryColumnManager column : this.allWriters) {
            if (column.isDirectEncoded()) continue;
            totalDictionaryRawBytes += column.getRawBytes();
            totalDictionaryBytes += (long)column.getDictionaryBytes();
            totalDictionaryIndexBytes += (long)column.getIndexBytes();
            totalDictionaryRawBytesPerRow = (long)((double)totalDictionaryRawBytesPerRow + column.getRawBytesPerRow());
            totalDictionaryBytesPerNewRow = (long)((double)totalDictionaryBytesPerNewRow + column.getDictionaryBytesPerFutureRow());
            totalDictionaryIndexBytesPerRow = (long)((double)totalDictionaryIndexBytesPerRow + column.getIndexBytesPerRow());
        }
        long totalUncompressedBytesPerRow = (long)totalNonDictionaryBytesPerRow + totalDictionaryRawBytesPerRow;
        DictionaryCompressionProjection maxProjectedCompression = null;
        for (DictionaryColumnManager column : this.directConversionCandidates) {
            long currentRawBytes = (long)totalNonDictionaryBytes + column.getRawBytes();
            long currentDictionaryBytes = totalDictionaryBytes - (long)column.getDictionaryBytes();
            long currentIndexBytes = totalDictionaryIndexBytes - (long)column.getIndexBytes();
            long currentTotalBytes = currentRawBytes + currentDictionaryBytes + currentIndexBytes;
            double rawBytesPerFutureRow = (double)totalNonDictionaryBytesPerRow + column.getRawBytesPerRow();
            double dictionaryBytesPerFutureRow = (double)totalDictionaryBytesPerNewRow - column.getDictionaryBytesPerFutureRow();
            double indexBytesPerFutureRow = (double)totalDictionaryIndexBytesPerRow - column.getIndexBytesPerRow();
            double totalBytesPerFutureRow = rawBytesPerFutureRow + dictionaryBytesPerFutureRow + indexBytesPerFutureRow;
            long rowsToDictionaryMemoryLimit = (long)((double)((long)this.dictionaryMemoryMaxBytesLow - currentDictionaryBytes) / dictionaryBytesPerFutureRow);
            long rowsToStripeMemoryLimit = (long)((double)((long)this.stripeMaxBytes - currentTotalBytes) / totalBytesPerFutureRow);
            long rowsToStripeRowLimit = this.stripeMaxRowCount - stripeRowCount;
            long rowsToLimit = Longs.min((long[])new long[]{rowsToDictionaryMemoryLimit, rowsToStripeMemoryLimit, rowsToStripeRowLimit});
            long predictedUncompressedSizeAtLimit = (long)totalNonDictionaryBytes + totalDictionaryRawBytes + totalUncompressedBytesPerRow * rowsToLimit;
            long predictedCompressedSizeAtLimit = (long)((double)currentTotalBytes + totalBytesPerFutureRow * (double)rowsToLimit);
            double predictedCompressionRatioAtLimit = 1.0 * (double)predictedUncompressedSizeAtLimit / (double)predictedCompressedSizeAtLimit;
            if (maxProjectedCompression != null && !(maxProjectedCompression.getPredictedFileCompressionRatio() < predictedCompressionRatioAtLimit)) continue;
            maxProjectedCompression = new DictionaryCompressionProjection(column, predictedCompressionRatioAtLimit);
        }
        return maxProjectedCompression;
    }

    private int getMaxDirectBytes(int bufferedBytes) {
        return Math.toIntExact(Math.min((long)this.stripeMaxBytes, (long)(this.stripeMaxBytes - bufferedBytes) + DIRECT_COLUMN_SIZE_RANGE.toBytes()));
    }

    public static int estimateIndexBytesPerValue(int dictionaryEntries) {
        if (dictionaryEntries <= 256) {
            return 1;
        }
        if (dictionaryEntries <= 65536) {
            return 2;
        }
        if (dictionaryEntries <= 0x1000000) {
            return 3;
        }
        return 4;
    }

    private static class DictionaryCompressionProjection {
        private final DictionaryColumnManager columnToConvert;
        private final double predictedFileCompressionRatio;

        public DictionaryCompressionProjection(DictionaryColumnManager columnToConvert, double predictedFileCompressionRatio) {
            this.columnToConvert = Objects.requireNonNull(columnToConvert, "columnToConvert is null");
            this.predictedFileCompressionRatio = predictedFileCompressionRatio;
        }

        public DictionaryColumnManager getColumnToConvert() {
            return this.columnToConvert;
        }

        public double getPredictedFileCompressionRatio() {
            return this.predictedFileCompressionRatio;
        }
    }

    private static class DictionaryColumnManager {
        private final DictionaryColumn dictionaryColumn;
        private boolean directEncoded;
        private int rowCount;
        private long pastValueCount;
        private int pastDictionaryEntries;
        private long pendingPastValueCount;
        private int pendingPastDictionaryEntries;

        public DictionaryColumnManager(DictionaryColumn dictionaryColumn) {
            this.dictionaryColumn = dictionaryColumn;
        }

        OptionalInt tryConvertToDirect(int maxDirectBytes) {
            OptionalInt directBytes = this.dictionaryColumn.tryConvertToDirect(maxDirectBytes);
            if (directBytes.isPresent()) {
                this.directEncoded = true;
            }
            return directBytes;
        }

        void reset() {
            this.directEncoded = false;
            this.pastValueCount = 0L;
            this.pastDictionaryEntries = 0;
            this.pendingPastValueCount = 0L;
            this.pendingPastDictionaryEntries = 0;
        }

        public void updateHistory(int rowCount) {
            this.rowCount = rowCount;
            long currentValueCount = this.dictionaryColumn.getValueCount();
            if (currentValueCount - this.pendingPastValueCount >= 1024L) {
                this.pastValueCount = this.pendingPastValueCount;
                this.pastDictionaryEntries = this.pendingPastDictionaryEntries;
                this.pendingPastValueCount = currentValueCount;
                this.pendingPastDictionaryEntries = this.dictionaryColumn.getDictionaryEntries();
            }
        }

        public long getRawBytes() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return this.dictionaryColumn.getRawBytes();
        }

        public double getRawBytesPerRow() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return 1.0 * (double)this.getRawBytes() / (double)this.rowCount;
        }

        public int getDictionaryBytes() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return this.dictionaryColumn.getDictionaryBytes();
        }

        public double getDictionaryBytesPerFutureRow() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            int currentDictionaryEntries = this.dictionaryColumn.getDictionaryEntries();
            long currentValueCount = this.dictionaryColumn.getValueCount();
            double dictionaryBytesPerEntry = 1.0 * (double)this.dictionaryColumn.getDictionaryBytes() / (double)currentDictionaryEntries;
            double dictionaryEntriesPerFutureValue = 1.0 * (double)(currentDictionaryEntries - this.pastDictionaryEntries) / (double)(currentValueCount - this.pastValueCount);
            return dictionaryBytesPerEntry * dictionaryEntriesPerFutureValue;
        }

        public int getIndexBytes() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return this.dictionaryColumn.getIndexBytes();
        }

        public double getIndexBytesPerRow() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return 1.0 * (double)this.getIndexBytes() / (double)this.rowCount;
        }

        public double getCompressionRatio() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return 1.0 * (double)this.getRawBytes() / (double)this.getBufferedBytes();
        }

        public long getBufferedBytes() {
            return this.dictionaryColumn.getBufferedBytes();
        }

        public boolean isDirectEncoded() {
            return this.directEncoded;
        }
    }

    public static interface DictionaryColumn {
        public long getValueCount();

        public long getNonNullValueCount();

        public long getRawBytes();

        public int getDictionaryEntries();

        public int getDictionaryBytes();

        public int getIndexBytes();

        public OptionalInt tryConvertToDirect(int var1);

        public long getBufferedBytes();
    }
}

