/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.log;

import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.apache.avro.Schema;
import org.apache.hadoop.fs.BufferedFSInputStream;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.table.log.HoodieLogFormat;
import org.apache.hudi.common.table.log.HoodieLogFormatVersion;
import org.apache.hudi.common.table.log.block.HoodieAvroDataBlock;
import org.apache.hudi.common.table.log.block.HoodieCommandBlock;
import org.apache.hudi.common.table.log.block.HoodieCorruptBlock;
import org.apache.hudi.common.table.log.block.HoodieDeleteBlock;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.common.util.FSUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.exception.CorruptedLogFileException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.HoodieNotSupportedException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

class HoodieLogFileReader
implements HoodieLogFormat.Reader {
    public static final int DEFAULT_BUFFER_SIZE = 0x1000000;
    private static final Logger LOG = LogManager.getLogger(HoodieLogFileReader.class);
    private final FSDataInputStream inputStream;
    private final HoodieLogFile logFile;
    private static final byte[] MAGIC_BUFFER = new byte[6];
    private final Schema readerSchema;
    private boolean readBlockLazily;
    private long reverseLogFilePosition;
    private long lastReverseLogFilePosition;
    private boolean reverseReader;
    private boolean closed = false;

    HoodieLogFileReader(FileSystem fs, HoodieLogFile logFile, Schema readerSchema, int bufferSize, boolean readBlockLazily, boolean reverseReader) throws IOException {
        FSDataInputStream fsDataInputStream = fs.open(logFile.getPath(), bufferSize);
        this.inputStream = fsDataInputStream.getWrappedStream() instanceof FSInputStream ? new FSDataInputStream((InputStream)new BufferedFSInputStream((FSInputStream)fsDataInputStream.getWrappedStream(), bufferSize)) : fsDataInputStream;
        this.logFile = logFile;
        this.readerSchema = readerSchema;
        this.readBlockLazily = readBlockLazily;
        this.reverseReader = reverseReader;
        if (this.reverseReader) {
            this.reverseLogFilePosition = this.lastReverseLogFilePosition = fs.getFileStatus(logFile.getPath()).getLen();
        }
        this.addShutDownHook();
    }

    HoodieLogFileReader(FileSystem fs, HoodieLogFile logFile, Schema readerSchema, boolean readBlockLazily, boolean reverseReader) throws IOException {
        this(fs, logFile, readerSchema, 0x1000000, readBlockLazily, reverseReader);
    }

    HoodieLogFileReader(FileSystem fs, HoodieLogFile logFile, Schema readerSchema) throws IOException {
        this(fs, logFile, readerSchema, 0x1000000, false, false);
    }

    @Override
    public HoodieLogFile getLogFile() {
        return this.logFile;
    }

    private void addShutDownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            try {
                this.close();
            }
            catch (Exception e) {
                LOG.warn((Object)("unable to close input stream for log file " + this.logFile), (Throwable)e);
            }
        }));
    }

    private HoodieLogBlock readBlock() throws IOException {
        int blocksize;
        HoodieLogBlock.HoodieLogBlockType blockType = null;
        Map<HoodieLogBlock.HeaderMetadataType, String> header = null;
        try {
            blocksize = (int)this.inputStream.readLong();
        }
        catch (EOFException | CorruptedLogFileException e) {
            return this.createCorruptBlock();
        }
        boolean isCorrupted = this.isBlockCorrupt(blocksize);
        if (isCorrupted) {
            return this.createCorruptBlock();
        }
        HoodieLogFormat.LogFormatVersion nextBlockVersion = this.readVersion();
        if (nextBlockVersion.getVersion() != 0) {
            int type = this.inputStream.readInt();
            ValidationUtils.checkArgument(type < HoodieLogBlock.HoodieLogBlockType.values().length, "Invalid block byte type found " + type);
            blockType = HoodieLogBlock.HoodieLogBlockType.values()[type];
        }
        if (nextBlockVersion.hasHeader()) {
            header = HoodieLogBlock.getLogMetadata((DataInputStream)this.inputStream);
        }
        int contentLength = blocksize;
        if (nextBlockVersion.getVersion() != 0) {
            contentLength = (int)this.inputStream.readLong();
        }
        long contentPosition = this.inputStream.getPos();
        byte[] content = HoodieLogBlock.readOrSkipContent(this.inputStream, contentLength, this.readBlockLazily);
        Map<HoodieLogBlock.HeaderMetadataType, String> footer = null;
        if (nextBlockVersion.hasFooter()) {
            footer = HoodieLogBlock.getLogMetadata((DataInputStream)this.inputStream);
        }
        long logBlockLength = 0L;
        if (nextBlockVersion.hasLogBlockLength()) {
            logBlockLength = this.inputStream.readLong();
        }
        long blockEndPos = this.inputStream.getPos();
        switch (Objects.requireNonNull(blockType)) {
            case AVRO_DATA_BLOCK: {
                if (nextBlockVersion.getVersion() == 0) {
                    return HoodieAvroDataBlock.getBlock(content, this.readerSchema);
                }
                return HoodieAvroDataBlock.getBlock(this.logFile, this.inputStream, Option.ofNullable(content), this.readBlockLazily, contentPosition, contentLength, blockEndPos, this.readerSchema, header, footer);
            }
            case DELETE_BLOCK: {
                return HoodieDeleteBlock.getBlock(this.logFile, this.inputStream, Option.ofNullable(content), this.readBlockLazily, contentPosition, contentLength, blockEndPos, header, footer);
            }
            case COMMAND_BLOCK: {
                return HoodieCommandBlock.getBlock(this.logFile, this.inputStream, Option.ofNullable(content), this.readBlockLazily, contentPosition, contentLength, blockEndPos, header, footer);
            }
        }
        throw new HoodieNotSupportedException("Unsupported Block " + (Object)((Object)blockType));
    }

    private HoodieLogBlock createCorruptBlock() throws IOException {
        LOG.info((Object)("Log " + this.logFile + " has a corrupted block at " + this.inputStream.getPos()));
        long currentPos = this.inputStream.getPos();
        long nextBlockOffset = this.scanForNextAvailableBlockOffset();
        this.inputStream.seek(currentPos);
        LOG.info((Object)("Next available block in " + this.logFile + " starts at " + nextBlockOffset));
        int corruptedBlockSize = (int)(nextBlockOffset - currentPos);
        long contentPosition = this.inputStream.getPos();
        byte[] corruptedBytes = HoodieLogBlock.readOrSkipContent(this.inputStream, corruptedBlockSize, this.readBlockLazily);
        return HoodieCorruptBlock.getBlock(this.logFile, this.inputStream, Option.ofNullable(corruptedBytes), this.readBlockLazily, contentPosition, corruptedBlockSize, corruptedBlockSize, new HashMap<HoodieLogBlock.HeaderMetadataType, String>(), new HashMap<HoodieLogBlock.HeaderMetadataType, String>());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean isBlockCorrupt(int blocksize) throws IOException {
        long currentPos = this.inputStream.getPos();
        try {
            if (FSUtils.isGCSInputStream(this.inputStream)) {
                this.inputStream.seek(currentPos + (long)blocksize - 1L);
            } else {
                this.inputStream.seek(currentPos + (long)blocksize);
            }
        }
        catch (EOFException e) {
            this.inputStream.seek(currentPos);
            return true;
        }
        try {
            this.readMagic();
            boolean e = false;
            return e;
        }
        catch (CorruptedLogFileException e) {
            boolean bl = true;
            return bl;
        }
        finally {
            this.inputStream.seek(currentPos);
        }
    }

    private long scanForNextAvailableBlockOffset() throws IOException {
        while (true) {
            long currentPos = this.inputStream.getPos();
            try {
                boolean hasNextMagic = this.hasNextMagic();
                if (hasNextMagic) {
                    return currentPos;
                }
                this.inputStream.seek(currentPos + 1L);
            }
            catch (EOFException e) {
                return this.inputStream.getPos();
            }
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.closed) {
            this.inputStream.close();
            this.closed = true;
        }
    }

    @Override
    public boolean hasNext() {
        try {
            return this.readMagic();
        }
        catch (IOException e) {
            throw new HoodieIOException("IOException when reading logfile " + this.logFile, e);
        }
    }

    private HoodieLogFormat.LogFormatVersion readVersion() throws IOException {
        return new HoodieLogFormatVersion(this.inputStream.readInt());
    }

    private boolean readMagic() throws IOException {
        try {
            boolean hasMagic = this.hasNextMagic();
            if (!hasMagic) {
                throw new CorruptedLogFileException(this.logFile + "could not be read. Did not find the magic bytes at the start of the block");
            }
            return hasMagic;
        }
        catch (EOFException e) {
            return false;
        }
    }

    private boolean hasNextMagic() throws IOException {
        long pos = this.inputStream.getPos();
        this.inputStream.readFully(MAGIC_BUFFER, 0, 6);
        return Arrays.equals(MAGIC_BUFFER, HoodieLogFormat.MAGIC);
    }

    @Override
    public HoodieLogBlock next() {
        try {
            return this.readBlock();
        }
        catch (IOException io) {
            throw new HoodieIOException("IOException when reading logblock from log file " + this.logFile, io);
        }
    }

    @Override
    public boolean hasPrev() {
        try {
            if (!this.reverseReader) {
                throw new HoodieNotSupportedException("Reverse log reader has not been enabled");
            }
            this.reverseLogFilePosition = this.lastReverseLogFilePosition;
            this.reverseLogFilePosition -= 8L;
            this.lastReverseLogFilePosition = this.reverseLogFilePosition;
            this.inputStream.seek(this.reverseLogFilePosition);
        }
        catch (Exception e) {
            return false;
        }
        return true;
    }

    @Override
    public HoodieLogBlock prev() throws IOException {
        if (!this.reverseReader) {
            throw new HoodieNotSupportedException("Reverse log reader has not been enabled");
        }
        long blockSize = this.inputStream.readLong();
        long blockEndPos = this.inputStream.getPos();
        try {
            this.inputStream.seek(this.reverseLogFilePosition - blockSize);
        }
        catch (Exception e) {
            this.inputStream.seek(blockEndPos);
            throw new CorruptedLogFileException("Found possible corrupted block, cannot read log file in reverse, fallback to forward reading of logfile");
        }
        boolean hasNext = this.hasNext();
        this.reverseLogFilePosition -= blockSize;
        this.lastReverseLogFilePosition = this.reverseLogFilePosition;
        return this.next();
    }

    public long moveToPrev() throws IOException {
        if (!this.reverseReader) {
            throw new HoodieNotSupportedException("Reverse log reader has not been enabled");
        }
        this.inputStream.seek(this.lastReverseLogFilePosition);
        long blockSize = this.inputStream.readLong();
        this.inputStream.seek(this.reverseLogFilePosition - blockSize);
        this.reverseLogFilePosition -= blockSize;
        this.lastReverseLogFilePosition = this.reverseLogFilePosition;
        return this.reverseLogFilePosition;
    }

    @Override
    public void remove() {
        throw new UnsupportedOperationException("Remove not supported for HoodieLogFileReader");
    }
}

