/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.hive.s3;

import com.amazonaws.AmazonWebServiceClient;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.InstanceProfileCredentialsProvider;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3EncryptionClient;
import com.amazonaws.services.s3.S3ClientOptions;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.EncryptionMaterials;
import com.amazonaws.services.s3.model.EncryptionMaterialsProvider;
import com.facebook.presto.hive.s3.HiveS3Config;
import com.facebook.presto.hive.s3.MockAmazonS3;
import com.facebook.presto.hive.s3.PrestoS3FileSystem;
import com.google.common.base.Preconditions;
import com.google.common.base.VerifyException;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import io.airlift.testing.Assertions;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Map;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.testng.Assert;
import org.testng.SkipException;
import org.testng.annotations.Test;

public class TestPrestoS3FileSystem {
    @Test
    public void testStaticCredentials() throws Exception {
        Configuration config = new Configuration();
        config.set("presto.s3.access-key", "test_secret_access_key");
        config.set("presto.s3.secret-key", "test_access_key_id");
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            Assertions.assertInstanceOf((Object)TestPrestoS3FileSystem.getAwsCredentialsProvider(fs), AWSStaticCredentialsProvider.class);
        }
    }

    @Test
    public void testCompatibleStaticCredentials() throws Exception {
        Configuration config = new Configuration();
        config.set("presto.s3.access-key", "test_secret_access_key");
        config.set("presto.s3.secret-key", "test_access_key_id");
        config.set("presto.s3.endpoint", "test.example.endpoint.com");
        config.set("presto.s3.signer-type", "S3SignerType");
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3a://test-bucket/"), config);
            Assertions.assertInstanceOf((Object)TestPrestoS3FileSystem.getAwsCredentialsProvider(fs), AWSStaticCredentialsProvider.class);
        }
    }

    @Test(expectedExceptions={VerifyException.class}, expectedExceptionsMessageRegExp="Invalid configuration: either endpoint can be set or S3 client can be pinned to the current region")
    public void testEndpointWithPinToCurrentRegionConfiguration() throws Exception {
        Configuration config = new Configuration();
        config.set("presto.s3.endpoint", "test.example.endpoint.com");
        config.set("presto.s3.pin-client-to-current-region", "true");
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3a://test-bucket/"), config);
        }
    }

    @Test
    public void testInstanceCredentialsEnabled() throws Exception {
        Configuration config = new Configuration();
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            Assertions.assertInstanceOf((Object)TestPrestoS3FileSystem.getAwsCredentialsProvider(fs), InstanceProfileCredentialsProvider.class);
        }
    }

    @Test(expectedExceptions={RuntimeException.class}, expectedExceptionsMessageRegExp="S3 credentials not configured")
    public void testInstanceCredentialsDisabled() throws Exception {
        Configuration config = new Configuration();
        config.setBoolean("presto.s3.use-instance-credentials", false);
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
        }
    }

    @Test
    public void testPathStyleAccess() throws Exception {
        Configuration config = new Configuration();
        config.setBoolean("presto.s3.path-style-access", true);
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            S3ClientOptions clientOptions = TestPrestoS3FileSystem.getFieldValue(fs.getS3Client(), AmazonS3Client.class, "clientOptions", S3ClientOptions.class);
            Assert.assertTrue((boolean)clientOptions.isPathStyleAccess());
        }
    }

    @Test
    public void testUnderscoreBucket() throws Exception {
        Configuration config = new Configuration();
        config.setBoolean("presto.s3.path-style-access", true);
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            String expectedBucketName = "test-bucket_underscore";
            fs.initialize(new URI("s3n://" + expectedBucketName + "/"), config);
            fs.setS3Client((AmazonS3)s3);
            fs.getS3ObjectMetadata(new Path("/test/path"));
            Assert.assertEquals((String)expectedBucketName, (String)s3.getGetObjectMetadataRequest().getBucketName());
        }
    }

    @Test
    public void testReadRetryCounters() throws Exception {
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            int maxRetries = 2;
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectHttpErrorCode(500);
            Configuration configuration = new Configuration();
            configuration.set("presto.s3.max-backoff-time", "1ms");
            configuration.set("presto.s3.max-retry-time", "5s");
            configuration.setInt("presto.s3.max-client-retries", maxRetries);
            fs.initialize(new URI("s3n://test-bucket/"), configuration);
            fs.setS3Client((AmazonS3)s3);
            try (FSDataInputStream inputStream = fs.open(new Path("s3n://test-bucket/test"));){
                inputStream.read();
            }
            catch (Throwable expected) {
                Assertions.assertInstanceOf((Object)expected, AmazonS3Exception.class);
                Assert.assertEquals((int)((AmazonS3Exception)expected).getStatusCode(), (int)500);
                Assert.assertEquals((long)PrestoS3FileSystem.getFileSystemStats().getReadRetries().getTotalCount(), (long)maxRetries);
                Assert.assertEquals((long)PrestoS3FileSystem.getFileSystemStats().getGetObjectRetries().getTotalCount(), (long)(((long)maxRetries + 1L) * (long)maxRetries));
            }
        }
    }

    @Test
    public void testGetMetadataRetryCounter() {
        int maxRetries = 2;
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectMetadataHttpCode(500);
            Configuration configuration = new Configuration();
            configuration.set("presto.s3.max-backoff-time", "1ms");
            configuration.set("presto.s3.max-retry-time", "5s");
            configuration.setInt("presto.s3.max-client-retries", maxRetries);
            fs.initialize(new URI("s3n://test-bucket/"), configuration);
            fs.setS3Client((AmazonS3)s3);
            fs.getS3ObjectMetadata(new Path("s3n://test-bucket/test"));
        }
        catch (Throwable expected) {
            Assertions.assertInstanceOf((Object)expected, AmazonS3Exception.class);
            Assert.assertEquals((int)((AmazonS3Exception)expected).getStatusCode(), (int)500);
            Assert.assertEquals((long)PrestoS3FileSystem.getFileSystemStats().getGetMetadataRetries().getTotalCount(), (long)maxRetries);
        }
    }

    @Test(expectedExceptions={RuntimeException.class}, expectedExceptionsMessageRegExp=".*Failing getObject call with 404.*")
    public void testReadNotFound() throws Exception {
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectHttpErrorCode(404);
            fs.initialize(new URI("s3n://test-bucket/"), new Configuration());
            fs.setS3Client((AmazonS3)s3);
            try (FSDataInputStream inputStream = fs.open(new Path("s3n://test-bucket/test"));){
                inputStream.read();
            }
        }
    }

    @Test(expectedExceptions={RuntimeException.class}, expectedExceptionsMessageRegExp=".*Failing getObject call with 403.*")
    public void testReadForbidden() throws Exception {
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectHttpErrorCode(403);
            fs.initialize(new URI("s3n://test-bucket/"), new Configuration());
            fs.setS3Client((AmazonS3)s3);
            try (FSDataInputStream inputStream = fs.open(new Path("s3n://test-bucket/test"));){
                inputStream.read();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCreateWithNonexistentStagingDirectory() throws Exception {
        java.nio.file.Path stagingParent = Files.createTempDirectory("test", new FileAttribute[0]);
        java.nio.file.Path staging = Paths.get(stagingParent.toString(), "staging");
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            Configuration conf = new Configuration();
            conf.set("presto.s3.staging-directory", staging.toString());
            fs.initialize(new URI("s3n://test-bucket/"), conf);
            fs.setS3Client((AmazonS3)s3);
            FSDataOutputStream stream = fs.create(new Path("s3n://test-bucket/test"));
            stream.close();
            Assert.assertTrue((boolean)Files.exists(staging, new LinkOption[0]));
        }
        catch (Throwable throwable) {
            MoreFiles.deleteRecursively((java.nio.file.Path)stagingParent, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            throw throwable;
        }
        MoreFiles.deleteRecursively((java.nio.file.Path)stagingParent, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expectedExceptions={IOException.class}, expectedExceptionsMessageRegExp="Configured staging path is not a directory: .*")
    public void testCreateWithStagingDirectoryFile() throws Exception {
        java.nio.file.Path staging = Files.createTempFile("staging", null, new FileAttribute[0]);
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            Configuration conf = new Configuration();
            conf.set("presto.s3.staging-directory", staging.toString());
            fs.initialize(new URI("s3n://test-bucket/"), conf);
            fs.setS3Client((AmazonS3)s3);
            fs.create(new Path("s3n://test-bucket/test"));
        }
        finally {
            Files.deleteIfExists(staging);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCreateWithStagingDirectorySymlink() throws Exception {
        java.nio.file.Path staging = Files.createTempDirectory("staging", new FileAttribute[0]);
        java.nio.file.Path link = Paths.get(staging + ".symlink", new String[0]);
        try {
            try {
                Files.createSymbolicLink(link, staging, new FileAttribute[0]);
            }
            catch (UnsupportedOperationException e) {
                throw new SkipException("Filesystem does not support symlinks", (Throwable)e);
            }
            try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
                MockAmazonS3 s3 = new MockAmazonS3();
                Configuration conf = new Configuration();
                conf.set("presto.s3.staging-directory", link.toString());
                fs.initialize(new URI("s3n://test-bucket/"), conf);
                fs.setS3Client((AmazonS3)s3);
                FSDataOutputStream stream = fs.create(new Path("s3n://test-bucket/test"));
                stream.close();
                Assert.assertTrue((boolean)Files.exists(link, new LinkOption[0]));
            }
        }
        catch (Throwable throwable) {
            MoreFiles.deleteRecursively((java.nio.file.Path)link, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            MoreFiles.deleteRecursively((java.nio.file.Path)staging, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            throw throwable;
        }
        MoreFiles.deleteRecursively((java.nio.file.Path)link, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
        MoreFiles.deleteRecursively((java.nio.file.Path)staging, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
    }

    @Test
    public void testReadRequestRangeNotSatisfiable() throws Exception {
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectHttpErrorCode(416);
            fs.initialize(new URI("s3n://test-bucket/"), new Configuration());
            fs.setS3Client((AmazonS3)s3);
            try (FSDataInputStream inputStream = fs.open(new Path("s3n://test-bucket/test"));){
                Assert.assertEquals((int)inputStream.read(), (int)-1);
            }
        }
    }

    @Test(expectedExceptions={RuntimeException.class}, expectedExceptionsMessageRegExp=".*Failing getObjectMetadata call with 403.*")
    public void testGetMetadataForbidden() throws Exception {
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectMetadataHttpCode(403);
            fs.initialize(new URI("s3n://test-bucket/"), new Configuration());
            fs.setS3Client((AmazonS3)s3);
            fs.getS3ObjectMetadata(new Path("s3n://test-bucket/test"));
        }
    }

    @Test
    public void testGetMetadataNotFound() throws Exception {
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            MockAmazonS3 s3 = new MockAmazonS3();
            s3.setGetObjectMetadataHttpCode(404);
            fs.initialize(new URI("s3n://test-bucket/"), new Configuration());
            fs.setS3Client((AmazonS3)s3);
            Assert.assertEquals((Object)fs.getS3ObjectMetadata(new Path("s3n://test-bucket/test")), null);
        }
    }

    @Test
    public void testEncryptionMaterialsProvider() throws Exception {
        Configuration config = new Configuration();
        config.set("presto.s3.encryption-materials-provider", TestEncryptionMaterialsProvider.class.getName());
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            Assertions.assertInstanceOf((Object)fs.getS3Client(), AmazonS3EncryptionClient.class);
        }
    }

    @Test
    public void testKMSEncryptionMaterialsProvider() throws Exception {
        Configuration config = new Configuration();
        config.set("presto.s3.kms-key-id", "test-key-id");
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            Assertions.assertInstanceOf((Object)fs.getS3Client(), AmazonS3EncryptionClient.class);
        }
    }

    @Test(expectedExceptions={PrestoS3FileSystem.UnrecoverableS3OperationException.class}, expectedExceptionsMessageRegExp=".*\\Q (Path: /tmp/test/path)\\E")
    public void testUnrecoverableS3ExceptionMessage() {
        throw new PrestoS3FileSystem.UnrecoverableS3OperationException(new Path("/tmp/test/path"), (Throwable)new IOException("test io exception"));
    }

    @Test
    public void testCustomCredentialsProvider() throws Exception {
        Configuration config = new Configuration();
        config.set("presto.s3.use-instance-credentials", "false");
        config.set("presto.s3.credentials-provider", TestCredentialsProvider.class.getName());
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            Assertions.assertInstanceOf((Object)TestPrestoS3FileSystem.getAwsCredentialsProvider(fs), TestCredentialsProvider.class);
        }
    }

    @Test(expectedExceptions={RuntimeException.class}, expectedExceptionsMessageRegExp="Error creating an instance of .*")
    public void testCustomCredentialsClassCannotBeFound() throws Exception {
        Configuration config = new Configuration();
        config.set("presto.s3.use-instance-credentials", "false");
        config.set("presto.s3.credentials-provider", "com.example.DoesNotExist");
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
        }
    }

    @Test
    public void testUserAgentPrefix() throws Exception {
        String userAgentPrefix = "agent_prefix";
        Configuration config = new Configuration();
        config.set("presto.s3.user-agent-prefix", userAgentPrefix);
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), config);
            ClientConfiguration clientConfig = TestPrestoS3FileSystem.getFieldValue(fs.getS3Client(), AmazonWebServiceClient.class, "clientConfiguration", ClientConfiguration.class);
            Assert.assertEquals((String)clientConfig.getUserAgentSuffix(), (String)"presto");
            Assert.assertEquals((String)clientConfig.getUserAgentPrefix(), (String)userAgentPrefix);
        }
    }

    @Test
    public void testDefaultS3ClientConfiguration() throws Exception {
        HiveS3Config defaults = new HiveS3Config();
        try (PrestoS3FileSystem fs = new PrestoS3FileSystem();){
            fs.initialize(new URI("s3n://test-bucket/"), new Configuration());
            ClientConfiguration config = TestPrestoS3FileSystem.getFieldValue(fs.getS3Client(), AmazonWebServiceClient.class, "clientConfiguration", ClientConfiguration.class);
            Assert.assertEquals((int)config.getMaxErrorRetry(), (int)defaults.getS3MaxErrorRetries());
            Assert.assertEquals((long)config.getConnectionTimeout(), (long)defaults.getS3ConnectTimeout().toMillis());
            Assert.assertEquals((long)config.getSocketTimeout(), (long)defaults.getS3SocketTimeout().toMillis());
            Assert.assertEquals((int)config.getMaxConnections(), (int)defaults.getS3MaxConnections());
            Assert.assertEquals((String)config.getUserAgentSuffix(), (String)"presto");
            Assert.assertEquals((String)config.getUserAgentPrefix(), (String)"");
        }
    }

    private static AWSCredentialsProvider getAwsCredentialsProvider(PrestoS3FileSystem fs) {
        return TestPrestoS3FileSystem.getFieldValue(fs.getS3Client(), "awsCredentialsProvider", AWSCredentialsProvider.class);
    }

    private static <T> T getFieldValue(Object instance, String name, Class<T> type) {
        return TestPrestoS3FileSystem.getFieldValue(instance, instance.getClass(), name, type);
    }

    private static <T> T getFieldValue(Object instance, Class<?> clazz, String name, Class<T> type) {
        try {
            Field field = clazz.getDeclaredField(name);
            Preconditions.checkArgument((field.getType() == type ? 1 : 0) != 0, (String)"expected %s but found %s", type, field.getType());
            field.setAccessible(true);
            return (T)field.get(instance);
        }
        catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    private static class TestCredentialsProvider
    implements AWSCredentialsProvider {
        public TestCredentialsProvider(URI uri, Configuration conf) {
        }

        public AWSCredentials getCredentials() {
            return null;
        }

        public void refresh() {
        }
    }

    private static class TestEncryptionMaterialsProvider
    implements EncryptionMaterialsProvider {
        private final EncryptionMaterials encryptionMaterials = new EncryptionMaterials((SecretKey)new SecretKeySpec(new byte[]{1, 2, 3}, "AES"));

        public void refresh() {
        }

        public EncryptionMaterials getEncryptionMaterials(Map<String, String> materialsDescription) {
            return this.encryptionMaterials;
        }

        public EncryptionMaterials getEncryptionMaterials() {
            return this.encryptionMaterials;
        }
    }
}

