/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.jdbc.internal.airlift.slice;

import com.facebook.presto.jdbc.internal.airlift.slice.InvalidCodePointException;
import com.facebook.presto.jdbc.internal.airlift.slice.InvalidUtf8Exception;
import com.facebook.presto.jdbc.internal.airlift.slice.Preconditions;
import com.facebook.presto.jdbc.internal.airlift.slice.Slice;
import com.facebook.presto.jdbc.internal.airlift.slice.Slices;
import java.util.OptionalInt;

public final class SliceUtf8 {
    private static final int REPLACEMENT_CODE_POINT = 65533;
    private static final int TOP_MASK32 = -2139062144;
    private static final long TOP_MASK64 = -9187201950435737472L;
    private static final int[] LOWER_CODE_POINTS = new int[0x110000];
    private static final int[] UPPER_CODE_POINTS = new int[0x110000];
    private static final boolean[] WHITESPACE_CODE_POINTS = new boolean[0x110000];

    private SliceUtf8() {
    }

    public static boolean isAscii(Slice utf8) {
        int offset;
        int length = utf8.length();
        int length8 = length & 0x7FFFFFF8;
        for (offset = 0; offset < length8; offset += 8) {
            if ((utf8.getLongUnchecked(offset) & 0x8080808080808080L) == 0L) continue;
            return false;
        }
        if (offset + 4 < length) {
            if ((utf8.getIntUnchecked(offset) & 0x80808080) != 0) {
                return false;
            }
            offset += 4;
        }
        while (offset < length) {
            if ((utf8.getByteUnchecked(offset) & 0x80) != 0) {
                return false;
            }
            ++offset;
        }
        return true;
    }

    public static int countCodePoints(Slice utf8) {
        return SliceUtf8.countCodePoints(utf8, 0, utf8.length());
    }

    public static int countCodePoints(Slice utf8, int offset, int length) {
        Preconditions.checkPositionIndexes(offset, offset + length, utf8.length());
        if (length == 0) {
            return 0;
        }
        int continuationBytesCount = 0;
        int length8 = length & 0x7FFFFFF8;
        while (offset < length8) {
            continuationBytesCount += SliceUtf8.countContinuationBytes(utf8.getLongUnchecked(offset));
            offset += 8;
        }
        if (offset + 4 < length) {
            continuationBytesCount += SliceUtf8.countContinuationBytes(utf8.getIntUnchecked(offset));
            offset += 4;
        }
        while (offset < length) {
            continuationBytesCount += SliceUtf8.countContinuationBytes(utf8.getByteUnchecked(offset));
            ++offset;
        }
        assert (continuationBytesCount <= length);
        return length - continuationBytesCount;
    }

    public static Slice substring(Slice utf8, int codePointStart, int codePointLength) {
        Preconditions.checkArgument(codePointStart >= 0, "codePointStart is negative");
        Preconditions.checkArgument(codePointLength >= 0, "codePointLength is negative");
        int indexStart = SliceUtf8.offsetOfCodePoint(utf8, codePointStart);
        if (indexStart < 0) {
            throw new IllegalArgumentException("UTF-8 does not contain " + codePointStart + " code points");
        }
        if (codePointLength == 0) {
            return Slices.EMPTY_SLICE;
        }
        int indexEnd = SliceUtf8.offsetOfCodePoint(utf8, indexStart, codePointLength - 1);
        if (indexEnd < 0) {
            throw new IllegalArgumentException("UTF-8 does not contain " + (codePointStart + codePointLength) + " code points");
        }
        if ((indexEnd += SliceUtf8.lengthOfCodePoint(utf8, indexEnd)) > utf8.length()) {
            throw new InvalidUtf8Exception("UTF-8 is not well formed");
        }
        return utf8.slice(indexStart, indexEnd - indexStart);
    }

    public static Slice reverse(Slice utf8) {
        int codePointLength;
        int length = utf8.length();
        Slice reverse = Slices.allocate(length);
        int reversePosition = length;
        for (int forwardPosition = 0; forwardPosition < length; forwardPosition += codePointLength) {
            codePointLength = SliceUtf8.lengthOfCodePointSafe(utf8, forwardPosition);
            if ((reversePosition -= codePointLength) < 0) {
                throw new InvalidUtf8Exception("UTF-8 is not well formed");
            }
            SliceUtf8.copyUtf8SequenceUnsafe(utf8, forwardPosition, reverse, reversePosition, codePointLength);
        }
        return reverse;
    }

    public static Slice toUpperCase(Slice utf8) {
        return SliceUtf8.translateCodePoints(utf8, UPPER_CODE_POINTS);
    }

    public static Slice toLowerCase(Slice utf8) {
        return SliceUtf8.translateCodePoints(utf8, LOWER_CODE_POINTS);
    }

    private static Slice translateCodePoints(Slice utf8, int[] codePointTranslationMap) {
        int length = utf8.length();
        Slice newUtf8 = Slices.allocate(length);
        int position = 0;
        int upperPosition = 0;
        while (position < length) {
            int nextUpperPosition;
            int codePoint = SliceUtf8.getCodePointAtSafe(utf8, position);
            if (codePoint >= 0) {
                int upperCodePoint = codePointTranslationMap[codePoint];
                nextUpperPosition = upperPosition + SliceUtf8.lengthOfCodePoint(upperCodePoint);
                if (nextUpperPosition > length) {
                    newUtf8 = Slices.ensureSize(newUtf8, nextUpperPosition);
                }
                SliceUtf8.setCodePointAt(upperCodePoint, newUtf8, upperPosition);
                position += SliceUtf8.lengthOfCodePoint(codePoint);
                upperPosition = nextUpperPosition;
                continue;
            }
            int skipLength = -codePoint;
            nextUpperPosition = upperPosition + skipLength;
            if (nextUpperPosition > length) {
                newUtf8 = Slices.ensureSize(newUtf8, nextUpperPosition);
            }
            SliceUtf8.copyUtf8SequenceUnsafe(utf8, position, newUtf8, upperPosition, skipLength);
            position += skipLength;
            upperPosition = nextUpperPosition;
        }
        return newUtf8.slice(0, upperPosition);
    }

    private static void copyUtf8SequenceUnsafe(Slice source, int sourcePosition, Slice destination, int destinationPosition, int length) {
        switch (length) {
            case 1: {
                destination.setByteUnchecked(destinationPosition, source.getByteUnchecked(sourcePosition));
                break;
            }
            case 2: {
                destination.setShortUnchecked(destinationPosition, source.getShortUnchecked(sourcePosition));
                break;
            }
            case 3: {
                destination.setShortUnchecked(destinationPosition, source.getShortUnchecked(sourcePosition));
                destination.setByteUnchecked(destinationPosition + 2, source.getByteUnchecked(sourcePosition + 2));
                break;
            }
            case 4: {
                destination.setIntUnchecked(destinationPosition, source.getIntUnchecked(sourcePosition));
                break;
            }
            case 5: {
                destination.setIntUnchecked(destinationPosition, source.getIntUnchecked(sourcePosition));
                destination.setByteUnchecked(destinationPosition + 4, source.getByteUnchecked(sourcePosition + 4));
                break;
            }
            case 6: {
                destination.setIntUnchecked(destinationPosition, source.getIntUnchecked(sourcePosition));
                destination.setShortUnchecked(destinationPosition + 4, source.getShortUnchecked(sourcePosition + 4));
                break;
            }
            default: {
                throw new IllegalStateException("Invalid code point length " + length);
            }
        }
    }

    public static Slice leftTrim(Slice utf8) {
        int length = utf8.length();
        int position = SliceUtf8.firstNonWhitespacePosition(utf8);
        return utf8.slice(position, length - position);
    }

    private static int firstNonWhitespacePosition(Slice utf8) {
        int position;
        int codePoint;
        int length = utf8.length();
        for (position = 0; position < length && (codePoint = SliceUtf8.getCodePointAtSafe(utf8, position)) >= 0 && WHITESPACE_CODE_POINTS[codePoint]; position += SliceUtf8.lengthOfCodePoint(codePoint)) {
        }
        return position;
    }

    public static Slice rightTrim(Slice utf8) {
        int position = SliceUtf8.lastNonWhitespacePosition(utf8, 0);
        return utf8.slice(0, position);
    }

    /*
     * WARNING - void declaration
     */
    private static int lastNonWhitespacePosition(Slice utf8, int minPosition) {
        int length;
        int position;
        void var5_5;
        for (position = length = utf8.length(); minPosition < position; position -= SliceUtf8.lengthOfCodePoint((int)var5_5)) {
            int codePoint;
            byte unsignedByte = utf8.getByte(position - 1);
            if (!SliceUtf8.isContinuationByte(unsignedByte)) {
                codePoint = unsignedByte & 0xFF;
            } else if (minPosition <= position - 2 && !SliceUtf8.isContinuationByte(utf8.getByte(position - 2))) {
                codePoint = SliceUtf8.getCodePointAtSafe(utf8, position - 2);
            } else if (minPosition <= position - 3 && !SliceUtf8.isContinuationByte(utf8.getByte(position - 3))) {
                codePoint = SliceUtf8.getCodePointAtSafe(utf8, position - 3);
            } else {
                if (minPosition > position - 4 || SliceUtf8.isContinuationByte(utf8.getByte(position - 4))) break;
                codePoint = SliceUtf8.getCodePointAtSafe(utf8, position - 4);
            }
            if (var5_5 < 0 || !WHITESPACE_CODE_POINTS[var5_5]) break;
        }
        return position;
    }

    public static Slice trim(Slice utf8) {
        int start = SliceUtf8.firstNonWhitespacePosition(utf8);
        int end = SliceUtf8.lastNonWhitespacePosition(utf8, start);
        return utf8.slice(start, end - start);
    }

    public static Slice fixInvalidUtf8(Slice slice) {
        return SliceUtf8.fixInvalidUtf8(slice, OptionalInt.of(65533));
    }

    public static Slice fixInvalidUtf8(Slice slice, OptionalInt replacementCodePoint) {
        if (SliceUtf8.isAscii(slice)) {
            return slice;
        }
        int replacementCodePointValue = -1;
        int replacementCodePointLength = 0;
        if (replacementCodePoint.isPresent()) {
            replacementCodePointValue = replacementCodePoint.getAsInt();
            replacementCodePointLength = SliceUtf8.lengthOfCodePoint(replacementCodePointValue);
        }
        int length = slice.length();
        Slice utf8 = Slices.allocate(length);
        int dataPosition = 0;
        int utf8Position = 0;
        while (dataPosition < length) {
            int codePointLength;
            int codePoint = SliceUtf8.getCodePointAtSafe(slice, dataPosition);
            if (codePoint >= 0) {
                codePointLength = SliceUtf8.lengthOfCodePoint(codePoint);
                dataPosition += codePointLength;
            } else {
                dataPosition += -codePoint;
                if (replacementCodePointValue < 0) continue;
                codePoint = replacementCodePointValue;
                codePointLength = replacementCodePointLength;
            }
            utf8 = Slices.ensureSize(utf8, utf8Position + codePointLength);
            utf8Position += SliceUtf8.setCodePointAt(codePoint, utf8, utf8Position);
        }
        return utf8.slice(0, utf8Position);
    }

    private static int getCodePointAtSafe(Slice utf8, int position) {
        byte firstByte = utf8.getByte(position);
        int length = SliceUtf8.lengthOfCodePointFromStartByteSafe(firstByte);
        if (length < 0) {
            return length;
        }
        if (length == 1) {
            return firstByte;
        }
        if (position + 1 >= utf8.length()) {
            return -1;
        }
        byte secondByte = utf8.getByteUnchecked(position + 1);
        if (!SliceUtf8.isContinuationByte(secondByte)) {
            return -1;
        }
        if (length == 2) {
            return (firstByte & 0x1F) << 6 | secondByte & 0x3F;
        }
        if (position + 2 >= utf8.length()) {
            return -2;
        }
        byte thirdByte = utf8.getByteUnchecked(position + 2);
        if (!SliceUtf8.isContinuationByte(thirdByte)) {
            return -2;
        }
        if (length == 3) {
            int codePoint = (firstByte & 0xF) << 12 | (secondByte & 0x3F) << 6 | thirdByte & 0x3F;
            if (55296 <= codePoint && codePoint <= 57343) {
                return -3;
            }
            return codePoint;
        }
        if (position + 3 >= utf8.length()) {
            return -3;
        }
        byte forthByte = utf8.getByteUnchecked(position + 3);
        if (!SliceUtf8.isContinuationByte(forthByte)) {
            return -3;
        }
        if (length == 4) {
            int codePoint = (firstByte & 7) << 18 | (secondByte & 0x3F) << 12 | (thirdByte & 0x3F) << 6 | forthByte & 0x3F;
            if (codePoint < 0x110000) {
                return codePoint;
            }
            return -4;
        }
        if (position + 4 >= utf8.length()) {
            return -4;
        }
        byte fifthByte = utf8.getByteUnchecked(position + 4);
        if (!SliceUtf8.isContinuationByte(fifthByte)) {
            return -4;
        }
        if (length == 5) {
            return -5;
        }
        if (position + 5 >= utf8.length()) {
            return -5;
        }
        byte sixthByte = utf8.getByteUnchecked(position + 5);
        if (!SliceUtf8.isContinuationByte(sixthByte)) {
            return -5;
        }
        if (length == 6) {
            return -6;
        }
        return -1;
    }

    static int lengthOfCodePointFromStartByteSafe(byte startByte) {
        int unsignedStartByte = startByte & 0xFF;
        if (unsignedStartByte < 128) {
            return 1;
        }
        if (unsignedStartByte < 192) {
            return -1;
        }
        if (unsignedStartByte < 224) {
            return 2;
        }
        if (unsignedStartByte < 240) {
            return 3;
        }
        if (unsignedStartByte < 248) {
            return 4;
        }
        if (unsignedStartByte < 252) {
            return 5;
        }
        if (unsignedStartByte < 254) {
            return 6;
        }
        return -1;
    }

    public static int offsetOfCodePoint(Slice utf8, int codePointCount) {
        return SliceUtf8.offsetOfCodePoint(utf8, 0, codePointCount);
    }

    public static int offsetOfCodePoint(Slice utf8, int position, int codePointCount) {
        int correctIndex;
        Preconditions.checkPositionIndex(position, utf8.length());
        Preconditions.checkArgument(codePointCount >= 0, "codePointPosition is negative");
        if (utf8.length() - position <= codePointCount) {
            return -1;
        }
        if (codePointCount == 0) {
            return position;
        }
        int length8 = utf8.length() & 0x7FFFFFF8;
        for (correctIndex = codePointCount + position; position < length8 && correctIndex >= position + 8; correctIndex += SliceUtf8.countContinuationBytes(utf8.getLongUnchecked(position)), position += 8) {
        }
        int length4 = utf8.length() & 0x7FFFFFFC;
        while (position < length4 && correctIndex >= position + 4) {
            correctIndex += SliceUtf8.countContinuationBytes(utf8.getIntUnchecked(position));
            position += 4;
        }
        while (position < utf8.length() && position != (correctIndex += SliceUtf8.countContinuationBytes(utf8.getByteUnchecked(position)))) {
            ++position;
        }
        if (position == correctIndex && correctIndex < utf8.length()) {
            return correctIndex;
        }
        return -1;
    }

    public static int lengthOfCodePoint(Slice utf8, int position) {
        return SliceUtf8.lengthOfCodePointFromStartByte(utf8.getByte(position));
    }

    public static int lengthOfCodePointSafe(Slice utf8, int position) {
        int length = SliceUtf8.lengthOfCodePointFromStartByteSafe(utf8.getByte(position));
        if (length < 0) {
            return -length;
        }
        if (length == 1 || position + 1 >= utf8.length() || !SliceUtf8.isContinuationByte(utf8.getByteUnchecked(position + 1))) {
            return 1;
        }
        if (length == 2 || position + 2 >= utf8.length() || !SliceUtf8.isContinuationByte(utf8.getByteUnchecked(position + 2))) {
            return 2;
        }
        if (length == 3 || position + 3 >= utf8.length() || !SliceUtf8.isContinuationByte(utf8.getByteUnchecked(position + 3))) {
            return 3;
        }
        if (length == 4 || position + 4 >= utf8.length() || !SliceUtf8.isContinuationByte(utf8.getByteUnchecked(position + 4))) {
            return 4;
        }
        if (length == 5 || position + 5 >= utf8.length() || !SliceUtf8.isContinuationByte(utf8.getByteUnchecked(position + 5))) {
            return 5;
        }
        if (length == 6) {
            return 6;
        }
        return 1;
    }

    public static int lengthOfCodePoint(int codePoint) {
        if (codePoint < 0) {
            throw new InvalidCodePointException(codePoint);
        }
        if (codePoint < 128) {
            return 1;
        }
        if (codePoint < 2048) {
            return 2;
        }
        if (codePoint < 65536) {
            return 3;
        }
        if (codePoint < 0x110000) {
            return 4;
        }
        throw new InvalidCodePointException(codePoint);
    }

    public static int lengthOfCodePointFromStartByte(byte startByte) {
        int unsignedStartByte = startByte & 0xFF;
        if (unsignedStartByte < 128) {
            return 1;
        }
        if (unsignedStartByte < 192) {
            throw new InvalidUtf8Exception("Illegal start 0x" + Integer.toHexString(unsignedStartByte).toUpperCase() + " of code point");
        }
        if (unsignedStartByte < 224) {
            return 2;
        }
        if (unsignedStartByte < 240) {
            return 3;
        }
        if (unsignedStartByte < 248) {
            return 4;
        }
        throw new InvalidUtf8Exception("Illegal start 0x" + Integer.toHexString(unsignedStartByte).toUpperCase() + " of code point");
    }

    public static int getCodePointAt(Slice utf8, int position) {
        int unsignedStartByte = utf8.getByte(position) & 0xFF;
        if (unsignedStartByte < 128) {
            return unsignedStartByte;
        }
        if (unsignedStartByte < 192) {
            throw new InvalidUtf8Exception("Illegal start 0x" + Integer.toHexString(unsignedStartByte).toUpperCase() + " of code point");
        }
        if (unsignedStartByte < 224) {
            if (position + 1 >= utf8.length()) {
                throw new InvalidUtf8Exception("UTF-8 sequence truncated");
            }
            return (unsignedStartByte & 0x1F) << 6 | utf8.getByte(position + 1) & 0x3F;
        }
        if (unsignedStartByte < 240) {
            if (position + 2 >= utf8.length()) {
                throw new InvalidUtf8Exception("UTF-8 sequence truncated");
            }
            return (unsignedStartByte & 0xF) << 12 | (utf8.getByteUnchecked(position + 1) & 0x3F) << 6 | utf8.getByteUnchecked(position + 2) & 0x3F;
        }
        if (unsignedStartByte < 248) {
            if (position + 3 >= utf8.length()) {
                throw new InvalidUtf8Exception("UTF-8 sequence truncated");
            }
            return (unsignedStartByte & 7) << 18 | (utf8.getByteUnchecked(position + 1) & 0x3F) << 12 | (utf8.getByteUnchecked(position + 2) & 0x3F) << 6 | utf8.getByteUnchecked(position + 3) & 0x3F;
        }
        throw new InvalidUtf8Exception("Illegal start 0x" + Integer.toHexString(unsignedStartByte).toUpperCase() + " of code point");
    }

    public static int getCodePointBefore(Slice utf8, int position) {
        byte unsignedByte = utf8.getByte(position - 1);
        if (!SliceUtf8.isContinuationByte(unsignedByte)) {
            return unsignedByte & 0xFF;
        }
        if (!SliceUtf8.isContinuationByte(utf8.getByte(position - 2))) {
            return SliceUtf8.getCodePointAt(utf8, position - 2);
        }
        if (!SliceUtf8.isContinuationByte(utf8.getByte(position - 3))) {
            return SliceUtf8.getCodePointAt(utf8, position - 3);
        }
        if (!SliceUtf8.isContinuationByte(utf8.getByte(position - 4))) {
            return SliceUtf8.getCodePointAt(utf8, position - 4);
        }
        throw new InvalidUtf8Exception("UTF-8 is not well formed");
    }

    private static boolean isContinuationByte(byte b) {
        return (b & 0xC0) == 128;
    }

    public static Slice codePointToUtf8(int codePoint) {
        Slice utf8 = Slices.allocate(SliceUtf8.lengthOfCodePoint(codePoint));
        SliceUtf8.setCodePointAt(codePoint, utf8, 0);
        return utf8;
    }

    public static int setCodePointAt(int codePoint, Slice utf8, int position) {
        if (codePoint < 0) {
            throw new InvalidCodePointException(codePoint);
        }
        if (codePoint < 128) {
            utf8.setByte(position, codePoint);
            return 1;
        }
        if (codePoint < 2048) {
            utf8.setByte(position, 0xC0 | codePoint >>> 6);
            utf8.setByte(position + 1, 0x80 | codePoint & 0x3F);
            return 2;
        }
        if (55296 <= codePoint && codePoint <= 57343) {
            throw new InvalidCodePointException(codePoint);
        }
        if (codePoint < 65536) {
            utf8.setByte(position, 0xE0 | codePoint >>> 12 & 0xF);
            utf8.setByte(position + 1, 0x80 | codePoint >>> 6 & 0x3F);
            utf8.setByte(position + 2, 0x80 | codePoint & 0x3F);
            return 3;
        }
        if (codePoint < 0x110000) {
            utf8.setByte(position, 0xF0 | codePoint >>> 18 & 7);
            utf8.setByte(position + 1, 0x80 | codePoint >>> 12 & 0x3F);
            utf8.setByte(position + 2, 0x80 | codePoint >>> 6 & 0x3F);
            utf8.setByte(position + 3, 0x80 | codePoint & 0x3F);
            return 4;
        }
        throw new InvalidCodePointException(codePoint);
    }

    private static int countContinuationBytes(byte i8) {
        int value = i8 & 0xFF;
        return value >>> 7 & ~value >>> 6;
    }

    private static int countContinuationBytes(int i32) {
        i32 = (i32 & 0x80808080) >>> 1 & ~i32;
        return Integer.bitCount(i32);
    }

    private static int countContinuationBytes(long i64) {
        i64 = (i64 & 0x8080808080808080L) >>> 1 & (i64 ^ 0xFFFFFFFFFFFFFFFFL);
        return Long.bitCount(i64);
    }

    static {
        for (int codePoint = 0; codePoint <= 0x10FFFF; ++codePoint) {
            int type = Character.getType(codePoint);
            if (type != 19) {
                SliceUtf8.LOWER_CODE_POINTS[codePoint] = Character.toLowerCase(codePoint);
                SliceUtf8.UPPER_CODE_POINTS[codePoint] = Character.toUpperCase(codePoint);
                SliceUtf8.WHITESPACE_CODE_POINTS[codePoint] = Character.isWhitespace(codePoint);
                continue;
            }
            SliceUtf8.LOWER_CODE_POINTS[codePoint] = 65533;
            SliceUtf8.UPPER_CODE_POINTS[codePoint] = 65533;
            SliceUtf8.WHITESPACE_CODE_POINTS[codePoint] = false;
        }
    }
}

