/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.jdbc.internal.airlift.stats.cardinality;

import com.facebook.presto.jdbc.internal.airlift.slice.BasicSliceInput;
import com.facebook.presto.jdbc.internal.airlift.slice.DynamicSliceOutput;
import com.facebook.presto.jdbc.internal.airlift.slice.SizeOf;
import com.facebook.presto.jdbc.internal.airlift.slice.Slice;
import com.facebook.presto.jdbc.internal.airlift.stats.cardinality.BiasCorrection;
import com.facebook.presto.jdbc.internal.airlift.stats.cardinality.Format;
import com.facebook.presto.jdbc.internal.airlift.stats.cardinality.HllInstance;
import com.facebook.presto.jdbc.internal.airlift.stats.cardinality.Utils;
import com.facebook.presto.jdbc.internal.guava.annotations.VisibleForTesting;
import com.facebook.presto.jdbc.internal.guava.base.Preconditions;
import com.facebook.presto.jdbc.internal.guava.primitives.Bytes;
import com.facebook.presto.jdbc.internal.guava.primitives.Ints;
import com.facebook.presto.jdbc.internal.jol.info.ClassLayout;
import java.util.Arrays;
import java.util.HashSet;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
final class DenseHll
implements HllInstance {
    private static final double LINEAR_COUNTING_MIN_EMPTY_BUCKETS = 0.4;
    private static final int BITS_PER_BUCKET = 4;
    private static final int MAX_DELTA = 15;
    private static final int BUCKET_MASK = 15;
    private static final int DENSE_INSTANCE_SIZE = ClassLayout.parseClass(DenseHll.class).instanceSize();
    private static final int OVERFLOW_GROW_INCREMENT = 5;
    private final byte indexBitLength;
    private byte baseline;
    private int baselineCount;
    private final byte[] deltas;
    private int overflows;
    private int[] overflowBuckets;
    private byte[] overflowValues;

    public DenseHll(int indexBitLength) {
        DenseHll.validatePrefixLength(indexBitLength);
        int numberOfBuckets = Utils.numberOfBuckets(indexBitLength);
        this.indexBitLength = (byte)indexBitLength;
        this.baselineCount = numberOfBuckets;
        this.deltas = new byte[numberOfBuckets * 4 / 8];
        this.overflowBuckets = new int[0];
        this.overflowValues = new byte[0];
    }

    public DenseHll(Slice serialized) {
        int i;
        BasicSliceInput input = serialized.getInput();
        byte formatTag = input.readByte();
        Preconditions.checkArgument(formatTag == Format.DENSE_V1.getTag() || formatTag == Format.DENSE_V2.getTag(), "Invalid format tag");
        this.indexBitLength = input.readByte();
        DenseHll.validatePrefixLength(this.indexBitLength);
        int numberOfBuckets = Utils.numberOfBuckets(this.indexBitLength);
        this.baseline = input.readByte();
        this.deltas = new byte[numberOfBuckets / 2];
        input.readBytes(this.deltas);
        if (formatTag == Format.DENSE_V1.getTag()) {
            short bucket = input.readShort();
            byte value = input.readByte();
            if (bucket >= 0 && value > 0) {
                Preconditions.checkArgument(bucket <= numberOfBuckets, "Overflow bucket index is out of range");
                this.overflows = 1;
                this.overflowBuckets = new int[]{bucket};
                this.overflowValues = new byte[]{value};
            } else {
                this.overflows = 0;
                this.overflowBuckets = new int[0];
                this.overflowValues = new byte[0];
            }
        } else if (formatTag == Format.DENSE_V2.getTag()) {
            this.overflows = input.readUnsignedShort();
            Preconditions.checkArgument(this.overflows <= numberOfBuckets, "Overflow entries is greater than actual number of buckets (possibly corrupt input)");
            this.overflowBuckets = new int[this.overflows];
            this.overflowValues = new byte[this.overflows];
            for (i = 0; i < this.overflows; ++i) {
                this.overflowBuckets[i] = input.readUnsignedShort();
                Preconditions.checkArgument(this.overflowBuckets[i] <= numberOfBuckets, "Overflow bucket index is out of range");
            }
            for (i = 0; i < this.overflows; ++i) {
                this.overflowValues[i] = input.readByte();
                Preconditions.checkArgument(this.overflowValues[i] > 0, "Overflow bucket value must be > 0");
            }
        } else {
            throw new IllegalArgumentException(String.format("Invalid format tag: %d", formatTag));
        }
        this.baselineCount = 0;
        for (i = 0; i < numberOfBuckets; ++i) {
            if (this.getDelta(i) != 0) continue;
            ++this.baselineCount;
        }
        Preconditions.checkArgument(!input.isReadable(), "input is too big");
    }

    public static boolean canDeserialize(Slice serialized) {
        byte formatTag = serialized.getByte(0);
        return formatTag == Format.DENSE_V1.getTag() || formatTag == Format.DENSE_V2.getTag();
    }

    @Override
    public void insertHash(long hash) {
        int index = Utils.computeIndex(hash, this.indexBitLength);
        int value = Utils.computeValue(hash, this.indexBitLength);
        this.insert(index, value);
    }

    @Override
    public int estimatedInMemorySize() {
        return (int)((long)DENSE_INSTANCE_SIZE + SizeOf.sizeOf(this.deltas) + SizeOf.sizeOf(this.overflowBuckets) + SizeOf.sizeOf(this.overflowValues));
    }

    @Override
    public int getIndexBitLength() {
        return this.indexBitLength;
    }

    @Override
    public long cardinality() {
        int numberOfBuckets = Utils.numberOfBuckets(this.indexBitLength);
        if (this.baseline == 0 && (double)this.baselineCount > 0.4 * (double)numberOfBuckets) {
            return Math.round(Utils.linearCounting(this.baselineCount, numberOfBuckets));
        }
        double sum = 0.0;
        for (int i = 0; i < numberOfBuckets; ++i) {
            int value = this.getValue(i);
            sum += 1.0 / (double)(1L << value);
        }
        double estimate = Utils.alpha(this.indexBitLength) * (double)numberOfBuckets * (double)numberOfBuckets / sum;
        estimate = this.correctBias(estimate);
        return Math.round(estimate);
    }

    private double correctBias(double rawEstimate) {
        double bias;
        double[] estimates = BiasCorrection.RAW_ESTIMATES[this.indexBitLength - 4];
        if (rawEstimate < estimates[0] || rawEstimate > estimates[estimates.length - 1]) {
            return rawEstimate;
        }
        double[] biases = BiasCorrection.BIAS[this.indexBitLength - 4];
        int position = this.search(rawEstimate, estimates);
        if (position >= 0) {
            bias = biases[position];
        } else {
            int insertionPoint = -(position + 1);
            double x0 = estimates[insertionPoint - 1];
            double y0 = biases[insertionPoint - 1];
            double x1 = estimates[insertionPoint];
            double y1 = biases[insertionPoint];
            bias = (rawEstimate - x0) * (y1 - y0) / (x1 - x0) + y0;
        }
        return rawEstimate - bias;
    }

    private int search(double rawEstimate, double[] estimateCurve) {
        int low = 0;
        int high = estimateCurve.length - 1;
        while (low <= high) {
            int middle = low + high >>> 1;
            double middleValue = estimateCurve[middle];
            if (rawEstimate > middleValue) {
                low = middle + 1;
                continue;
            }
            if (rawEstimate < middleValue) {
                high = middle - 1;
                continue;
            }
            return middle;
        }
        return -(low + 1);
    }

    public void insert(int bucket, int value) {
        int delta = value - this.baseline;
        int oldDelta = this.getDelta(bucket);
        if (delta <= oldDelta || oldDelta == 15 && delta <= oldDelta + this.getOverflow(bucket)) {
            return;
        }
        if (delta > 15) {
            int overflow = delta - 15;
            if (!this.setOverflow(bucket, overflow)) {
                this.overflowBuckets = Ints.ensureCapacity(this.overflowBuckets, this.overflows + 1, 5);
                this.overflowValues = Bytes.ensureCapacity(this.overflowValues, this.overflows + 1, 5);
                this.overflowBuckets[this.overflows] = bucket;
                this.overflowValues[this.overflows] = (byte)overflow;
                ++this.overflows;
            }
            delta = 15;
        }
        this.setDelta(bucket, delta);
        if (oldDelta == 0) {
            --this.baselineCount;
            this.adjustBaselineIfNeeded();
        }
    }

    private int getOverflow(int bucket) {
        for (int i = 0; i < this.overflows; ++i) {
            if (this.overflowBuckets[i] != bucket) continue;
            return this.overflowValues[i];
        }
        return 0;
    }

    private boolean setOverflow(int bucket, int overflow) {
        for (int i = 0; i < this.overflows; ++i) {
            if (this.overflowBuckets[i] != bucket) continue;
            this.overflowValues[i] = (byte)overflow;
            return true;
        }
        return false;
    }

    @Override
    public Slice serialize() {
        int i;
        int size = this.estimatedSerializedSize();
        DynamicSliceOutput output = new DynamicSliceOutput(size).appendByte(Format.DENSE_V2.getTag()).appendByte(this.indexBitLength).appendByte(this.baseline).appendBytes(this.deltas).appendShort(this.overflows);
        this.sortOverflows();
        for (i = 0; i < this.overflows; ++i) {
            output.appendShort(this.overflowBuckets[i]);
        }
        for (i = 0; i < this.overflows; ++i) {
            output.appendByte(this.overflowValues[i]);
        }
        return output.slice();
    }

    private void sortOverflows() {
        for (int i = 1; i < this.overflows; ++i) {
            for (int j = i; j > 0 && this.overflowBuckets[j - 1] > this.overflowBuckets[j]; --j) {
                int bucket = this.overflowBuckets[j];
                byte value = this.overflowValues[j];
                this.overflowBuckets[j] = this.overflowBuckets[j - 1];
                this.overflowValues[j] = this.overflowValues[j - 1];
                this.overflowBuckets[j - 1] = bucket;
                this.overflowValues[j - 1] = value;
            }
        }
    }

    @Override
    public DenseHll toDense() {
        return this;
    }

    @Override
    public int estimatedSerializedSize() {
        return 3 + Utils.numberOfBuckets(this.indexBitLength) * 1 / 2 + 2 + 2 * this.overflows + 1 * this.overflows;
    }

    private void setDelta(int bucket, int value) {
        int slot = DenseHll.bucketToSlot(bucket);
        byte clearMask = (byte)(15 << DenseHll.shiftForBucket(bucket));
        int n = slot;
        this.deltas[n] = (byte)(this.deltas[n] & ~clearMask);
        byte setMask = (byte)(value << DenseHll.shiftForBucket(bucket));
        int n2 = slot;
        this.deltas[n2] = (byte)(this.deltas[n2] | setMask);
    }

    private int getDelta(int bucket) {
        int slot = DenseHll.bucketToSlot(bucket);
        return this.deltas[slot] >> DenseHll.shiftForBucket(bucket) & 0xF;
    }

    @VisibleForTesting
    int getValue(int bucket) {
        int delta = this.getDelta(bucket);
        if (delta == 15) {
            delta += this.getOverflow(bucket);
        }
        return this.baseline + delta;
    }

    private void adjustBaselineIfNeeded() {
        while (this.baselineCount == 0) {
            this.baseline = (byte)(this.baseline + 1);
            for (int bucket = 0; bucket < Utils.numberOfBuckets(this.indexBitLength); ++bucket) {
                int delta = this.getDelta(bucket);
                boolean hasOverflow = false;
                if (delta == 15) {
                    for (int i = 0; i < this.overflows; ++i) {
                        if (this.overflowBuckets[i] != bucket) continue;
                        hasOverflow = true;
                        int n = i;
                        this.overflowValues[n] = (byte)(this.overflowValues[n] - 1);
                        if (this.overflowValues[i] != 0) break;
                        int lastEntry = this.overflows - 1;
                        if (i < lastEntry) {
                            this.overflowBuckets[i] = this.overflowBuckets[lastEntry];
                            this.overflowValues[i] = this.overflowValues[lastEntry];
                            this.overflowBuckets[lastEntry] = -1;
                            this.overflowValues[lastEntry] = 0;
                        }
                        --this.overflows;
                        break;
                    }
                }
                if (!hasOverflow) {
                    this.setDelta(bucket, --delta);
                }
                if (delta != 0) continue;
                ++this.baselineCount;
            }
        }
    }

    public DenseHll mergeWith(DenseHll other) {
        if (this.indexBitLength != other.indexBitLength) {
            throw new IllegalArgumentException(String.format("Cannot merge HLLs with different number of buckets: %s vs %s", Utils.numberOfBuckets(this.indexBitLength), Utils.numberOfBuckets(other.indexBitLength)));
        }
        int baseline = Math.max(this.baseline, other.baseline);
        int baselineCount = 0;
        int overflows = 0;
        int[] overflowBuckets = new int[5];
        byte[] overflowValues = new byte[5];
        int numberOfBuckets = Utils.numberOfBuckets(this.indexBitLength);
        for (int i = 0; i < numberOfBuckets; ++i) {
            int value = Math.max(this.getValue(i), other.getValue(i));
            int delta = value - baseline;
            if (delta == 0) {
                ++baselineCount;
            } else if (delta > 15) {
                overflowBuckets = Ints.ensureCapacity(overflowBuckets, overflows + 1, 5);
                overflowValues = Bytes.ensureCapacity(overflowValues, overflows + 1, 5);
                overflowBuckets[overflows] = i;
                overflowValues[overflows] = (byte)(delta - 15);
                ++overflows;
                delta = 15;
            }
            this.setDelta(i, delta);
        }
        this.baseline = (byte)baseline;
        this.baselineCount = baselineCount;
        this.overflows = overflows;
        this.overflowBuckets = overflowBuckets;
        this.overflowValues = overflowValues;
        this.adjustBaselineIfNeeded();
        return this;
    }

    public static int estimatedInMemorySize(int indexBitLength) {
        return (int)((long)DENSE_INSTANCE_SIZE + SizeOf.sizeOfByteArray(Utils.numberOfBuckets(indexBitLength) / 2));
    }

    private static int bucketToSlot(int bucket) {
        return bucket >> 1;
    }

    private static int shiftForBucket(int bucket) {
        return (~bucket & 1) << 2;
    }

    private static void validatePrefixLength(int indexBitLength) {
        Preconditions.checkArgument(indexBitLength >= 1 && indexBitLength <= 16, "indexBitLength is out of range");
    }

    @Override
    public void verify() {
        int zeroDeltas = 0;
        for (int i = 0; i < Utils.numberOfBuckets(this.indexBitLength); ++i) {
            if (this.getDelta(i) != 0) continue;
            ++zeroDeltas;
        }
        Preconditions.checkState(zeroDeltas == this.baselineCount, "baselineCount (%s) doesn't match number of zero deltas (%s)", this.baselineCount, zeroDeltas);
        HashSet<Integer> overflows = new HashSet<Integer>();
        for (int i = 0; i < this.overflows; ++i) {
            int bucket = this.overflowBuckets[i];
            overflows.add(bucket);
            Preconditions.checkState(this.overflowValues[i] > 0, "Overflow at %s for bucket %s is 0", i, bucket);
            Preconditions.checkState(this.getDelta(bucket) == 15, "delta in bucket %s is less than MAX_DELTA (%s < %s) even though there's an associated overflow entry", bucket, this.getDelta(bucket), 15);
        }
        Preconditions.checkState(overflows.size() == this.overflows, "Duplicate overflow buckets: %s", Ints.asList(Arrays.copyOf(this.overflowBuckets, this.overflows)));
    }
}

