/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.facebook.presto.jdbc.internal.spi.security;

import com.facebook.presto.jdbc.internal.common.CatalogSchemaName;
import com.facebook.presto.jdbc.internal.common.QualifiedObjectName;
import com.facebook.presto.jdbc.internal.common.Subfield;
import com.facebook.presto.jdbc.internal.common.transaction.TransactionId;
import com.facebook.presto.jdbc.internal.spi.SchemaTableName;

import java.security.Principal;
import java.util.Collections;
import java.util.Optional;
import java.util.Set;

import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyAddColumn;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyCatalogAccess;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyCreateRole;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyCreateSchema;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyCreateTable;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyCreateView;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyCreateViewWithSelect;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyDeleteTable;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyDropColumn;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyDropRole;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyDropSchema;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyDropTable;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyDropView;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyGrantRoles;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyGrantTablePrivilege;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyInsertTable;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyQueryIntegrityCheck;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyRenameColumn;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyRenameSchema;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyRenameTable;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyRevokeRoles;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyRevokeTablePrivilege;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denySelectColumns;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denySetCatalogSessionProperty;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denySetRole;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denySetSystemSessionProperty;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denySetUser;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyShowCurrentRoles;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyShowRoleGrants;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyShowRoles;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyShowSchemas;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyShowTablesMetadata;
import static com.facebook.presto.jdbc.internal.spi.security.AccessDeniedException.denyTruncateTable;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toSet;

public class DenyAllAccessControl
        implements AccessControl
{
    @Override
    public void checkCanSetUser(Identity identity, AccessControlContext context, Optional<Principal> principal, String userName)
    {
        denySetUser(principal, userName);
    }

    @Override
    public void checkQueryIntegrity(Identity identity, AccessControlContext context, String query)
    {
        denyQueryIntegrityCheck();
    }

    @Override
    public Set<String> filterCatalogs(Identity identity, AccessControlContext context, Set<String> catalogs)
    {
        return emptySet();
    }

    @Override
    public void checkCanAccessCatalog(Identity identity, AccessControlContext context, String catalogName)
    {
        denyCatalogAccess(catalogName);
    }

    @Override
    public void checkCanCreateSchema(TransactionId transactionId, Identity identity, AccessControlContext context, CatalogSchemaName schemaName)
    {
        denyCreateSchema(schemaName.toString());
    }

    @Override
    public void checkCanDropSchema(TransactionId transactionId, Identity identity, AccessControlContext context, CatalogSchemaName schemaName)
    {
        denyDropSchema(schemaName.toString());
    }

    @Override
    public void checkCanRenameSchema(TransactionId transactionId, Identity identity, AccessControlContext context, CatalogSchemaName schemaName, String newSchemaName)
    {
        denyRenameSchema(schemaName.toString(), newSchemaName);
    }

    @Override
    public void checkCanCreateTable(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName)
    {
        denyCreateTable(tableName.toString());
    }

    @Override
    public void checkCanDropTable(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName)
    {
        denyDropTable(tableName.toString());
    }

    @Override
    public void checkCanRenameTable(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName, QualifiedObjectName newTableName)
    {
        denyRenameTable(tableName.toString(), newTableName.toString());
    }

    @Override
    public void checkCanShowTablesMetadata(TransactionId transactionId, Identity identity, AccessControlContext context, CatalogSchemaName schema)
    {
        denyShowTablesMetadata(schema.toString());
    }

    @Override
    public Set<SchemaTableName> filterTables(TransactionId transactionId, Identity identity, AccessControlContext context, String catalogName, Set<SchemaTableName> tableNames)
    {
        return emptySet();
    }

    @Override
    public void checkCanShowSchemas(TransactionId transactionId, Identity identity, AccessControlContext context, String catalogName)
    {
        denyShowSchemas();
    }

    @Override
    public Set<String> filterSchemas(TransactionId transactionId, Identity identity, AccessControlContext context, String catalogName, Set<String> schemaNames)
    {
        return emptySet();
    }

    @Override
    public void checkCanAddColumns(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName)
    {
        denyAddColumn(tableName.toString());
    }

    @Override
    public void checkCanRenameColumn(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName)
    {
        denyRenameColumn(tableName.toString());
    }

    @Override
    public void checkCanDropColumn(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName)
    {
        denyDropColumn(tableName.toString());
    }

    @Override
    public void checkCanInsertIntoTable(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName)
    {
        denyInsertTable(tableName.toString());
    }

    @Override
    public void checkCanDeleteFromTable(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName)
    {
        denyDeleteTable(tableName.toString());
    }

    @Override
    public void checkCanTruncateTable(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName)
    {
        denyTruncateTable(tableName.toString());
    }

    @Override
    public void checkCanCreateView(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName viewName)
    {
        denyCreateView(viewName.toString());
    }

    @Override
    public void checkCanDropView(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName viewName)
    {
        denyDropView(viewName.toString());
    }

    @Override
    public void checkCanCreateViewWithSelectFromColumns(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName, Set<String> columnNames)
    {
        denyCreateViewWithSelect(tableName.toString(), identity);
    }

    @Override
    public void checkCanGrantTablePrivilege(TransactionId transactionId, Identity identity, AccessControlContext context, Privilege privilege, QualifiedObjectName tableName, PrestoPrincipal grantee, boolean withGrantOption)
    {
        denyGrantTablePrivilege(privilege.name(), tableName.toString());
    }

    @Override
    public void checkCanRevokeTablePrivilege(TransactionId transactionId, Identity identity, AccessControlContext context, Privilege privilege, QualifiedObjectName tableName, PrestoPrincipal revokee, boolean grantOptionFor)
    {
        denyRevokeTablePrivilege(privilege.name(), tableName.toString());
    }

    @Override
    public void checkCanSetSystemSessionProperty(Identity identity, AccessControlContext context, String propertyName)
    {
        denySetSystemSessionProperty(propertyName);
    }

    @Override
    public void checkCanSetCatalogSessionProperty(TransactionId transactionId, Identity identity, AccessControlContext context, String catalogName, String propertyName)
    {
        denySetCatalogSessionProperty(catalogName, propertyName);
    }

    @Override
    public void checkCanSelectFromColumns(TransactionId transactionId, Identity identity, AccessControlContext context, QualifiedObjectName tableName, Set<Subfield> columnOrSubfieldNames)
    {
        denySelectColumns(tableName.toString(), columnOrSubfieldNames.stream().map(Subfield::getRootName).collect(collectingAndThen(toSet(), Collections::unmodifiableSet)));
    }

    @Override
    public void checkCanCreateRole(TransactionId transactionId, Identity identity, AccessControlContext context, String role, Optional<PrestoPrincipal> grantor, String catalogName)
    {
        denyCreateRole(role);
    }

    @Override
    public void checkCanDropRole(TransactionId transactionId, Identity identity, AccessControlContext context, String role, String catalogName)
    {
        denyDropRole(role);
    }

    @Override
    public void checkCanGrantRoles(TransactionId transactionId, Identity identity, AccessControlContext context, Set<String> roles, Set<PrestoPrincipal> grantees, boolean withAdminOption, Optional<PrestoPrincipal> grantor, String catalogName)
    {
        denyGrantRoles(roles, grantees);
    }

    @Override
    public void checkCanRevokeRoles(TransactionId transactionId, Identity identity, AccessControlContext context, Set<String> roles, Set<PrestoPrincipal> grantees, boolean adminOptionFor, Optional<PrestoPrincipal> grantor, String catalogName)
    {
        denyRevokeRoles(roles, grantees);
    }

    @Override
    public void checkCanSetRole(TransactionId requiredTransactionId, Identity identity, AccessControlContext context, String role, String catalog)
    {
        denySetRole(role);
    }

    @Override
    public void checkCanShowRoles(TransactionId transactionId, Identity identity, AccessControlContext context, String catalogName)
    {
        denyShowRoles(catalogName);
    }

    @Override
    public void checkCanShowCurrentRoles(TransactionId transactionId, Identity identity, AccessControlContext context, String catalogName)
    {
        denyShowCurrentRoles(catalogName);
    }

    @Override
    public void checkCanShowRoleGrants(TransactionId transactionId, Identity identity, AccessControlContext context, String catalogName)
    {
        denyShowRoleGrants(catalogName);
    }
}
