/*
 * Copyright (c) 2013 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package com.facebook.presto.jdbc.internal.google.api.client.util;

/**
 * Simple static methods to be called at the start of your own methods to verify correct arguments
 * and state.
 *
 * <p>
 * NOTE: proxy for the Guava implementation of {@link com.facebook.presto.jdbc.internal.guava.base.Preconditions}.
 * </p>
 *
 * @since 1.14
 * @author Yaniv Inbar
 */
public final class Preconditions {

  /**
   * Ensures the truth of an expression involving one or more parameters to the calling method.
   *
   * @param expression a boolean expression
   * @throws IllegalArgumentException if {@code expression} is false
   */
  public static void checkArgument(boolean expression) {
    com.facebook.presto.jdbc.internal.guava.base.Preconditions.checkArgument(expression);
  }

  /**
   * Ensures the truth of an expression involving one or more parameters to the calling method.
   *
   * @param expression a boolean expression
   * @param errorMessage the exception message to use if the check fails; will be converted to a
   *        string using {@link String#valueOf(Object)}
   * @throws IllegalArgumentException if {@code expression} is false
   */
  public static void checkArgument(boolean expression, Object errorMessage) {
    com.facebook.presto.jdbc.internal.guava.base.Preconditions.checkArgument(expression, errorMessage);
  }

  /**
   * Ensures the truth of an expression involving one or more parameters to the calling method.
   *
   * @param expression a boolean expression
   * @param errorMessageTemplate a template for the exception message should the check fail. The
   *        message is formed by replacing each {@code %s} placeholder in the template with an
   *        argument. These are matched by position - the first {@code %s} gets
   *        {@code errorMessageArgs[0]}, etc. Unmatched arguments will be appended to the formatted
   *        message in square braces. Unmatched placeholders will be left as-is.
   * @param errorMessageArgs the arguments to be substituted into the message template. Arguments
   *        are converted to strings using {@link String#valueOf(Object)}.
   * @throws IllegalArgumentException if {@code expression} is false
   * @throws NullPointerException if the check fails and either {@code errorMessageTemplate} or
   *         {@code errorMessageArgs} is null (don't let this happen)
   */
  public static void checkArgument(
      boolean expression, String errorMessageTemplate, Object... errorMessageArgs) {
    com.facebook.presto.jdbc.internal.guava.base.Preconditions.checkArgument(
        expression, errorMessageTemplate, errorMessageArgs);
  }

  /**
   * Ensures the truth of an expression involving the state of the calling instance, but not
   * involving any parameters to the calling method.
   *
   * @param expression a boolean expression
   * @throws IllegalStateException if {@code expression} is false
   */
  public static void checkState(boolean expression) {
    com.facebook.presto.jdbc.internal.guava.base.Preconditions.checkState(expression);
  }

  /**
   * Ensures the truth of an expression involving the state of the calling instance, but not
   * involving any parameters to the calling method.
   *
   * @param expression a boolean expression
   * @param errorMessage the exception message to use if the check fails; will be converted to a
   *        string using {@link String#valueOf(Object)}
   * @throws IllegalStateException if {@code expression} is false
   */
  public static void checkState(boolean expression, Object errorMessage) {
    com.facebook.presto.jdbc.internal.guava.base.Preconditions.checkState(expression, errorMessage);
  }

  /**
   * Ensures the truth of an expression involving the state of the calling instance, but not
   * involving any parameters to the calling method.
   *
   * @param expression a boolean expression
   * @param errorMessageTemplate a template for the exception message should the check fail. The
   *        message is formed by replacing each {@code %s} placeholder in the template with an
   *        argument. These are matched by position - the first {@code %s} gets
   *        {@code errorMessageArgs[0]}, etc. Unmatched arguments will be appended to the formatted
   *        message in square braces. Unmatched placeholders will be left as-is.
   * @param errorMessageArgs the arguments to be substituted into the message template. Arguments
   *        are converted to strings using {@link String#valueOf(Object)}.
   * @throws IllegalStateException if {@code expression} is false
   * @throws NullPointerException if the check fails and either {@code errorMessageTemplate} or
   *         {@code errorMessageArgs} is null (don't let this happen)
   */
  public static void checkState(
      boolean expression, String errorMessageTemplate, Object... errorMessageArgs) {
    com.facebook.presto.jdbc.internal.guava.base.Preconditions.checkState(
        expression, errorMessageTemplate, errorMessageArgs);
  }

  /**
   * Ensures that an object reference passed as a parameter to the calling method is not null.
   *
   * @param reference an object reference
   * @return the non-null reference that was validated
   * @throws NullPointerException if {@code reference} is null
   */
  public static <T> T checkNotNull(T reference) {
    return com.google.common.base.Preconditions.checkNotNull(reference);
  }

  /**
   * Ensures that an object reference passed as a parameter to the calling method is not null.
   *
   * @param reference an object reference
   * @param errorMessage the exception message to use if the check fails; will be converted to a
   *        string using {@link String#valueOf(Object)}
   * @return the non-null reference that was validated
   * @throws NullPointerException if {@code reference} is null
   */
  public static <T> T checkNotNull(T reference, Object errorMessage) {
    return com.google.common.base.Preconditions.checkNotNull(reference, errorMessage);
  }

  /**
   * Ensures that an object reference passed as a parameter to the calling method is not null.
   *
   * @param reference an object reference
   * @param errorMessageTemplate a template for the exception message should the check fail. The
   *        message is formed by replacing each {@code %s} placeholder in the template with an
   *        argument. These are matched by position - the first {@code %s} gets
   *        {@code errorMessageArgs[0]}, etc. Unmatched arguments will be appended to the formatted
   *        message in square braces. Unmatched placeholders will be left as-is.
   * @param errorMessageArgs the arguments to be substituted into the message template. Arguments
   *        are converted to strings using {@link String#valueOf(Object)}.
   * @return the non-null reference that was validated
   * @throws NullPointerException if {@code reference} is null
   */
  public static <T> T checkNotNull(
      T reference, String errorMessageTemplate, Object... errorMessageArgs) {
    return com.google.common.base.Preconditions.checkNotNull(
        reference, errorMessageTemplate, errorMessageArgs);
  }

  private Preconditions() {
  }
}
