package com.facebook.presto.jdbc.internal.jackson.datatype.joda.deser;

import java.io.IOException;

import com.facebook.presto.jdbc.internal.jackson.core.*;
import com.facebook.presto.jdbc.internal.jackson.core.io.NumberInput;
import com.facebook.presto.jdbc.internal.jackson.databind.DeserializationContext;
import com.facebook.presto.jdbc.internal.jackson.databind.JavaType;
import com.facebook.presto.jdbc.internal.jackson.databind.cfg.CoercionAction;
import com.facebook.presto.jdbc.internal.jackson.databind.deser.std.StdScalarDeserializer;
import com.facebook.presto.jdbc.internal.jackson.databind.jsontype.TypeDeserializer;
import com.facebook.presto.jdbc.internal.jackson.databind.type.LogicalType;
import com.facebook.presto.jdbc.internal.jackson.databind.util.ClassUtil;

@SuppressWarnings("serial")
abstract class JodaDeserializerBase<T> extends StdScalarDeserializer<T>
{
    protected JodaDeserializerBase(Class<?> cls) {
        super(cls);
    }

    protected JodaDeserializerBase(JodaDeserializerBase<?> src) {
        super(src);
    }
    
    @Override
    public Object deserializeWithType(JsonParser p, DeserializationContext ctxt,
            TypeDeserializer typeDeserializer) throws IOException
    {
        return typeDeserializer.deserializeTypedFromAny(p, ctxt);
    }

    @Override
    public LogicalType logicalType() { return LogicalType.DateTime; }

    // @since 2.12
    protected boolean _isValidTimestampString(String str) {
        // 14-Jul-2020, tatu: Need to support "numbers as Strings" for data formats
        //    that only have String values for scalars (CSV, Properties, XML)
        // NOTE: we do allow negative values, but has to fit in 64-bits:
        return _isIntNumber(str) && NumberInput.inLongRange(str, (str.charAt(0) == '-'));
    }

    /**
     * Helper method for specific case of deserialization
     * from empty or blank String.
     *
     * @since 2.12
     */
    @SuppressWarnings("unchecked")
    protected T _fromEmptyString(JsonParser p, DeserializationContext ctxt,
            String str)
        throws IOException
    {
        final CoercionAction act = _checkFromStringCoercion(ctxt, str);
        switch (act) { // note: Fail handled above
        case AsEmpty:
            return (T) getEmptyValue(ctxt);
        case TryConvert:
        case AsNull:
        default:
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    public T _handleNotNumberOrString(JsonParser p, DeserializationContext ctxt)
        throws IOException
    {
        final JavaType type = getValueType(ctxt);
        return (T) ctxt.handleUnexpectedToken(type, p.currentToken(), p,
                String.format("Cannot deserialize value of type %s from `JsonToken.%s`: expected Number or String",
                        ClassUtil.getTypeDescription(type), p.currentToken()));
    }
}
