/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.facebook.presto.jdbc.internal.spi.plan;

import com.facebook.presto.jdbc.internal.spi.relation.VariableReferenceExpression;
import com.facebook.presto.jdbc.internal.jackson.annotation.JsonCreator;
import com.facebook.presto.jdbc.internal.jackson.annotation.JsonProperty;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static com.facebook.presto.jdbc.internal.common.Utils.checkArgument;
import static com.facebook.presto.jdbc.internal.spi.plan.ExchangeEncoding.COLUMNAR;
import static com.facebook.presto.jdbc.internal.spi.plan.ExchangeEncoding.ROW_WISE;
import static java.lang.String.format;
import static java.util.Collections.unmodifiableList;
import static java.util.Collections.unmodifiableSet;
import static java.util.Objects.requireNonNull;

public class PartitioningScheme
{
    private final Partitioning partitioning;
    private final List<VariableReferenceExpression> outputLayout;
    private final Optional<VariableReferenceExpression> hashColumn;
    private final boolean replicateNullsAndAny;
    private final boolean scaleWriters;
    private final ExchangeEncoding encoding;
    private final Optional<int[]> bucketToPartition;

    public PartitioningScheme(Partitioning partitioning, List<VariableReferenceExpression> outputLayout, boolean scaleWriters)
    {
        this(
                partitioning,
                outputLayout,
                Optional.empty(),
                false,
                scaleWriters,
                COLUMNAR,
                Optional.empty());
    }

    public PartitioningScheme(Partitioning partitioning, List<VariableReferenceExpression> outputLayout)
    {
        this(
                partitioning,
                outputLayout,
                Optional.empty(),
                false,
                false,
                COLUMNAR,
                Optional.empty());
    }

    public PartitioningScheme(Partitioning partitioning, List<VariableReferenceExpression> outputLayout, Optional<VariableReferenceExpression> hashColumn)
    {
        this(
                partitioning,
                outputLayout,
                hashColumn,
                false,
                false,
                COLUMNAR,
                Optional.empty());
    }

    @JsonCreator
    public PartitioningScheme(
            @JsonProperty("partitioning") Partitioning partitioning,
            @JsonProperty("outputLayout") List<VariableReferenceExpression> outputLayout,
            @JsonProperty("hashColumn") Optional<VariableReferenceExpression> hashColumn,
            @JsonProperty("replicateNullsAndAny") boolean replicateNullsAndAny,
            @JsonProperty("scaleWriters") boolean scaleWriters,
            @JsonProperty("encoding") ExchangeEncoding encoding,
            @JsonProperty("bucketToPartition") Optional<int[]> bucketToPartition)
    {
        this.partitioning = requireNonNull(partitioning, "partitioning is null");
        this.outputLayout = unmodifiableList(requireNonNull(outputLayout, "outputLayout is null"));

        Set<VariableReferenceExpression> columns = partitioning.getVariableReferences();
        Set<VariableReferenceExpression> outputLayoutSet = unmodifiableSet(new HashSet<>(outputLayout));
        checkArgument(outputLayoutSet.containsAll(columns),
                format("Output layout (%s) don't include all partition columns (%s)", outputLayout, columns));

        this.hashColumn = requireNonNull(hashColumn, "hashColumn is null");
        hashColumn.ifPresent(column -> checkArgument(outputLayout.contains(column),
                format("Output layout (%s) don't include hash column (%s)", outputLayout, column)));

        checkArgument(!replicateNullsAndAny || columns.size() <= 1, "Must have at most one partitioning column when nullPartition is REPLICATE.");
        this.replicateNullsAndAny = replicateNullsAndAny;
        this.scaleWriters = scaleWriters;
        this.encoding = requireNonNull(encoding, "encoding is null");
        this.bucketToPartition = requireNonNull(bucketToPartition, "bucketToPartition is null");
    }

    @JsonProperty
    public Partitioning getPartitioning()
    {
        return partitioning;
    }

    @JsonProperty
    public List<VariableReferenceExpression> getOutputLayout()
    {
        return outputLayout;
    }

    @JsonProperty
    public Optional<VariableReferenceExpression> getHashColumn()
    {
        return hashColumn;
    }

    @JsonProperty
    public boolean isReplicateNullsAndAny()
    {
        return replicateNullsAndAny;
    }

    @JsonProperty
    public ExchangeEncoding getEncoding()
    {
        return encoding;
    }

    @JsonProperty
    public boolean isScaleWriters()
    {
        return scaleWriters;
    }

    @JsonProperty
    public Optional<int[]> getBucketToPartition()
    {
        return bucketToPartition;
    }

    public PartitioningScheme withBucketToPartition(Optional<int[]> bucketToPartition)
    {
        return new PartitioningScheme(
                partitioning,
                outputLayout,
                hashColumn,
                replicateNullsAndAny,
                scaleWriters,
                encoding,
                bucketToPartition);
    }

    public PartitioningScheme withRowWiseEncoding()
    {
        return new PartitioningScheme(
                partitioning,
                outputLayout,
                hashColumn,
                replicateNullsAndAny,
                scaleWriters,
                ROW_WISE,
                bucketToPartition);
    }

    public boolean isSingleOrBroadcastOrArbitrary()
    {
        return partitioning.isSingleOrBroadcastOrArbitrary();
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PartitioningScheme that = (PartitioningScheme) o;
        return Objects.equals(partitioning, that.partitioning) &&
                Objects.equals(outputLayout, that.outputLayout) &&
                replicateNullsAndAny == that.replicateNullsAndAny &&
                scaleWriters == that.scaleWriters &&
                encoding == that.encoding &&
                Objects.equals(bucketToPartition, that.bucketToPartition);
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(partitioning, outputLayout, replicateNullsAndAny, encoding, scaleWriters, bucketToPartition);
    }

    @Override
    public String toString()
    {
        String sb = "ConnectorTableMetadata{" + "partitioning=" + partitioning +
                ", outputLayout=" + outputLayout +
                ", hashChannel=" + hashColumn +
                ", replicateNullsAndAny=" + replicateNullsAndAny +
                ", scaleWriters=" + scaleWriters +
                ", encoding=" + encoding +
                ", bucketToPartition=" + bucketToPartition +
                '}';
        return sb;
    }
}
