/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.execution;

import com.facebook.airlift.log.Logger;
import com.facebook.presto.execution.MemoryRevokingUtils;
import com.facebook.presto.execution.SqlTask;
import com.facebook.presto.execution.SqlTaskManager;
import com.facebook.presto.execution.TaskId;
import com.facebook.presto.execution.TaskManagementExecutor;
import com.facebook.presto.execution.TaskState;
import com.facebook.presto.memory.LocalMemoryManager;
import com.facebook.presto.memory.MemoryPool;
import com.facebook.presto.memory.MemoryPoolListener;
import com.facebook.presto.memory.TraversingQueryContextVisitor;
import com.facebook.presto.memory.VoidTraversingQueryContextVisitor;
import com.facebook.presto.operator.OperatorContext;
import com.facebook.presto.operator.PipelineContext;
import com.facebook.presto.operator.TaskContext;
import com.facebook.presto.spi.memory.MemoryPoolId;
import com.facebook.presto.sql.analyzer.FeaturesConfig;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Ordering;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;

public class MemoryRevokingScheduler {
    private static final Logger log = Logger.get(MemoryRevokingScheduler.class);
    private static final Ordering<SqlTask> ORDER_BY_CREATE_TIME = Ordering.natural().onResultOf(SqlTask::getTaskCreatedTime);
    private final List<MemoryPool> memoryPools;
    private final Supplier<List<SqlTask>> currentTasksSupplier;
    private final ScheduledExecutorService taskManagementExecutor;
    private final double memoryRevokingThreshold;
    private final double memoryRevokingTarget;
    private final FeaturesConfig.TaskSpillingStrategy spillingStrategy;
    private final MemoryPoolListener memoryPoolListener = MemoryPoolListener.onMemoryReserved(this::onMemoryReserved);
    @Nullable
    private ScheduledFuture<?> scheduledFuture;
    private final AtomicBoolean checkPending = new AtomicBoolean();

    @Inject
    public MemoryRevokingScheduler(LocalMemoryManager localMemoryManager, SqlTaskManager sqlTaskManager, TaskManagementExecutor taskManagementExecutor, FeaturesConfig config) {
        this((List<MemoryPool>)ImmutableList.copyOf(MemoryRevokingUtils.getMemoryPools(localMemoryManager)), Objects.requireNonNull(sqlTaskManager, "sqlTaskManager cannot be null")::getAllTasks, Objects.requireNonNull(taskManagementExecutor, "taskManagementExecutor cannot be null").getExecutor(), config.getMemoryRevokingThreshold(), config.getMemoryRevokingTarget(), config.getTaskSpillingStrategy());
    }

    @VisibleForTesting
    MemoryRevokingScheduler(List<MemoryPool> memoryPools, Supplier<List<SqlTask>> currentTasksSupplier, ScheduledExecutorService taskManagementExecutor, double memoryRevokingThreshold, double memoryRevokingTarget, FeaturesConfig.TaskSpillingStrategy taskSpillingStrategy) {
        this.memoryPools = ImmutableList.copyOf((Collection)Objects.requireNonNull(memoryPools, "memoryPools is null"));
        this.currentTasksSupplier = Objects.requireNonNull(currentTasksSupplier, "currentTasksSupplier is null");
        this.taskManagementExecutor = Objects.requireNonNull(taskManagementExecutor, "taskManagementExecutor is null");
        this.memoryRevokingThreshold = MemoryRevokingScheduler.checkFraction(memoryRevokingThreshold, "memoryRevokingThreshold");
        this.memoryRevokingTarget = MemoryRevokingScheduler.checkFraction(memoryRevokingTarget, "memoryRevokingTarget");
        this.spillingStrategy = Objects.requireNonNull(taskSpillingStrategy, "taskSpillingStrategy is null");
        Preconditions.checkArgument((this.spillingStrategy != FeaturesConfig.TaskSpillingStrategy.PER_TASK_MEMORY_THRESHOLD ? 1 : 0) != 0, (Object)"spilling strategy cannot be PER_TASK_MEMORY_THRESHOLD in MemoryRevokingScheduler");
        Preconditions.checkArgument((memoryRevokingTarget <= memoryRevokingThreshold ? 1 : 0) != 0, (String)"memoryRevokingTarget should be less than or equal memoryRevokingThreshold, but got %s and %s respectively", (Object)memoryRevokingTarget, (Object)memoryRevokingThreshold);
    }

    private static double checkFraction(double value, String valueName) {
        Objects.requireNonNull(valueName, "valueName is null");
        Preconditions.checkArgument((0.0 <= value && value <= 1.0 ? 1 : 0) != 0, (String)"%s should be within [0, 1] range, got %s", (Object)valueName, (Object)value);
        return value;
    }

    @PostConstruct
    public void start() {
        this.registerPeriodicCheck();
        this.registerPoolListeners();
    }

    private void registerPeriodicCheck() {
        this.scheduledFuture = this.taskManagementExecutor.scheduleWithFixedDelay(() -> {
            try {
                this.requestMemoryRevokingIfNeeded();
            }
            catch (Throwable e) {
                log.error(e, "Error requesting system memory revoking");
            }
        }, 1L, 1L, TimeUnit.SECONDS);
    }

    @PreDestroy
    public void stop() {
        if (this.scheduledFuture != null) {
            this.scheduledFuture.cancel(true);
            this.scheduledFuture = null;
        }
        this.memoryPools.forEach(memoryPool -> memoryPool.removeListener(this.memoryPoolListener));
    }

    @VisibleForTesting
    void registerPoolListeners() {
        this.memoryPools.forEach(memoryPool -> memoryPool.addListener(this.memoryPoolListener));
    }

    private void onMemoryReserved(MemoryPool memoryPool) {
        try {
            if (!this.memoryRevokingNeeded(memoryPool)) {
                return;
            }
            if (this.checkPending.compareAndSet(false, true)) {
                log.debug("Scheduling check for %s", new Object[]{memoryPool});
                this.scheduleRevoking();
            }
        }
        catch (Throwable e) {
            log.error(e, "Error when acting on memory pool reservation");
        }
    }

    @VisibleForTesting
    void requestMemoryRevokingIfNeeded() {
        if (this.checkPending.compareAndSet(false, true)) {
            this.runMemoryRevoking();
        }
    }

    private void scheduleRevoking() {
        this.taskManagementExecutor.execute(() -> {
            try {
                this.runMemoryRevoking();
            }
            catch (Throwable e) {
                log.error(e, "Error requesting memory revoking");
            }
        });
    }

    private synchronized void runMemoryRevoking() {
        if (this.checkPending.getAndSet(false)) {
            Collection allTasks = null;
            for (MemoryPool memoryPool : this.memoryPools) {
                if (!this.memoryRevokingNeeded(memoryPool)) continue;
                if (allTasks == null) {
                    allTasks = Objects.requireNonNull(this.currentTasksSupplier.get());
                }
                this.requestMemoryRevoking(memoryPool, allTasks);
            }
        }
    }

    private void requestMemoryRevoking(MemoryPool memoryPool, Collection<SqlTask> allTasks) {
        long remainingBytesToRevoke = (long)((double)(-memoryPool.getFreeBytes()) + (double)memoryPool.getMaxBytes() * (1.0 - this.memoryRevokingTarget));
        ArrayList<SqlTask> runningTasksInPool = MemoryRevokingScheduler.findRunningTasksInMemoryPool(allTasks, memoryPool);
        if ((remainingBytesToRevoke -= this.getMemoryAlreadyBeingRevoked(runningTasksInPool, remainingBytesToRevoke)) > 0L) {
            this.requestRevoking(memoryPool.getId(), runningTasksInPool, remainingBytesToRevoke);
        }
    }

    private boolean memoryRevokingNeeded(MemoryPool memoryPool) {
        return memoryPool.getReservedRevocableBytes() > 0L && (double)memoryPool.getFreeBytes() <= (double)memoryPool.getMaxBytes() * (1.0 - this.memoryRevokingThreshold);
    }

    private long getMemoryAlreadyBeingRevoked(List<SqlTask> sqlTasks, long targetRevokingLimit) {
        TraversingQueryContextVisitor<Void, Long> visitor = new TraversingQueryContextVisitor<Void, Long>(){

            @Override
            public Long visitOperatorContext(OperatorContext operatorContext, Void context) {
                if (operatorContext.isMemoryRevokingRequested()) {
                    return operatorContext.getReservedRevocableBytes();
                }
                return 0L;
            }

            @Override
            public Long mergeResults(List<Long> childrenResults) {
                return childrenResults.stream().mapToLong(i -> i).sum();
            }
        };
        long currentRevoking = 0L;
        for (SqlTask task : sqlTasks) {
            Optional<TaskContext> taskContext = task.getTaskContext();
            if (!taskContext.isPresent() || (currentRevoking += taskContext.get().accept(visitor, null).longValue()) <= targetRevokingLimit) continue;
            return currentRevoking;
        }
        return currentRevoking;
    }

    private void requestRevoking(final MemoryPoolId memoryPoolId, ArrayList<SqlTask> sqlTasks, long remainingBytesToRevoke) {
        VoidTraversingQueryContextVisitor<AtomicLong> visitor = new VoidTraversingQueryContextVisitor<AtomicLong>(){

            @Override
            public Void visitPipelineContext(PipelineContext pipelineContext, AtomicLong remainingBytesToRevoke) {
                if (remainingBytesToRevoke.get() <= 0L) {
                    return null;
                }
                return (Void)super.visitPipelineContext(pipelineContext, remainingBytesToRevoke);
            }

            @Override
            public Void visitOperatorContext(OperatorContext operatorContext, AtomicLong remainingBytesToRevoke) {
                long revokedBytes;
                if (remainingBytesToRevoke.get() > 0L && (revokedBytes = operatorContext.requestMemoryRevoking()) > 0L) {
                    remainingBytesToRevoke.addAndGet(-revokedBytes);
                    log.debug("memoryPool=%s: requested revoking %s; remaining %s", new Object[]{memoryPoolId, revokedBytes, remainingBytesToRevoke.get()});
                }
                return null;
            }
        };
        log.debug("Ordering by %s", new Object[]{this.spillingStrategy});
        MemoryRevokingScheduler.sortTasksToTraversalOrder(sqlTasks, this.spillingStrategy);
        AtomicLong remainingBytesToRevokeAtomic = new AtomicLong(remainingBytesToRevoke);
        for (SqlTask task : sqlTasks) {
            Optional<TaskContext> taskContext = task.getTaskContext();
            if (!taskContext.isPresent()) continue;
            taskContext.get().accept(visitor, remainingBytesToRevokeAtomic);
            if (remainingBytesToRevokeAtomic.get() > 0L) continue;
            return;
        }
    }

    private static void sortTasksToTraversalOrder(ArrayList<SqlTask> sqlTasks, FeaturesConfig.TaskSpillingStrategy spillingStrategy) {
        switch (spillingStrategy) {
            case ORDER_BY_CREATE_TIME: {
                sqlTasks.sort((Comparator<SqlTask>)ORDER_BY_CREATE_TIME);
                break;
            }
            case ORDER_BY_REVOCABLE_BYTES: {
                HashMap<TaskId, Long> taskRevocableReservations = new HashMap<TaskId, Long>();
                for (SqlTask sqlTask : sqlTasks) {
                    taskRevocableReservations.put(sqlTask.getTaskId(), sqlTask.getTaskInfo().getStats().getRevocableMemoryReservationInBytes());
                }
                sqlTasks.sort((Comparator<SqlTask>)Ordering.natural().reverse().onResultOf(task -> task == null ? Long.valueOf(0L) : (Comparable)taskRevocableReservations.getOrDefault(task.getTaskId(), 0L)));
                break;
            }
            case PER_TASK_MEMORY_THRESHOLD: {
                throw new IllegalArgumentException("spilling strategy cannot be PER_TASK_MEMORY_THRESHOLD in MemoryRevokingScheduler");
            }
            default: {
                throw new UnsupportedOperationException("Unexpected spilling strategy in MemoryRevokingScheduler");
            }
        }
    }

    private static ArrayList<SqlTask> findRunningTasksInMemoryPool(Collection<SqlTask> allCurrentTasks, MemoryPool memoryPool) {
        ArrayList<SqlTask> sqlTasks = new ArrayList<SqlTask>();
        allCurrentTasks.stream().filter(task -> task.getTaskState() == TaskState.RUNNING && task.getQueryContext().getMemoryPool() == memoryPool).forEach(sqlTasks::add);
        return sqlTasks;
    }
}

