/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.operator;

import com.facebook.airlift.log.Logger;
import com.facebook.presto.common.Page;
import com.facebook.presto.common.block.BlockEncodingSerde;
import com.facebook.presto.execution.buffer.PagesSerdeFactory;
import com.facebook.presto.metadata.Split;
import com.facebook.presto.operator.FileFragmentResultCacheConfig;
import com.facebook.presto.operator.FragmentCacheStats;
import com.facebook.presto.operator.FragmentResultCacheManager;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.page.PagesSerde;
import com.facebook.presto.spi.page.PagesSerdeUtil;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.AbstractIterator;
import com.google.common.util.concurrent.Futures;
import io.airlift.slice.InputStreamSliceInput;
import io.airlift.slice.OutputStreamSliceOutput;
import io.airlift.slice.SliceInput;
import io.airlift.slice.SliceOutput;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;
import org.weakref.jmx.Managed;

public class FileFragmentResultCacheManager
implements FragmentResultCacheManager {
    private static final Logger log = Logger.get(FileFragmentResultCacheManager.class);
    private final Path baseDirectory;
    private final long maxInFlightBytes;
    private final long maxSinglePagesBytes;
    private final long maxCacheBytes;
    private final PagesSerdeFactory pagesSerdeFactory;
    private final FragmentCacheStats fragmentCacheStats;
    private final ExecutorService flushExecutor;
    private final ExecutorService removalExecutor;
    private final Cache<CacheKey, CacheEntry> cache;

    @Inject
    public FileFragmentResultCacheManager(FileFragmentResultCacheConfig cacheConfig, BlockEncodingSerde blockEncodingSerde, FragmentCacheStats fragmentCacheStats, ExecutorService flushExecutor, ExecutorService removalExecutor) {
        Objects.requireNonNull(cacheConfig, "cacheConfig is null");
        Objects.requireNonNull(blockEncodingSerde, "blockEncodingSerde is null");
        this.baseDirectory = Paths.get(cacheConfig.getBaseDirectory());
        this.maxInFlightBytes = cacheConfig.getMaxInFlightSize().toBytes();
        this.maxSinglePagesBytes = cacheConfig.getMaxSinglePagesSize().toBytes();
        this.maxCacheBytes = cacheConfig.getMaxCacheSize().toBytes();
        this.pagesSerdeFactory = new PagesSerdeFactory(blockEncodingSerde, cacheConfig.isBlockEncodingCompressionEnabled());
        this.fragmentCacheStats = Objects.requireNonNull(fragmentCacheStats, "fragmentCacheStats is null");
        this.flushExecutor = Objects.requireNonNull(flushExecutor, "flushExecutor is null");
        this.removalExecutor = Objects.requireNonNull(removalExecutor, "removalExecutor is null");
        this.cache = CacheBuilder.newBuilder().maximumSize((long)cacheConfig.getMaxCachedEntries()).expireAfterAccess(cacheConfig.getCacheTtl().toMillis(), TimeUnit.MILLISECONDS).removalListener((RemovalListener)new CacheRemovalListener()).recordStats().build();
        File target = new File(this.baseDirectory.toUri());
        if (!target.exists()) {
            try {
                Files.createDirectories(target.toPath(), new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "cannot create cache directory " + target, (Throwable)e);
            }
        } else {
            File[] files = target.listFiles();
            if (files == null) {
                return;
            }
            this.removalExecutor.submit(() -> Arrays.stream(files).forEach(file -> {
                try {
                    Files.delete(file.toPath());
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }));
        }
    }

    @Override
    public Future<?> put(String serializedPlan, Split split, List<Page> result) {
        CacheKey key = new CacheKey(serializedPlan, split.getSplitIdentifier());
        long resultSize = FileFragmentResultCacheManager.getPagesSize(result);
        if (this.fragmentCacheStats.getInFlightBytes() + resultSize > this.maxInFlightBytes || this.cache.getIfPresent((Object)key) != null || resultSize > this.maxSinglePagesBytes || this.fragmentCacheStats.getCacheSizeInBytes() + resultSize > this.maxCacheBytes) {
            return Futures.immediateFuture(null);
        }
        this.fragmentCacheStats.addInFlightBytes(resultSize);
        Path path = this.baseDirectory.resolve(UUID.randomUUID().toString().replaceAll("-", "_"));
        return this.flushExecutor.submit(() -> this.cachePages(key, path, result, resultSize));
    }

    private static long getPagesSize(List<Page> pages) {
        return pages.stream().mapToLong(Page::getSizeInBytes).sum();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cachePages(CacheKey key, Path path, List<Page> pages, long resultSize) {
        try {
            Files.createFile(path, new FileAttribute[0]);
            try (OutputStreamSliceOutput output = new OutputStreamSliceOutput(Files.newOutputStream(path, StandardOpenOption.APPEND));){
                PagesSerdeUtil.writePages((PagesSerde)this.pagesSerdeFactory.createPagesSerde(), (SliceOutput)output, pages.iterator());
                long resultPhysicalBytes = output.size();
                this.cache.put((Object)key, (Object)new CacheEntry(path, resultPhysicalBytes));
                this.fragmentCacheStats.incrementCacheEntries();
                this.fragmentCacheStats.addCacheSizeInBytes(resultPhysicalBytes);
            }
            catch (IOException | UncheckedIOException e) {
                log.warn((Throwable)e, "%s encountered an error while writing to path %s", new Object[]{Thread.currentThread().getName(), path});
                FileFragmentResultCacheManager.tryDeleteFile(path);
            }
        }
        catch (IOException | UncheckedIOException e) {
            log.warn((Throwable)e, "%s encountered an error while writing to path %s", new Object[]{Thread.currentThread().getName(), path});
            FileFragmentResultCacheManager.tryDeleteFile(path);
        }
        finally {
            this.fragmentCacheStats.addInFlightBytes(-resultSize);
        }
    }

    private static void tryDeleteFile(Path path) {
        try {
            File file = new File(path.toUri());
            if (file.exists()) {
                Files.delete(file.toPath());
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Override
    public Optional<Iterator<Page>> get(String serializedPlan, Split split) {
        CacheKey key = new CacheKey(serializedPlan, split.getSplitIdentifier());
        CacheEntry cacheEntry = (CacheEntry)this.cache.getIfPresent((Object)key);
        if (cacheEntry == null) {
            this.fragmentCacheStats.incrementCacheMiss();
            return Optional.empty();
        }
        try {
            InputStream inputStream = Files.newInputStream(cacheEntry.getPath(), new OpenOption[0]);
            Iterator result = PagesSerdeUtil.readPages((PagesSerde)this.pagesSerdeFactory.createPagesSerde(), (SliceInput)new InputStreamSliceInput(inputStream));
            this.fragmentCacheStats.incrementCacheHit();
            return Optional.of(FileFragmentResultCacheManager.closeWhenExhausted(result, inputStream));
        }
        catch (IOException | UncheckedIOException e) {
            log.error((Throwable)e, "read path %s error", new Object[]{cacheEntry.getPath()});
            this.fragmentCacheStats.incrementCacheMiss();
            return Optional.empty();
        }
    }

    @Managed
    public void invalidateAllCache() {
        this.cache.invalidateAll();
    }

    private static <T> Iterator<T> closeWhenExhausted(final Iterator<T> iterator, final Closeable resource) {
        Objects.requireNonNull(iterator, "iterator is null");
        Objects.requireNonNull(resource, "resource is null");
        return new AbstractIterator<T>(){

            protected T computeNext() {
                if (iterator.hasNext()) {
                    return iterator.next();
                }
                try {
                    resource.close();
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
                return this.endOfData();
            }
        };
    }

    private class CacheRemovalListener
    implements RemovalListener<CacheKey, CacheEntry> {
        private CacheRemovalListener() {
        }

        public void onRemoval(RemovalNotification<CacheKey, CacheEntry> notification) {
            CacheEntry cacheEntry = (CacheEntry)notification.getValue();
            FileFragmentResultCacheManager.this.removalExecutor.submit(() -> FileFragmentResultCacheManager.tryDeleteFile(cacheEntry.getPath()));
            FileFragmentResultCacheManager.this.fragmentCacheStats.incrementCacheRemoval();
            FileFragmentResultCacheManager.this.fragmentCacheStats.decrementCacheEntries();
            FileFragmentResultCacheManager.this.fragmentCacheStats.addCacheSizeInBytes(-cacheEntry.getResultBytes());
        }
    }

    private static class CacheEntry {
        private final Path path;
        private final long resultBytes;

        public Path getPath() {
            return this.path;
        }

        public long getResultBytes() {
            return this.resultBytes;
        }

        public CacheEntry(Path path, long resultBytes) {
            this.path = Objects.requireNonNull(path, "path is null");
            this.resultBytes = resultBytes;
        }
    }

    public static class CacheKey {
        private final String serializedPlan;
        private final Split.SplitIdentifier splitIdentifier;

        public CacheKey(String serializedPlan, Split.SplitIdentifier splitIdentifier) {
            this.serializedPlan = Objects.requireNonNull(serializedPlan, "serializedPlan is null");
            this.splitIdentifier = Objects.requireNonNull(splitIdentifier, "splitIdentifier is null");
        }

        public String getSerializedPlan() {
            return this.serializedPlan;
        }

        public Split.SplitIdentifier getSplitIdentifier() {
            return this.splitIdentifier;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CacheKey cacheKey = (CacheKey)o;
            return Objects.equals(this.serializedPlan, cacheKey.serializedPlan) && Objects.equals(this.splitIdentifier, cacheKey.splitIdentifier);
        }

        public int hashCode() {
            return Objects.hash(this.serializedPlan, this.splitIdentifier);
        }
    }
}

