/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.geospatial;

import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.ogc.OGCGeometry;
import com.facebook.presto.common.PageBuilder;
import com.facebook.presto.common.block.Block;
import com.facebook.presto.common.block.BlockBuilder;
import com.facebook.presto.common.function.OperatorType;
import com.facebook.presto.common.type.BigintType;
import com.facebook.presto.common.type.IntegerType;
import com.facebook.presto.common.type.RowType;
import com.facebook.presto.geospatial.BingTile;
import com.facebook.presto.geospatial.BingTileUtils;
import com.facebook.presto.geospatial.GeometryUtils;
import com.facebook.presto.geospatial.serde.EsriGeometrySerde;
import com.facebook.presto.geospatial.serde.GeometrySerializationType;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.function.Description;
import com.facebook.presto.spi.function.ScalarFunction;
import com.facebook.presto.spi.function.ScalarOperator;
import com.facebook.presto.spi.function.SqlType;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.util.List;

public class BingTileFunctions {
    private static final int OPTIMIZED_TILING_MIN_ZOOM_LEVEL = 10;
    private static final Block EMPTY_TILE_ARRAY = BigintType.BIGINT.createFixedSizeBlockBuilder(0).build();

    private BingTileFunctions() {
    }

    @Description(value="Encodes a Bing tile into a bigint")
    @ScalarOperator(value=OperatorType.CAST)
    @SqlType(value="bigint")
    public static long castToBigint(@SqlType(value="BingTile") long tile) {
        return tile;
    }

    @Description(value="Decodes a Bing tile from a bigint")
    @ScalarOperator(value=OperatorType.CAST)
    @SqlType(value="BingTile")
    public static long castFromBigint(@SqlType(value="bigint") long tile) {
        try {
            BingTile.decode(tile);
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_CAST_ARGUMENT, String.format("Invalid bigint tile encoding: %s", tile));
        }
        return tile;
    }

    @Description(value="Creates a Bing tile from XY coordinates and zoom level")
    @ScalarFunction(value="bing_tile")
    @SqlType(value="BingTile")
    public static long toBingTile(@SqlType(value="integer") long tileX, @SqlType(value="integer") long tileY, @SqlType(value="integer") long zoomLevel) {
        BingTileUtils.checkZoomLevel(zoomLevel);
        BingTileUtils.checkCoordinate(tileX, zoomLevel);
        BingTileUtils.checkCoordinate(tileY, zoomLevel);
        return BingTile.fromCoordinates(Math.toIntExact(tileX), Math.toIntExact(tileY), Math.toIntExact(zoomLevel)).encode();
    }

    @Description(value="Given a Bing tile, returns its QuadKey")
    @ScalarFunction(value="bing_tile_quadkey")
    @SqlType(value="varchar")
    public static Slice toQuadKey(@SqlType(value="BingTile") long input) {
        return Slices.utf8Slice((String)BingTile.decode(input).toQuadKey());
    }

    @Description(value="Given a Bing tile, returns zoom level of the tile")
    @ScalarFunction(value="bing_tile_zoom_level")
    @SqlType(value="tinyint")
    public static long bingTileZoomLevel(@SqlType(value="BingTile") long input) {
        return BingTile.decode(input).getZoomLevel();
    }

    @Description(value="Creates a Bing tile from a QuadKey")
    @ScalarFunction(value="bing_tile")
    @SqlType(value="BingTile")
    public static long toBingTile(@SqlType(value="varchar") Slice quadKey) {
        BingTileUtils.checkQuadKey(quadKey);
        return BingTile.fromQuadKey(quadKey.toStringUtf8()).encode();
    }

    @Description(value="Given a (latitude, longitude) point, returns the containing Bing tile at the specified zoom level")
    @ScalarFunction(value="bing_tile_at")
    @SqlType(value="BingTile")
    public static long bingTileAt(@SqlType(value="double") double latitude, @SqlType(value="double") double longitude, @SqlType(value="integer") long zoomLevel) {
        BingTileUtils.checkLatitude(latitude, "Latitude must be between -85.05112878 and 85.05112878");
        BingTileUtils.checkLongitude(longitude, "Longitude must be between -180.0 and 180.0");
        BingTileUtils.checkZoomLevel(zoomLevel);
        return BingTileUtils.latitudeLongitudeToTile(latitude, longitude, Math.toIntExact(zoomLevel)).encode();
    }

    @Description(value="Given a (longitude, latitude) point, returns the surrounding Bing tiles at the specified zoom level")
    @ScalarFunction(value="bing_tiles_around")
    @SqlType(value="array(BingTile)")
    public static Block bingTilesAround(@SqlType(value="double") double latitude, @SqlType(value="double") double longitude, @SqlType(value="integer") long zoomLevel) {
        BingTileUtils.checkLatitude(latitude, "Latitude must be between -85.05112878 and 85.05112878");
        BingTileUtils.checkLongitude(longitude, "Longitude must be between -180.0 and 180.0");
        BingTileUtils.checkZoomLevel(zoomLevel);
        long mapSize = BingTileUtils.mapSize(Math.toIntExact(zoomLevel));
        long maxTileIndex = mapSize / 256L - 1L;
        int tileX = BingTileUtils.longitudeToTileX(longitude, mapSize);
        int tileY = BingTileUtils.latitudeToTileY(latitude, mapSize);
        BlockBuilder blockBuilder = BigintType.BIGINT.createBlockBuilder(null, 9);
        for (int i = -1; i <= 1; ++i) {
            for (int j = -1; j <= 1; ++j) {
                int x = tileX + i;
                int y = tileY + j;
                if (x < 0 || (long)x > maxTileIndex || y < 0 || (long)y > maxTileIndex) continue;
                BigintType.BIGINT.writeLong(blockBuilder, BingTile.fromCoordinates(x, y, Math.toIntExact(zoomLevel)).encode());
            }
        }
        return blockBuilder.build();
    }

    @Description(value="Given a (latitude, longitude) point, a radius in kilometers and a zoom level, returns a minimum set of Bing tiles at specified zoom level that cover a circle of specified radius around the specified point.")
    @ScalarFunction(value="bing_tiles_around")
    @SqlType(value="array(BingTile)")
    public static Block bingTilesAround(@SqlType(value="double") double latitude, @SqlType(value="double") double longitude, @SqlType(value="integer") long zoomLevelAsLong, @SqlType(value="double") double radiusInKm) {
        BingTile tile;
        int y;
        boolean include;
        int x;
        BingTileUtils.checkLatitude(latitude, "Latitude must be between -85.05112878 and 85.05112878");
        BingTileUtils.checkLongitude(longitude, "Longitude must be between -180.0 and 180.0");
        BingTileUtils.checkZoomLevel(zoomLevelAsLong);
        BingTileUtils.checkCondition(radiusInKm >= 0.0, "Radius must be >= 0", new Object[0]);
        BingTileUtils.checkCondition(radiusInKm <= 1000.0, "Radius must be <= 1,000 km", new Object[0]);
        int zoomLevel = Math.toIntExact(zoomLevelAsLong);
        long mapSize = BingTileUtils.mapSize(zoomLevel);
        int maxTileIndex = (int)(mapSize / 256L) - 1;
        int tileY = BingTileUtils.latitudeToTileY(latitude, mapSize);
        int tileX = BingTileUtils.longitudeToTileX(longitude, mapSize);
        double topLatitude = BingTileFunctions.addDistanceToLatitude(latitude, radiusInKm, 0.0);
        BingTile topTile = BingTileUtils.latitudeLongitudeToTile(topLatitude, longitude, zoomLevel);
        double bottomLatitude = BingTileFunctions.addDistanceToLatitude(latitude, radiusInKm, 180.0);
        BingTile bottomTile = BingTileUtils.latitudeLongitudeToTile(bottomLatitude, longitude, zoomLevel);
        double leftLongitude = BingTileFunctions.addDistanceToLongitude(latitude, longitude, radiusInKm, 270.0);
        BingTile leftTile = BingTileUtils.latitudeLongitudeToTile(latitude, leftLongitude, zoomLevel);
        double rightLongitude = BingTileFunctions.addDistanceToLongitude(latitude, longitude, radiusInKm, 90.0);
        BingTile rightTile = BingTileUtils.latitudeLongitudeToTile(latitude, rightLongitude, zoomLevel);
        boolean wrapAroundX = rightTile.getX() < leftTile.getX();
        int tileCountX = wrapAroundX ? rightTile.getX() + maxTileIndex - leftTile.getX() + 2 : rightTile.getX() - leftTile.getX() + 1;
        int tileCountY = bottomTile.getY() - topTile.getY() + 1;
        int totalTileCount = tileCountX * tileCountY;
        BingTileUtils.checkCondition(totalTileCount <= 1000000, "The number of tiles covering input rectangle exceeds the limit of 1M. Number of tiles: %d. Radius: %.1f km. Zoom level: %d.", totalTileCount, radiusInKm, zoomLevel);
        BlockBuilder blockBuilder = BigintType.BIGINT.createBlockBuilder(null, totalTileCount);
        for (int i = 0; i < tileCountX; ++i) {
            x = (leftTile.getX() + i) % (maxTileIndex + 1);
            BigintType.BIGINT.writeLong(blockBuilder, BingTile.fromCoordinates(x, tileY, zoomLevel).encode());
        }
        for (int y2 = topTile.getY(); y2 <= bottomTile.getY(); ++y2) {
            if (y2 == tileY) continue;
            BigintType.BIGINT.writeLong(blockBuilder, BingTile.fromCoordinates(tileX, y2, zoomLevel).encode());
        }
        GreatCircleDistanceToPoint distanceToCenter = new GreatCircleDistanceToPoint(latitude, longitude);
        x = rightTile.getX();
        while (x != tileX) {
            include = false;
            for (y = topTile.getY(); y < tileY; ++y) {
                tile = BingTile.fromCoordinates(x, y, zoomLevel);
                if (include) {
                    BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
                    continue;
                }
                Point bottomLeftCorner = BingTileUtils.tileXYToLatitudeLongitude(tile.getX(), tile.getY() + 1, tile.getZoomLevel());
                if (!BingTileFunctions.withinDistance(distanceToCenter, radiusInKm, bottomLeftCorner)) continue;
                include = true;
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            include = false;
            for (y = bottomTile.getY(); y > tileY; --y) {
                tile = BingTile.fromCoordinates(x, y, zoomLevel);
                if (include) {
                    BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
                    continue;
                }
                Point topLeftCorner = BingTileUtils.tileXYToLatitudeLongitude(tile.getX(), tile.getY(), tile.getZoomLevel());
                if (!BingTileFunctions.withinDistance(distanceToCenter, radiusInKm, topLeftCorner)) continue;
                include = true;
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            x = x == 0 ? maxTileIndex : x - 1;
        }
        x = leftTile.getX();
        while (x != tileX) {
            include = false;
            for (y = topTile.getY(); y < tileY; ++y) {
                tile = BingTile.fromCoordinates(x, y, zoomLevel);
                if (include) {
                    BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
                    continue;
                }
                Point bottomRightCorner = BingTileUtils.tileXYToLatitudeLongitude(tile.getX() + 1, tile.getY() + 1, tile.getZoomLevel());
                if (!BingTileFunctions.withinDistance(distanceToCenter, radiusInKm, bottomRightCorner)) continue;
                include = true;
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            include = false;
            for (y = bottomTile.getY(); y > tileY; --y) {
                tile = BingTile.fromCoordinates(x, y, zoomLevel);
                if (include) {
                    BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
                    continue;
                }
                Point topRightCorner = BingTileUtils.tileXYToLatitudeLongitude(tile.getX() + 1, tile.getY(), tile.getZoomLevel());
                if (!BingTileFunctions.withinDistance(distanceToCenter, radiusInKm, topRightCorner)) continue;
                include = true;
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            x = (x + 1) % (maxTileIndex + 1);
        }
        return blockBuilder.build();
    }

    @Description(value="Given a Bing tile, returns the polygon representation of the tile")
    @ScalarFunction(value="bing_tile_polygon")
    @SqlType(value="Geometry")
    public static Slice bingTilePolygon(@SqlType(value="BingTile") long input) {
        BingTile tile = BingTile.decode(input);
        return EsriGeometrySerde.serialize((Envelope)BingTileUtils.tileToEnvelope(tile));
    }

    @Description(value="Return the parent for a Bing tile")
    @ScalarFunction(value="bing_tile_parent")
    @SqlType(value="BingTile")
    public static long bingTileParent(@SqlType(value="BingTile") long input) {
        BingTile tile = BingTile.decode(input);
        try {
            return tile.findParent().encode();
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, e.getMessage(), (Throwable)e);
        }
    }

    @Description(value="Return the parent for the given zoom level for a Bing tile")
    @ScalarFunction(value="bing_tile_parent")
    @SqlType(value="BingTile")
    public static long bingTileParent(@SqlType(value="BingTile") long input, @SqlType(value="integer") long newZoom) {
        BingTile tile = BingTile.decode(input);
        try {
            return tile.findParent(Math.toIntExact(newZoom)).encode();
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, e.getMessage(), (Throwable)e);
        }
    }

    @Description(value="Return the children for a Bing tile")
    @ScalarFunction(value="bing_tile_children")
    @SqlType(value="array(BingTile)")
    public static Block bingTileChildren(@SqlType(value="BingTile") long input) {
        BingTile tile = BingTile.decode(input);
        try {
            List<BingTile> children = tile.findChildren();
            BlockBuilder blockBuilder = BigintType.BIGINT.createBlockBuilder(null, children.size());
            children.stream().forEach(child -> BigintType.BIGINT.writeLong(blockBuilder, child.encode()));
            return blockBuilder.build();
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, e.getMessage(), (Throwable)e);
        }
    }

    @Description(value="Return the children for the given zoom level for a Bing tile")
    @ScalarFunction(value="bing_tile_children")
    @SqlType(value="array(BingTile)")
    public static Block bingTileChildren(@SqlType(value="BingTile") long input, @SqlType(value="integer") long newZoom) {
        BingTile tile = BingTile.decode(input);
        try {
            List<BingTile> children = tile.findChildren(Math.toIntExact(newZoom));
            BlockBuilder blockBuilder = BigintType.BIGINT.createBlockBuilder(null, children.size());
            children.stream().forEach(child -> BigintType.BIGINT.writeLong(blockBuilder, child.encode()));
            return blockBuilder.build();
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, e.getMessage(), (Throwable)e);
        }
    }

    @Description(value="Given a geometry and a maximum zoom level, returns the minimum set of dissolved Bing tiles that fully covers that geometry")
    @ScalarFunction(value="geometry_to_dissolved_bing_tiles")
    @SqlType(value="array(BingTile)")
    public static Block geometryToDissolvedBingTiles(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long maxZoomLevel) {
        OGCGeometry ogcGeometry = EsriGeometrySerde.deserialize((Slice)input);
        if (ogcGeometry.isEmpty()) {
            return EMPTY_TILE_ARRAY;
        }
        List<BingTile> covering = BingTileUtils.findDissolvedTileCovering(ogcGeometry, Math.toIntExact(maxZoomLevel));
        BlockBuilder blockBuilder = BigintType.BIGINT.createBlockBuilder(null, covering.size());
        for (BingTile tile : covering) {
            BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
        }
        return blockBuilder.build();
    }

    @Description(value="Given a geometry and a zoom level, returns the minimum set of Bing tiles of that zoom level that fully covers that geometry")
    @ScalarFunction(value="geometry_to_bing_tiles")
    @SqlType(value="array(BingTile)")
    public static Block geometryToBingTiles(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long zoomLevelInput) {
        OGCGeometry ogcGeometry;
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return EMPTY_TILE_ARRAY;
        }
        int zoomLevel = Math.toIntExact(zoomLevelInput);
        GeometrySerializationType type = EsriGeometrySerde.deserializeType((Slice)input);
        List<BingTile> covering = type == GeometrySerializationType.POINT || type == GeometrySerializationType.ENVELOPE ? BingTileUtils.findMinimalTileCovering(envelope, zoomLevel) : (GeometryUtils.isPointOrRectangle((OGCGeometry)(ogcGeometry = EsriGeometrySerde.deserialize((Slice)input)), (Envelope)envelope) ? BingTileUtils.findMinimalTileCovering(envelope, zoomLevel) : BingTileUtils.findMinimalTileCovering(ogcGeometry, zoomLevel));
        BlockBuilder blockBuilder = BigintType.BIGINT.createBlockBuilder(null, covering.size());
        for (BingTile tile : covering) {
            BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
        }
        return blockBuilder.build();
    }

    private static double addDistanceToLongitude(@SqlType(value="double") double latitude, @SqlType(value="double") double longitude, @SqlType(value="double") double radiusInKm, @SqlType(value="double") double bearing) {
        double latitudeInRadians = Math.toRadians(latitude);
        double longitudeInRadians = Math.toRadians(longitude);
        double bearingInRadians = Math.toRadians(bearing);
        double radiusRatio = radiusInKm / 6371.01;
        double newLongitude = Math.toDegrees(longitudeInRadians + Math.atan2(Math.sin(bearingInRadians) * Math.sin(radiusRatio) * Math.cos(latitudeInRadians), Math.cos(radiusRatio) - Math.sin(latitudeInRadians) * Math.sin(latitudeInRadians)));
        if (newLongitude > 180.0) {
            return -180.0 + (newLongitude - 180.0);
        }
        if (newLongitude < -180.0) {
            return 180.0 + (newLongitude - -180.0);
        }
        return newLongitude;
    }

    private static double addDistanceToLatitude(@SqlType(value="double") double latitude, @SqlType(value="double") double radiusInKm, @SqlType(value="double") double bearing) {
        double latitudeInRadians = Math.toRadians(latitude);
        double bearingInRadians = Math.toRadians(bearing);
        double radiusRatio = radiusInKm / 6371.01;
        double newLatitude = Math.toDegrees(Math.asin(Math.sin(latitudeInRadians) * Math.cos(radiusRatio) + Math.cos(latitudeInRadians) * Math.sin(radiusRatio) * Math.cos(bearingInRadians)));
        if (newLatitude > 85.05112878) {
            return 85.05112878;
        }
        if (newLatitude < -85.05112878) {
            return -85.05112878;
        }
        return newLatitude;
    }

    private static BingTile[] getTilesInBetween(BingTile leftUpperTile, BingTile rightLowerTile, int zoomLevel) {
        Preconditions.checkArgument((leftUpperTile.getZoomLevel() == rightLowerTile.getZoomLevel() ? 1 : 0) != 0);
        Preconditions.checkArgument((leftUpperTile.getZoomLevel() > zoomLevel ? 1 : 0) != 0);
        int divisor = 1 << leftUpperTile.getZoomLevel() - zoomLevel;
        int minX = (int)Math.floor(leftUpperTile.getX() / divisor);
        int maxX = (int)Math.floor(rightLowerTile.getX() / divisor);
        int minY = (int)Math.floor(leftUpperTile.getY() / divisor);
        int maxY = (int)Math.floor(rightLowerTile.getY() / divisor);
        BingTile[] tiles = new BingTile[(maxX - minX + 1) * (maxY - minY + 1)];
        int index = 0;
        for (int x = minX; x <= maxX; ++x) {
            for (int y = minY; y <= maxY; ++y) {
                tiles[index] = BingTile.fromCoordinates(x, y, 10);
                ++index;
            }
        }
        return tiles;
    }

    private static void appendIntersectingSubtiles(OGCGeometry ogcGeometry, int zoomLevel, BingTile tile, BlockBuilder blockBuilder) {
        int tileZoomLevel = tile.getZoomLevel();
        Preconditions.checkArgument((tileZoomLevel <= zoomLevel ? 1 : 0) != 0);
        Envelope tileEnvelope = BingTileUtils.tileToEnvelope(tile);
        if (tileZoomLevel == zoomLevel) {
            if (!GeometryUtils.disjoint((Envelope)tileEnvelope, (OGCGeometry)ogcGeometry)) {
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            return;
        }
        if (GeometryUtils.contains((OGCGeometry)ogcGeometry, (Envelope)tileEnvelope)) {
            int subTileCount = 1 << zoomLevel - tileZoomLevel;
            int minX = subTileCount * tile.getX();
            int minY = subTileCount * tile.getY();
            for (int x = minX; x < minX + subTileCount; ++x) {
                for (int y = minY; y < minY + subTileCount; ++y) {
                    BigintType.BIGINT.writeLong(blockBuilder, BingTile.fromCoordinates(x, y, zoomLevel).encode());
                }
            }
            return;
        }
        if (GeometryUtils.disjoint((Envelope)tileEnvelope, (OGCGeometry)ogcGeometry)) {
            return;
        }
        int minX = 2 * tile.getX();
        int minY = 2 * tile.getY();
        int nextZoomLevel = tileZoomLevel + 1;
        Verify.verify((nextZoomLevel <= 23 ? 1 : 0) != 0);
        for (int x = minX; x < minX + 2; ++x) {
            for (int y = minY; y < minY + 2; ++y) {
                BingTileFunctions.appendIntersectingSubtiles(ogcGeometry, zoomLevel, BingTile.fromCoordinates(x, y, nextZoomLevel), blockBuilder);
            }
        }
    }

    private static boolean withinDistance(GreatCircleDistanceToPoint distanceFunction, double maxDistance, Point point) {
        return distanceFunction.distance(point.getY(), point.getX()) <= maxDistance;
    }

    private static final class GreatCircleDistanceToPoint {
        private double sinLatitude;
        private double cosLatitude;
        private double radianLongitude;

        private GreatCircleDistanceToPoint(double latitude, double longitude) {
            double radianLatitude = Math.toRadians(latitude);
            this.sinLatitude = Math.sin(radianLatitude);
            this.cosLatitude = Math.cos(radianLatitude);
            this.radianLongitude = Math.toRadians(longitude);
        }

        public double distance(double latitude2, double longitude2) {
            double radianLatitude2 = Math.toRadians(latitude2);
            double sin2 = Math.sin(radianLatitude2);
            double cos2 = Math.cos(radianLatitude2);
            double deltaLongitude = this.radianLongitude - Math.toRadians(longitude2);
            double cosDeltaLongitude = Math.cos(deltaLongitude);
            double t1 = cos2 * Math.sin(deltaLongitude);
            double t2 = this.cosLatitude * sin2 - this.sinLatitude * cos2 * cosDeltaLongitude;
            double t3 = this.sinLatitude * sin2 + this.cosLatitude * cos2 * cosDeltaLongitude;
            return Math.atan2(Math.sqrt(t1 * t1 + t2 * t2), t3) * 6371.01;
        }
    }

    @Description(value="Given a Bing tile, returns XY coordinates of the tile")
    @ScalarFunction(value="bing_tile_coordinates")
    public static final class BingTileCoordinatesFunction {
        private static final RowType BING_TILE_COORDINATES_ROW_TYPE = RowType.anonymous((List)ImmutableList.of((Object)IntegerType.INTEGER, (Object)IntegerType.INTEGER));
        private final PageBuilder pageBuilder = new PageBuilder((List)ImmutableList.of((Object)BING_TILE_COORDINATES_ROW_TYPE));

        @SqlType(value="row(x integer,y integer)")
        public Block bingTileCoordinates(@SqlType(value="BingTile") long input) {
            if (this.pageBuilder.isFull()) {
                this.pageBuilder.reset();
            }
            BlockBuilder blockBuilder = this.pageBuilder.getBlockBuilder(0);
            BingTile tile = BingTile.decode(input);
            BlockBuilder tileBlockBuilder = blockBuilder.beginBlockEntry();
            IntegerType.INTEGER.writeLong(tileBlockBuilder, (long)tile.getX());
            IntegerType.INTEGER.writeLong(tileBlockBuilder, (long)tile.getY());
            blockBuilder.closeEntry();
            this.pageBuilder.declarePosition();
            return BING_TILE_COORDINATES_ROW_TYPE.getObject((Block)blockBuilder, blockBuilder.getPositionCount() - 1);
        }
    }
}

