/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.parquet;

import com.facebook.presto.parquet.ParquetCorruptionException;
import com.google.common.base.Preconditions;
import io.airlift.compress.Decompressor;
import io.airlift.compress.lz4.Lz4Decompressor;
import io.airlift.compress.lzo.LzoDecompressor;
import io.airlift.compress.snappy.SnappyDecompressor;
import io.airlift.compress.zstd.ZstdDecompressor;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;
import java.util.zip.GZIPInputStream;
import org.apache.parquet.hadoop.metadata.CompressionCodecName;
import sun.misc.Unsafe;

public final class ParquetCompressionUtils {
    private static final int GZIP_BUFFER_SIZE = 8192;

    private ParquetCompressionUtils() {
    }

    public static Slice decompress(CompressionCodecName codec, Slice input, int uncompressedSize) throws IOException {
        Objects.requireNonNull(input, "input is null");
        if (input.length() == 0) {
            return Slices.EMPTY_SLICE;
        }
        switch (codec) {
            case GZIP: {
                return ParquetCompressionUtils.decompressGzip(input, uncompressedSize);
            }
            case SNAPPY: {
                return ParquetCompressionUtils.decompressSnappy(input, uncompressedSize);
            }
            case UNCOMPRESSED: {
                return input;
            }
            case LZO: {
                return ParquetCompressionUtils.decompressLZO(input, uncompressedSize);
            }
            case LZ4: {
                return ParquetCompressionUtils.decompressLz4(input, uncompressedSize);
            }
            case ZSTD: {
                return ParquetCompressionUtils.decompressZstd(input, uncompressedSize);
            }
        }
        throw new ParquetCorruptionException("Codec not supported in Parquet: " + codec);
    }

    private static Slice decompressSnappy(Slice input, int uncompressedSize) {
        byte[] buffer = new byte[uncompressedSize];
        ParquetCompressionUtils.decompress((Decompressor)new SnappyDecompressor(), input, 0, input.length(), buffer, 0);
        return Slices.wrappedBuffer((byte[])buffer);
    }

    private static Slice decompressZstd(Slice input, int uncompressedSize) {
        byte[] buffer = new byte[uncompressedSize];
        ParquetCompressionUtils.decompress((Decompressor)new ZstdDecompressor(), input, 0, input.length(), buffer, 0);
        return Slices.wrappedBuffer((byte[])buffer);
    }

    private static Slice decompressGzip(Slice input, int uncompressedSize) throws IOException {
        if (uncompressedSize == 0) {
            return Slices.EMPTY_SLICE;
        }
        byte[] buffer = new byte[uncompressedSize];
        int bytesRead = 0;
        boolean eos = false;
        try (GZIPInputStream gzipInputStream = new GZIPInputStream((InputStream)input.getInput(), Math.min(8192, input.length()));){
            int n;
            do {
                if ((n = gzipInputStream.read(buffer, bytesRead, buffer.length - bytesRead)) >= 0) continue;
                eos = true;
                break;
            } while ((bytesRead += n) < buffer.length);
            Preconditions.checkArgument((eos || gzipInputStream.read() == -1 ? 1 : 0) != 0, (String)"Invalid uncompressedSize for GZIP input. Actual size exceeds %s bytes", (int)uncompressedSize);
        }
        if (bytesRead != uncompressedSize) {
            throw new IllegalArgumentException(String.format("Invalid uncompressedSize for GZIP input. Expected %s, actual: %s", uncompressedSize, bytesRead));
        }
        return Slices.wrappedBuffer((byte[])buffer, (int)0, (int)bytesRead);
    }

    private static Slice decompressLz4(Slice input, int uncompressedSize) {
        byte[] buffer = new byte[uncompressedSize];
        ParquetCompressionUtils.decompress((Decompressor)new Lz4Decompressor(), input, 0, input.length(), buffer, 0);
        return Slices.wrappedBuffer((byte[])buffer);
    }

    private static Slice decompressLZO(Slice input, int uncompressedSize) {
        LzoDecompressor lzoDecompressor = new LzoDecompressor();
        long totalDecompressedCount = 0L;
        byte[] output = new byte[uncompressedSize + 8];
        int outputOffset = 0;
        int inputOffset = 0;
        int cumulativeUncompressedBlockLength = 0;
        while (totalDecompressedCount < (long)uncompressedSize) {
            if (totalDecompressedCount == (long)cumulativeUncompressedBlockLength) {
                cumulativeUncompressedBlockLength += Integer.reverseBytes(input.getInt(inputOffset));
                inputOffset += 4;
            }
            int compressedChunkLength = Integer.reverseBytes(input.getInt(inputOffset));
            int decompressionSize = ParquetCompressionUtils.decompress((Decompressor)lzoDecompressor, input, inputOffset += 4, compressedChunkLength, output, outputOffset);
            totalDecompressedCount += (long)decompressionSize;
            outputOffset += decompressionSize;
            inputOffset += compressedChunkLength;
        }
        Preconditions.checkArgument((outputOffset == uncompressedSize ? 1 : 0) != 0);
        return Slices.wrappedBuffer((byte[])output, (int)0, (int)uncompressedSize);
    }

    private static int decompress(Decompressor decompressor, Slice input, int inputOffset, int inputLength, byte[] output, int outputOffset) {
        byte[] byteArray = (byte[])input.getBase();
        int byteArrayOffset = inputOffset + (int)(input.getAddress() - (long)Unsafe.ARRAY_BYTE_BASE_OFFSET);
        int size = decompressor.decompress(byteArray, byteArrayOffset, inputLength, output, outputOffset, output.length - outputOffset);
        return size;
    }
}

