/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.parquet.writer;

import com.facebook.presto.common.Page;
import com.facebook.presto.common.type.Type;
import com.facebook.presto.parquet.writer.ColumnChunk;
import com.facebook.presto.parquet.writer.ColumnWriter;
import com.facebook.presto.parquet.writer.MessageTypeConverter;
import com.facebook.presto.parquet.writer.ParquetDataOutput;
import com.facebook.presto.parquet.writer.ParquetWriterOptions;
import com.facebook.presto.parquet.writer.ParquetWriters;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.OutputStreamSliceOutput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.airlift.units.DataSize;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.parquet.column.ParquetProperties;
import org.apache.parquet.format.ColumnMetaData;
import org.apache.parquet.format.FileMetaData;
import org.apache.parquet.format.RowGroup;
import org.apache.parquet.format.Util;
import org.apache.parquet.hadoop.metadata.CompressionCodecName;
import org.apache.parquet.schema.MessageType;
import org.openjdk.jol.info.ClassLayout;

public class ParquetWriter
implements Closeable {
    private static final int INSTANCE_SIZE = ClassLayout.parseClass(ParquetWriter.class).instanceSize();
    private static final int CHUNK_MAX_BYTES = Math.toIntExact(DataSize.valueOf((String)"128MB").toBytes());
    private static final int DEFAULT_ROW_GROUP_MAX_ROW_COUNT = 10000;
    private final List<ColumnWriter> columnWriters;
    private final OutputStreamSliceOutput outputStream;
    private final List<Type> types;
    private final ParquetWriterOptions writerOption;
    private final List<String> names;
    private final MessageType messageType;
    private final int chunkMaxLogicalBytes;
    private ImmutableList.Builder<RowGroup> rowGroupBuilder = ImmutableList.builder();
    private int rows;
    private boolean closed;
    private boolean writeHeader;
    public static final Slice MAGIC = Slices.wrappedBuffer((byte[])"PAR1".getBytes(StandardCharsets.US_ASCII));

    public ParquetWriter(OutputStream outputStream, MessageType messageType, Map<List<String>, Type> primitiveTypes, List<String> columnNames, List<Type> types, ParquetWriterOptions writerOption, String compressionCodecClass) {
        this.outputStream = new OutputStreamSliceOutput(Objects.requireNonNull(outputStream, "outputstream is null"));
        this.names = ImmutableList.copyOf((Collection)Objects.requireNonNull(columnNames, "columnNames is null"));
        this.types = ImmutableList.copyOf((Collection)Objects.requireNonNull(types, "types is null"));
        this.writerOption = Objects.requireNonNull(writerOption, "writerOption is null");
        Preconditions.checkArgument((types.size() == columnNames.size() ? 1 : 0) != 0, (String)"type size %s is not equal to name size %s", (int)types.size(), (int)columnNames.size());
        this.messageType = Objects.requireNonNull(messageType, "messageType is null");
        ParquetProperties parquetProperties = ParquetProperties.builder().withWriterVersion(ParquetProperties.WriterVersion.PARQUET_2_0).withPageSize(writerOption.getMaxPageSize()).build();
        CompressionCodecName compressionCodecName = this.getCompressionCodecName(compressionCodecClass);
        this.columnWriters = ParquetWriters.getColumnWriters(messageType, primitiveTypes, parquetProperties, compressionCodecName);
        this.chunkMaxLogicalBytes = Math.max(1, CHUNK_MAX_BYTES / 2);
    }

    public long getWrittenBytes() {
        return this.outputStream.size();
    }

    public long getBufferedBytes() {
        return this.columnWriters.stream().mapToLong(ColumnWriter::getBufferedBytes).sum();
    }

    public long getRetainedBytes() {
        return (long)INSTANCE_SIZE + this.outputStream.getRetainedSize() + this.columnWriters.stream().mapToLong(ColumnWriter::getRetainedBytes).sum();
    }

    public void write(Page page) throws IOException {
        Objects.requireNonNull(page, "page is null");
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"writer is closed");
        if (page.getPositionCount() == 0) {
            return;
        }
        Preconditions.checkArgument((page.getChannelCount() == this.columnWriters.size() ? 1 : 0) != 0);
        while (page != null) {
            int chunkRows = Math.min(page.getPositionCount(), 10000);
            Page chunk = page.getRegion(0, chunkRows);
            while (chunkRows > 1 && chunk.getLogicalSizeInBytes() > (long)this.chunkMaxLogicalBytes) {
                chunk = chunk.getRegion(0, chunkRows /= 2);
            }
            page = chunkRows < page.getPositionCount() ? page.getRegion(chunkRows, page.getPositionCount() - chunkRows) : null;
            this.writeChunk(chunk);
        }
    }

    private void writeChunk(Page page) throws IOException {
        long bufferedBytes = 0L;
        for (int channel = 0; channel < page.getChannelCount(); ++channel) {
            ColumnWriter writer = this.columnWriters.get(channel);
            writer.writeBlock(new ColumnChunk(page.getBlock(channel)));
            bufferedBytes += writer.getBufferedBytes();
        }
        this.rows += page.getPositionCount();
        if (bufferedBytes >= (long)this.writerOption.getMaxRowGroupSize()) {
            this.columnWriters.forEach(ColumnWriter::close);
            this.flush();
            this.columnWriters.forEach(ColumnWriter::reset);
            this.rows = 0;
        }
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        this.columnWriters.forEach(ColumnWriter::close);
        this.flush();
        this.writeFooter();
        this.outputStream.close();
    }

    private void flush() throws IOException {
        if (!this.writeHeader) {
            ParquetDataOutput.createDataOutput(MAGIC).writeData((SliceOutput)this.outputStream);
            this.writeHeader = true;
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        for (ColumnWriter columnWriter : this.columnWriters) {
            columnWriter.getBuffer().forEach(arg_0 -> ((ImmutableList.Builder)builder).add(arg_0));
        }
        ImmutableList bufferDataList = builder.build();
        long stripeStartOffset = this.outputStream.longSize();
        List metadatas = (List)bufferDataList.stream().map(ColumnWriter.BufferData::getMetaData).collect(ImmutableList.toImmutableList());
        this.updateRowGroups(this.updateColumnMetadataOffset(metadatas, stripeStartOffset));
        bufferDataList.stream().map(ColumnWriter.BufferData::getData).flatMap(Collection::stream).forEach(data -> data.writeData((SliceOutput)this.outputStream));
    }

    private void writeFooter() throws IOException {
        Preconditions.checkState((boolean)this.closed);
        Slice footer = ParquetWriter.getFooter((List<RowGroup>)this.rowGroupBuilder.build(), this.messageType);
        ParquetDataOutput.createDataOutput(footer).writeData((SliceOutput)this.outputStream);
        Slice footerSize = Slices.allocate((int)4);
        footerSize.setInt(0, footer.length());
        ParquetDataOutput.createDataOutput(footerSize).writeData((SliceOutput)this.outputStream);
        ParquetDataOutput.createDataOutput(MAGIC).writeData((SliceOutput)this.outputStream);
    }

    static Slice getFooter(List<RowGroup> rowGroups, MessageType messageType) throws IOException {
        FileMetaData fileMetaData = new FileMetaData();
        fileMetaData.setVersion(1);
        fileMetaData.setSchema(MessageTypeConverter.toParquetSchema(messageType));
        long totalRows = rowGroups.stream().mapToLong(RowGroup::getNum_rows).sum();
        fileMetaData.setNum_rows(totalRows);
        fileMetaData.setRow_groups((List)ImmutableList.copyOf(rowGroups));
        DynamicSliceOutput dynamicSliceOutput = new DynamicSliceOutput(40);
        Util.writeFileMetaData((FileMetaData)fileMetaData, (OutputStream)dynamicSliceOutput);
        return dynamicSliceOutput.slice();
    }

    private void updateRowGroups(List<ColumnMetaData> columnMetaData) {
        long totalBytes = columnMetaData.stream().mapToLong(ColumnMetaData::getTotal_compressed_size).sum();
        ImmutableList columnChunks = (ImmutableList)columnMetaData.stream().map(ParquetWriter::toColumnChunk).collect(ImmutableList.toImmutableList());
        this.rowGroupBuilder.add((Object)new RowGroup((List)columnChunks, totalBytes, (long)this.rows));
    }

    private static org.apache.parquet.format.ColumnChunk toColumnChunk(ColumnMetaData metaData) {
        org.apache.parquet.format.ColumnChunk columnChunk = new org.apache.parquet.format.ColumnChunk(0L);
        columnChunk.setMeta_data(metaData);
        return columnChunk;
    }

    private List<ColumnMetaData> updateColumnMetadataOffset(List<ColumnMetaData> columns, long offset) {
        ImmutableList.Builder builder = ImmutableList.builder();
        long currentOffset = offset;
        for (ColumnMetaData column : columns) {
            ColumnMetaData columnMetaData = new ColumnMetaData(column.type, column.encodings, column.path_in_schema, column.codec, column.num_values, column.total_uncompressed_size, column.total_compressed_size, currentOffset);
            columnMetaData.setStatistics(column.getStatistics());
            builder.add((Object)columnMetaData);
            currentOffset += column.getTotal_compressed_size();
        }
        return builder.build();
    }

    private CompressionCodecName getCompressionCodecName(String compressionCodecClass) {
        if (compressionCodecClass == null) {
            return CompressionCodecName.UNCOMPRESSED;
        }
        if (compressionCodecClass.equals("parquet.hadoop.codec.SnappyCodec") || compressionCodecClass.equals("org.apache.parquet.hadoop.codec.SnappyCodec")) {
            return CompressionCodecName.SNAPPY;
        }
        if (compressionCodecClass.equals("org.apache.hadoop.io.compress.GzipCodec")) {
            return CompressionCodecName.GZIP;
        }
        if (compressionCodecClass.equals("com.hadoop.compression.lzo.LzoCodec")) {
            return CompressionCodecName.LZO;
        }
        if (compressionCodecClass.equals("org.apache.hadoop.io.compress.BrotliCodec")) {
            return CompressionCodecName.BROTLI;
        }
        if (compressionCodecClass.equals("org.apache.hadoop.io.compress.Lz4Codec")) {
            return CompressionCodecName.LZ4;
        }
        if (compressionCodecClass.equals("org.apache.hadoop.io.compress.ZStandardCodec")) {
            return CompressionCodecName.ZSTD;
        }
        throw new IllegalArgumentException("Invalid compressionCodec: " + compressionCodecClass);
    }
}

