/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.$internal.org.apache.pinot.core.segment.index.converter;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.EnumSet;
import java.util.Set;
import org.apache.pinot.$internal.com.google.common.annotations.VisibleForTesting;
import org.apache.pinot.$internal.com.google.common.base.Preconditions;
import org.apache.pinot.$internal.org.apache.commons.configuration.ConfigurationException;
import org.apache.pinot.$internal.org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.pinot.$internal.org.apache.commons.io.FileUtils;
import org.apache.pinot.$internal.org.apache.commons.io.IOUtils;
import org.apache.pinot.$internal.org.apache.pinot.core.indexsegment.generator.SegmentVersion;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.index.SegmentMetadataImpl;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.index.converter.SegmentFormatConverter;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.memory.PinotDataBuffer;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.store.ColumnIndexType;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.store.SegmentDirectory;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.store.SegmentDirectoryPaths;
import org.apache.pinot.common.segment.ReadMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentV1V2ToV3FormatConverter
implements SegmentFormatConverter {
    private static Logger LOGGER = LoggerFactory.getLogger(SegmentV1V2ToV3FormatConverter.class);
    private static final String V3_TEMP_DIR_SUFFIX = ".v3.tmp";

    @Override
    public void convert(File v2SegmentDirectory) throws Exception {
        Preconditions.checkNotNull(v2SegmentDirectory, "Segment directory should not be null");
        Preconditions.checkState(v2SegmentDirectory.exists() && v2SegmentDirectory.isDirectory(), "Segment directory: " + v2SegmentDirectory.toString() + " must exist and should be a directory");
        LOGGER.info("Converting segment: {} to v3 format", (Object)v2SegmentDirectory);
        SegmentMetadataImpl v2Metadata = new SegmentMetadataImpl(v2SegmentDirectory);
        SegmentVersion oldVersion = SegmentVersion.valueOf(v2Metadata.getVersion());
        Preconditions.checkState(oldVersion != SegmentVersion.v3, "Segment {} is already in v3 format but at wrong path", (Object)v2Metadata.getName());
        Preconditions.checkArgument(oldVersion == SegmentVersion.v1 || oldVersion == SegmentVersion.v2, "Can not convert segment version: {} at path: {} ", (Object)oldVersion, (Object)v2SegmentDirectory);
        this.deleteStaleConversionDirectories(v2SegmentDirectory);
        File v3TempDirectory = this.v3ConversionTempDirectory(v2SegmentDirectory);
        this.setDirectoryPermissions(v3TempDirectory);
        this.createMetadataFile(v2SegmentDirectory, v3TempDirectory);
        this.copyCreationMetadataIfExists(v2SegmentDirectory, v3TempDirectory);
        this.copyIndexData(v2SegmentDirectory, v2Metadata, v3TempDirectory);
        File newLocation = SegmentDirectoryPaths.segmentDirectoryFor(v2SegmentDirectory, SegmentVersion.v3);
        LOGGER.info("v3 segment location for segment: {} is {}", (Object)v2Metadata.getName(), (Object)newLocation);
        v3TempDirectory.renameTo(newLocation);
        this.deleteV2Files(v2SegmentDirectory);
    }

    private void deleteV2Files(File v2SegmentDirectory) {
        LOGGER.info("Deleting files in v1 segment directory: {}", (Object)v2SegmentDirectory);
        File[] files = v2SegmentDirectory.listFiles();
        if (files == null) {
            LOGGER.error("v1 segment directory: {}  returned null list of files", (Object)v2SegmentDirectory);
            return;
        }
        for (File file : files) {
            if (!file.isFile() || !file.exists()) continue;
            FileUtils.deleteQuietly(file);
        }
    }

    @VisibleForTesting
    public File v3ConversionTempDirectory(File v2SegmentDirectory) throws IOException {
        File v3TempDirectory = Files.createTempDirectory(v2SegmentDirectory.toPath(), v2SegmentDirectory.getName() + V3_TEMP_DIR_SUFFIX, new FileAttribute[0]).toFile();
        return v3TempDirectory;
    }

    private void setDirectoryPermissions(File v3Directory) throws IOException {
        EnumSet<PosixFilePermission[]> permissions = EnumSet.of(PosixFilePermission.OWNER_READ, new PosixFilePermission[]{PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_EXECUTE});
        try {
            Files.setPosixFilePermissions(v3Directory.toPath(), permissions);
        }
        catch (UnsupportedOperationException ex) {
            LOGGER.error("unsupported non-posix filesystem permissions setting");
        }
    }

    private void copyIndexData(File v2Directory, SegmentMetadataImpl v2Metadata, File v3Directory) throws Exception {
        SegmentMetadataImpl v3Metadata = new SegmentMetadataImpl(v3Directory);
        try (SegmentDirectory v2Segment = SegmentDirectory.createFromLocalFS(v2Directory, v2Metadata, ReadMode.mmap);
             SegmentDirectory v3Segment = SegmentDirectory.createFromLocalFS(v3Directory, v3Metadata, ReadMode.mmap);){
            Set<String> allColumns = v2Metadata.getAllColumns();
            try (SegmentDirectory.Reader v2DataReader = v2Segment.createReader();
                 SegmentDirectory.Writer v3DataWriter = v3Segment.createWriter();){
                for (String column : allColumns) {
                    LOGGER.debug("Converting segment: {} , column: {}", (Object)v2Directory, (Object)column);
                    if (v2Metadata.hasDictionary(column)) {
                        this.copyDictionary(v2DataReader, v3DataWriter, column);
                    }
                    this.copyForwardIndex(v2DataReader, v3DataWriter, column);
                }
                for (String column : allColumns) {
                    this.copyExistingInvertedIndex(v2DataReader, v3DataWriter, column);
                }
                this.copyStarTree(v2DataReader, v3DataWriter);
                v3DataWriter.saveAndClose();
            }
        }
        this.copyStarTreeV2(v2Directory, v3Directory);
    }

    private void copyStarTreeV2(File src, File dest) throws IOException {
        File indexFile = new File(src, "star_tree_index");
        if (indexFile.exists()) {
            FileUtils.copyFile(indexFile, new File(dest, "star_tree_index"));
            FileUtils.copyFile(new File(src, "star_tree_index_map"), new File(dest, "star_tree_index_map"));
        }
    }

    private void copyStarTree(SegmentDirectory.Reader v2DataReader, SegmentDirectory.Writer v3DataWriter) throws IOException {
        if (!v2DataReader.hasStarTree()) {
            return;
        }
        InputStream v2StarTreeStream = v2DataReader.getStarTreeStream();
        OutputStream v3StarTreeStream = v3DataWriter.starTreeOutputStream();
        IOUtils.copy(v2StarTreeStream, v3StarTreeStream);
    }

    private void copyDictionary(SegmentDirectory.Reader reader, SegmentDirectory.Writer writer, String column) throws IOException {
        this.readCopyBuffers(reader, writer, column, ColumnIndexType.DICTIONARY);
    }

    private void copyForwardIndex(SegmentDirectory.Reader reader, SegmentDirectory.Writer writer, String column) throws IOException {
        this.readCopyBuffers(reader, writer, column, ColumnIndexType.FORWARD_INDEX);
    }

    private void copyExistingInvertedIndex(SegmentDirectory.Reader reader, SegmentDirectory.Writer writer, String column) throws IOException {
        if (reader.hasIndexFor(column, ColumnIndexType.INVERTED_INDEX)) {
            this.readCopyBuffers(reader, writer, column, ColumnIndexType.INVERTED_INDEX);
        }
    }

    private void readCopyBuffers(SegmentDirectory.Reader reader, SegmentDirectory.Writer writer, String column, ColumnIndexType indexType) throws IOException {
        PinotDataBuffer oldBuffer = reader.getIndexFor(column, indexType);
        PinotDataBuffer newDictBuffer = writer.newIndexFor(column, indexType, oldBuffer.size());
        oldBuffer.copyTo(0L, newDictBuffer, 0L, oldBuffer.size());
    }

    private void createMetadataFile(File currentDir, File v3Dir) throws ConfigurationException {
        File v2MetadataFile = new File(currentDir, "metadata.properties");
        File v3MetadataFile = new File(v3Dir, "metadata.properties");
        PropertiesConfiguration properties = new PropertiesConfiguration(v2MetadataFile);
        properties.setProperty("segment.index.version", SegmentVersion.v3.toString());
        properties.save(v3MetadataFile);
    }

    private void copyCreationMetadataIfExists(File currentDir, File v3Dir) throws IOException {
        File v2CreationFile = new File(currentDir, "creation.meta");
        if (v2CreationFile.exists()) {
            File v3CreationFile = new File(v3Dir, "creation.meta");
            Files.copy(v2CreationFile.toPath(), v3CreationFile.toPath(), new CopyOption[0]);
        }
    }

    private void deleteStaleConversionDirectories(File segmentDirectory) {
        File[] files;
        final String prefix = segmentDirectory.getName() + V3_TEMP_DIR_SUFFIX;
        for (File file : files = segmentDirectory.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.startsWith(prefix);
            }
        })) {
            LOGGER.info("Deleting stale v3 directory: {}", (Object)file);
            FileUtils.deleteQuietly(file);
        }
    }

    public static void main(String[] args) throws Exception {
        if (args.length < 1) {
            System.err.println("Usage: $0 <table directory with segments>");
            System.exit(1);
        }
        File tableDirectory = new File(args[0]);
        Preconditions.checkState(tableDirectory.exists(), "Directory: {} does not exist", (Object)tableDirectory);
        Preconditions.checkState(tableDirectory.isDirectory(), "Path: {} is not a directory", (Object)tableDirectory);
        File[] files = tableDirectory.listFiles();
        SegmentV1V2ToV3FormatConverter converter = new SegmentV1V2ToV3FormatConverter();
        for (File file : files) {
            if (!file.isDirectory()) {
                System.out.println("Path: " + file + " is not a directory. Skipping...");
                continue;
            }
            long startTimeNano = System.nanoTime();
            converter.convert(file);
            long endTimeNano = System.nanoTime();
            long latency = (endTimeNano - startTimeNano) / 1000000L;
            System.out.println("Converting segment: " + file + " took " + latency + " milliseconds");
        }
    }
}

