/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.$internal.org.apache.pinot.core.segment.store;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import org.apache.pinot.$internal.com.google.common.base.Preconditions;
import org.apache.pinot.$internal.org.apache.commons.configuration.ConfigurationException;
import org.apache.pinot.$internal.org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.index.SegmentMetadataImpl;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.memory.PinotDataBuffer;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.store.ColumnIndexDirectory;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.store.ColumnIndexType;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.store.IndexEntry;
import org.apache.pinot.$internal.org.apache.pinot.core.segment.store.IndexKey;
import org.apache.pinot.common.segment.ReadMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SingleFileIndexDirectory
extends ColumnIndexDirectory {
    private static Logger LOGGER = LoggerFactory.getLogger(SingleFileIndexDirectory.class);
    private static final String DEFAULT_INDEX_FILE_NAME = "columns.psf";
    private static final String INDEX_MAP_FILE = "index_map";
    private static final long MAGIC_MARKER = -2401053088876085587L;
    private static final int MAGIC_MARKER_SIZE_BYTES = 8;
    private static final String MAP_KEY_SEPARATOR = ".";
    private static final String MAP_KEY_NAME_START_OFFSET = "startOffset";
    private static final String MAP_KEY_NAME_SIZE = "size";
    private static final int MAX_ALLOCATION_SIZE = 0x7D000000;
    private File indexFile;
    private Map<IndexKey, IndexEntry> columnEntries;
    private List<PinotDataBuffer> allocBuffers;

    public SingleFileIndexDirectory(File segmentDirectory, SegmentMetadataImpl metadata, ReadMode readMode) throws IOException, ConfigurationException {
        super(segmentDirectory, metadata, readMode);
        this.indexFile = new File(segmentDirectory, DEFAULT_INDEX_FILE_NAME);
        if (!this.indexFile.exists()) {
            this.indexFile.createNewFile();
        }
        this.columnEntries = new HashMap<IndexKey, IndexEntry>(metadata.getAllColumns().size());
        this.allocBuffers = new ArrayList<PinotDataBuffer>();
        this.load();
    }

    @Override
    public PinotDataBuffer getDictionaryBufferFor(String column) throws IOException {
        return this.checkAndGetIndexBuffer(column, ColumnIndexType.DICTIONARY);
    }

    @Override
    public PinotDataBuffer getForwardIndexBufferFor(String column) throws IOException {
        return this.checkAndGetIndexBuffer(column, ColumnIndexType.FORWARD_INDEX);
    }

    @Override
    public PinotDataBuffer getInvertedIndexBufferFor(String column) throws IOException {
        return this.checkAndGetIndexBuffer(column, ColumnIndexType.INVERTED_INDEX);
    }

    @Override
    public PinotDataBuffer getBloomFilterBufferFor(String column) throws IOException {
        return this.checkAndGetIndexBuffer(column, ColumnIndexType.BLOOM_FILTER);
    }

    @Override
    public boolean hasIndexFor(String column, ColumnIndexType type) {
        IndexKey key = new IndexKey(column, type);
        return this.columnEntries.containsKey(key);
    }

    @Override
    public PinotDataBuffer newDictionaryBuffer(String column, long sizeBytes) throws IOException {
        return this.allocNewBufferInternal(column, ColumnIndexType.DICTIONARY, sizeBytes, "dictionary.create");
    }

    @Override
    public PinotDataBuffer newForwardIndexBuffer(String column, long sizeBytes) throws IOException {
        return this.allocNewBufferInternal(column, ColumnIndexType.FORWARD_INDEX, sizeBytes, "forward_index.create");
    }

    @Override
    public PinotDataBuffer newInvertedIndexBuffer(String column, long sizeBytes) throws IOException {
        return this.allocNewBufferInternal(column, ColumnIndexType.INVERTED_INDEX, sizeBytes, "inverted_index.create");
    }

    @Override
    public PinotDataBuffer newBloomFilterBuffer(String column, long sizeBytes) throws IOException {
        return this.allocNewBufferInternal(column, ColumnIndexType.BLOOM_FILTER, sizeBytes, "bloom_filter.create");
    }

    private PinotDataBuffer checkAndGetIndexBuffer(String column, ColumnIndexType type) {
        IndexKey key = new IndexKey(column, type);
        IndexEntry entry = this.columnEntries.get(key);
        if (entry == null || entry.buffer == null) {
            throw new RuntimeException("Could not find index for column: " + column + ", type: " + (Object)((Object)type) + ", segment: " + this.segmentDirectory.toString());
        }
        return entry.buffer;
    }

    private PinotDataBuffer allocNewBufferInternal(String column, ColumnIndexType indexType, long size, String context) throws IOException {
        IndexKey key = new IndexKey(column, indexType);
        this.checkKeyNotPresent(key);
        String allocContext = this.allocationContext(key) + context;
        IndexEntry entry = new IndexEntry(key);
        entry.startOffset = this.indexFile.length();
        entry.size = size + 8L;
        PinotDataBuffer appendBuffer = PinotDataBuffer.mapFile(this.indexFile, false, entry.startOffset, entry.size, ByteOrder.BIG_ENDIAN, allocContext);
        LOGGER.debug("Allotted buffer for key: {}, startOffset: {}, size: {}", new Object[]{key, entry.startOffset, entry.size});
        appendBuffer.putLong(0, -2401053088876085587L);
        this.allocBuffers.add(appendBuffer);
        entry.buffer = appendBuffer.view(8L, entry.size);
        this.columnEntries.put(key, entry);
        this.persistIndexMap(entry);
        return entry.buffer;
    }

    private void checkKeyNotPresent(IndexKey key) {
        if (this.columnEntries.containsKey(key)) {
            throw new RuntimeException("Attempt to re-create an existing index for key: " + key.toString() + ", for segmentDirectory: " + this.segmentDirectory.getAbsolutePath());
        }
    }

    private void validateMagicMarker(PinotDataBuffer buffer, int startOffset) {
        long actualMarkerValue = buffer.getLong(startOffset);
        if (actualMarkerValue != -2401053088876085587L) {
            LOGGER.error("Missing magic marker in index file: {} at position: {}", (Object)this.indexFile, (Object)startOffset);
            throw new RuntimeException("Inconsistent data read. Index data file " + this.indexFile.toString() + " is possibly corrupted");
        }
    }

    private void load() throws IOException, ConfigurationException {
        this.loadMap();
        this.mapBufferEntries();
    }

    private void loadMap() throws ConfigurationException {
        File mapFile = new File(this.segmentDirectory, INDEX_MAP_FILE);
        PropertiesConfiguration mapConfig = new PropertiesConfiguration(mapFile);
        Iterator keys = mapConfig.getKeys();
        while (keys.hasNext()) {
            String key = (String)keys.next();
            int lastSeparatorPos = key.lastIndexOf(MAP_KEY_SEPARATOR);
            Preconditions.checkState(lastSeparatorPos != -1, "Key separator not found: " + key + ", segment: " + this.segmentDirectory);
            String propertyName = key.substring(lastSeparatorPos + 1);
            int indexSeparatorPos = key.lastIndexOf(MAP_KEY_SEPARATOR, lastSeparatorPos - 1);
            Preconditions.checkState(indexSeparatorPos != -1, "Index separator not found: " + key + " , segment: " + this.segmentDirectory);
            String indexName = key.substring(indexSeparatorPos + 1, lastSeparatorPos);
            String columnName = key.substring(0, indexSeparatorPos);
            IndexKey indexKey = new IndexKey(columnName, ColumnIndexType.getValue(indexName));
            IndexEntry entry = this.columnEntries.get(indexKey);
            if (entry == null) {
                entry = new IndexEntry(indexKey);
                this.columnEntries.put(indexKey, entry);
            }
            if (propertyName.equals(MAP_KEY_NAME_START_OFFSET)) {
                entry.startOffset = mapConfig.getLong(key);
                continue;
            }
            if (propertyName.equals(MAP_KEY_NAME_SIZE)) {
                entry.size = mapConfig.getLong(key);
                continue;
            }
            throw new ConfigurationException("Invalid map file key: " + key + ", segmentDirectory: " + this.segmentDirectory.toString());
        }
        for (Map.Entry<IndexKey, IndexEntry> colIndexEntry : this.columnEntries.entrySet()) {
            IndexEntry entry = colIndexEntry.getValue();
            if (entry.size >= 0L && entry.startOffset >= 0L) continue;
            throw new ConfigurationException("Invalid map entry for key: " + colIndexEntry.getKey().toString() + ", segment: " + this.segmentDirectory.toString());
        }
    }

    private void mapBufferEntries() throws IOException {
        TreeMap<Long, IndexEntry> indexStartMap = new TreeMap<Long, IndexEntry>();
        for (Map.Entry<IndexKey, IndexEntry> columnEntry : this.columnEntries.entrySet()) {
            long startOffset = columnEntry.getValue().startOffset;
            indexStartMap.put(startOffset, columnEntry.getValue());
        }
        long runningSize = 0L;
        ArrayList<Long> offsetAccum = new ArrayList<Long>();
        for (Map.Entry offsetEntry : indexStartMap.entrySet()) {
            IndexEntry entry = (IndexEntry)offsetEntry.getValue();
            if ((runningSize += entry.size) >= 0x7D000000L) {
                this.mapAndSliceFile(indexStartMap, offsetAccum, (Long)offsetEntry.getKey());
                runningSize = entry.size;
                offsetAccum.clear();
            }
            offsetAccum.add((Long)offsetEntry.getKey());
        }
        if (offsetAccum.size() > 0) {
            this.mapAndSliceFile(indexStartMap, offsetAccum, (Long)offsetAccum.get(0) + runningSize);
        }
    }

    private void mapAndSliceFile(SortedMap<Long, IndexEntry> startOffsets, List<Long> offsetAccum, long endOffset) throws IOException {
        Preconditions.checkNotNull(startOffsets);
        Preconditions.checkNotNull(offsetAccum);
        Preconditions.checkArgument(offsetAccum.size() >= 1);
        long fromFilePos = offsetAccum.get(0);
        long size = endOffset - fromFilePos;
        String context = this.allocationContext(this.indexFile, "single_file_index.rw.." + String.valueOf(fromFilePos) + MAP_KEY_SEPARATOR + String.valueOf(size));
        PinotDataBuffer buffer = this.readMode == ReadMode.heap ? PinotDataBuffer.loadFile(this.indexFile, fromFilePos, size, ByteOrder.BIG_ENDIAN, context) : PinotDataBuffer.mapFile(this.indexFile, true, fromFilePos, size, ByteOrder.BIG_ENDIAN, context);
        this.allocBuffers.add(buffer);
        int prevSlicePoint = 0;
        for (Long fileOffset : offsetAccum) {
            PinotDataBuffer viewBuffer;
            IndexEntry entry = (IndexEntry)startOffsets.get(fileOffset);
            int endSlicePoint = prevSlicePoint + (int)entry.size;
            this.validateMagicMarker(buffer, prevSlicePoint);
            entry.buffer = viewBuffer = buffer.view(prevSlicePoint + 8, endSlicePoint);
            prevSlicePoint = endSlicePoint;
        }
    }

    private void persistIndexMap(IndexEntry entry) throws IOException {
        File mapFile = new File(this.segmentDirectory, INDEX_MAP_FILE);
        try (PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(mapFile, true)));){
            String startKey = this.getKey(entry.key.name, entry.key.type.getIndexName(), true);
            StringBuilder sb = new StringBuilder();
            sb.append(startKey).append(" = ").append(entry.startOffset);
            writer.println(sb.toString());
            String endKey = this.getKey(entry.key.name, entry.key.type.getIndexName(), false);
            sb = new StringBuilder();
            sb.append(endKey).append(" = ").append(entry.size);
            writer.println(sb.toString());
        }
    }

    private String getKey(String column, String indexName, boolean isStartOffset) {
        return column + MAP_KEY_SEPARATOR + indexName + MAP_KEY_SEPARATOR + (isStartOffset ? MAP_KEY_NAME_START_OFFSET : MAP_KEY_NAME_SIZE);
    }

    private String allocationContext(IndexKey key) {
        return this.getClass().getSimpleName() + key.toString();
    }

    @Override
    public void close() throws IOException {
        for (PinotDataBuffer buf : this.allocBuffers) {
            buf.close();
        }
        this.columnEntries.clear();
        this.allocBuffers.clear();
    }

    @Override
    public void removeIndex(String columnName, ColumnIndexType indexType) {
        throw new UnsupportedOperationException("Index removal is not supported for single file index format. Requested colum: " + columnName + " indexType: " + (Object)((Object)indexType));
    }

    @Override
    public boolean isIndexRemovalSupported() {
        return false;
    }

    public String toString() {
        return this.segmentDirectory.toString() + "/" + this.indexFile.toString();
    }
}

