/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.task;

import com.google.common.collect.Lists;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import org.apache.helix.HelixAdmin;
import org.apache.helix.HelixDataAccessor;
import org.apache.helix.HelixManager;
import org.apache.helix.HelixProperty;
import org.apache.helix.PropertyKey;
import org.apache.helix.ZNRecord;
import org.apache.helix.controller.stages.ClusterDataCache;
import org.apache.helix.controller.stages.CurrentStateOutput;
import org.apache.helix.model.IdealState;
import org.apache.helix.model.Resource;
import org.apache.helix.model.ResourceAssignment;
import org.apache.helix.model.builder.CustomModeISBuilder;
import org.apache.helix.task.JobConfig;
import org.apache.helix.task.JobDag;
import org.apache.helix.task.JobRebalancer;
import org.apache.helix.task.ScheduleConfig;
import org.apache.helix.task.TargetState;
import org.apache.helix.task.TaskConfig;
import org.apache.helix.task.TaskDriver;
import org.apache.helix.task.TaskRebalancer;
import org.apache.helix.task.TaskState;
import org.apache.helix.task.TaskUtil;
import org.apache.helix.task.Workflow;
import org.apache.helix.task.WorkflowConfig;
import org.apache.helix.task.WorkflowContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WorkflowRebalancer
extends TaskRebalancer {
    private static final Logger LOG = LoggerFactory.getLogger(WorkflowRebalancer.class);
    private static final Set<TaskState> finalStates = new HashSet<TaskState>(Arrays.asList(TaskState.COMPLETED, TaskState.FAILED, TaskState.ABORTED, TaskState.TIMED_OUT));

    @Override
    public ResourceAssignment computeBestPossiblePartitionState(ClusterDataCache clusterData, IdealState taskIs, Resource resource, CurrentStateOutput currStateOutput) {
        String workflow = resource.getResourceName();
        LOG.debug("Computer Best Partition for workflow: " + workflow);
        WorkflowConfig workflowCfg = clusterData.getWorkflowConfig(workflow);
        if (workflowCfg == null) {
            LOG.warn("Workflow configuration is NULL for " + workflow);
            return this.buildEmptyAssignment(workflow, currStateOutput);
        }
        WorkflowContext workflowCtx = this.getOrInitializeWorkflowContext(clusterData, workflow);
        TargetState targetState = workflowCfg.getTargetState();
        if (targetState == TargetState.DELETE) {
            LOG.info("Workflow is marked as deleted " + workflow + " cleaning up the workflow context.");
            this.cleanupWorkflow(workflow, workflowCfg);
            return this.buildEmptyAssignment(workflow, currStateOutput);
        }
        if (!workflowCfg.isJobQueue() && !finalStates.contains((Object)workflowCtx.getWorkflowState())) {
            this.scheduleRebalanceForTimeout(workflow, workflowCtx.getStartTime(), workflowCfg.getTimeout());
            if (!TaskState.TIMED_OUT.equals((Object)workflowCtx.getWorkflowState()) && this.isTimeout(workflowCtx.getStartTime(), workflowCfg.getTimeout())) {
                workflowCtx.setWorkflowState(TaskState.TIMED_OUT);
                clusterData.updateWorkflowContext(workflow, workflowCtx, this._manager.getHelixDataAccessor());
            }
        }
        if (!finalStates.contains((Object)workflowCtx.getWorkflowState()) && TargetState.STOP.equals((Object)targetState)) {
            LOG.info("Workflow " + workflow + "is marked as stopped.");
            if (this.isWorkflowStopped(workflowCtx, workflowCfg)) {
                workflowCtx.setWorkflowState(TaskState.STOPPED);
                clusterData.updateWorkflowContext(workflow, workflowCtx, this._manager.getHelixDataAccessor());
            }
            return this.buildEmptyAssignment(workflow, currStateOutput);
        }
        long currentTime = System.currentTimeMillis();
        if (workflowCtx.getFinishTime() == -1L && this.isWorkflowFinished(workflowCtx, workflowCfg, clusterData.getJobConfigMap(), clusterData)) {
            workflowCtx.setFinishTime(currentTime);
            this.updateWorkflowMonitor(workflowCtx, workflowCfg);
            clusterData.updateWorkflowContext(workflow, workflowCtx, this._manager.getHelixDataAccessor());
        }
        if (workflowCtx.getFinishTime() != -1L) {
            LOG.info("Workflow " + workflow + " is finished.");
            long expiryTime = workflowCfg.getExpiry();
            if (workflowCtx.getFinishTime() + expiryTime <= currentTime) {
                LOG.info("Workflow " + workflow + " passed expiry time, cleaning up the workflow context.");
                this.cleanupWorkflow(workflow, workflowCfg);
            } else {
                long cleanupTime = workflowCtx.getFinishTime() + expiryTime;
                _rebalanceScheduler.scheduleRebalance(this._manager, workflow, cleanupTime);
            }
            return this.buildEmptyAssignment(workflow, currStateOutput);
        }
        if (!this.isWorkflowReadyForSchedule(workflowCfg)) {
            LOG.info("Workflow " + workflow + " is not ready to schedule");
            _rebalanceScheduler.scheduleRebalance(this._manager, workflow, workflowCfg.getStartTime().getTime());
            return this.buildEmptyAssignment(workflow, currStateOutput);
        }
        boolean isReady = this.scheduleWorkflowIfReady(workflow, workflowCfg, workflowCtx, clusterData);
        if (isReady) {
            this.scheduleJobs(workflow, workflowCfg, workflowCtx, clusterData.getJobConfigMap(), clusterData);
        } else {
            LOG.debug("Workflow " + workflow + " is not ready to be scheduled.");
        }
        if (!workflowCfg.isTerminable() || workflowCfg.isJobQueue()) {
            this.purgeExpiredJobs(workflow, workflowCfg, workflowCtx);
        }
        clusterData.updateWorkflowContext(workflow, workflowCtx, this._manager.getHelixDataAccessor());
        return this.buildEmptyAssignment(workflow, currStateOutput);
    }

    private WorkflowContext getOrInitializeWorkflowContext(ClusterDataCache clusterData, String workflowName) {
        WorkflowContext workflowCtx = clusterData.getWorkflowContext(workflowName);
        if (workflowCtx == null) {
            WorkflowConfig config = clusterData.getWorkflowConfig(workflowName);
            workflowCtx = new WorkflowContext(new ZNRecord("WorkflowContext"));
            workflowCtx.setStartTime(System.currentTimeMillis());
            workflowCtx.setName(workflowName);
            LOG.debug("Workflow context is created for " + workflowName);
        }
        return workflowCtx;
    }

    private void scheduleJobs(String workflow, WorkflowConfig workflowCfg, WorkflowContext workflowCtx, Map<String, JobConfig> jobConfigMap, ClusterDataCache clusterDataCache) {
        long currentScheduledTime;
        ScheduleConfig scheduleConfig = workflowCfg.getScheduleConfig();
        if (scheduleConfig != null && scheduleConfig.isRecurring()) {
            LOG.debug("Jobs from recurring workflow are not schedule-able");
            return;
        }
        int inCompleteAllJobCount = TaskUtil.getInCompleteJobCount(workflowCfg, workflowCtx);
        int scheduledJobs = 0;
        long timeToSchedule = Long.MAX_VALUE;
        for (String job : workflowCfg.getJobDag().getAllNodes()) {
            TaskState jobState = workflowCtx.getJobState(job);
            if (jobState != null && !jobState.equals((Object)TaskState.NOT_STARTED)) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Job " + job + " is already started or completed.");
                continue;
            }
            if (workflowCfg.isJobQueue() && scheduledJobs >= workflowCfg.getParallelJobs()) {
                if (!LOG.isDebugEnabled()) break;
                LOG.debug(String.format("Workflow %s already have enough job in progress, scheduledJobs(s)=%d, stop scheduling more jobs", workflow, scheduledJobs));
                break;
            }
            if (!this.isJobReadyToSchedule(job, workflowCfg, workflowCtx, inCompleteAllJobCount, jobConfigMap, clusterDataCache)) continue;
            JobConfig jobConfig = jobConfigMap.get(job);
            if (jobConfig == null) {
                LOG.error(String.format("The job config is missing for job %s", job));
                continue;
            }
            long calculatedStartTime = workflowCtx.getJobStartTime(job);
            if (calculatedStartTime < 0L) {
                calculatedStartTime = System.currentTimeMillis();
                if (jobConfig.getExecutionDelay() >= 0L) {
                    calculatedStartTime += jobConfig.getExecutionDelay();
                }
                calculatedStartTime = Math.max(calculatedStartTime, jobConfig.getExecutionStart());
                workflowCtx.setJobStartTime(job, calculatedStartTime);
            }
            if (System.currentTimeMillis() < calculatedStartTime) {
                timeToSchedule = Math.min(timeToSchedule, calculatedStartTime);
                continue;
            }
            this.scheduleSingleJob(job, jobConfig);
            workflowCtx.setJobState(job, TaskState.NOT_STARTED);
            ++scheduledJobs;
        }
        long l = currentScheduledTime = _rebalanceScheduler.getRebalanceTime(workflow) == -1L ? Long.MAX_VALUE : _rebalanceScheduler.getRebalanceTime(workflow);
        if (timeToSchedule < currentScheduledTime) {
            _rebalanceScheduler.scheduleRebalance(this._manager, workflow, timeToSchedule);
        }
    }

    private void scheduleSingleJob(String jobResource, JobConfig jobConfig) {
        int numIndependentTasks;
        HelixAdmin admin = this._manager.getClusterManagmentTool();
        IdealState jobIS = admin.getResourceIdealState(this._manager.getClusterName(), jobResource);
        if (jobIS != null) {
            LOG.info("Job " + jobResource + " idealstate already exists!");
            return;
        }
        TaskUtil.createUserContent(this._manager.getHelixPropertyStore(), jobResource, new ZNRecord("UserContent"));
        int numPartitions = numIndependentTasks = jobConfig.getTaskConfigMap().size();
        if (numPartitions == 0) {
            IdealState targetIs = admin.getResourceIdealState(this._manager.getClusterName(), jobConfig.getTargetResource());
            if (targetIs == null) {
                LOG.warn("Target resource does not exist for job " + jobResource);
            } else {
                numPartitions = targetIs.getPartitionSet().size();
            }
        }
        admin.addResource(this._manager.getClusterName(), jobResource, numPartitions, "Task");
        HelixDataAccessor accessor = this._manager.getHelixDataAccessor();
        PropertyKey.Builder keyBuilder = accessor.keyBuilder();
        HelixProperty resourceConfig = new HelixProperty(jobResource);
        resourceConfig.getRecord().getSimpleFields().putAll(jobConfig.getResourceConfigMap());
        Map<String, TaskConfig> taskConfigMap = jobConfig.getTaskConfigMap();
        if (taskConfigMap != null) {
            for (TaskConfig taskConfig : taskConfigMap.values()) {
                resourceConfig.getRecord().setMapField(taskConfig.getId(), taskConfig.getConfigMap());
            }
        }
        accessor.setProperty(keyBuilder.resourceConfig(jobResource), resourceConfig);
        CustomModeISBuilder builder = new CustomModeISBuilder(jobResource);
        builder.setRebalancerMode(IdealState.RebalanceMode.TASK);
        builder.setNumReplica(1);
        builder.setNumPartitions(numPartitions);
        builder.setStateModel("Task");
        if (jobConfig.getInstanceGroupTag() != null) {
            builder.setNodeGroup(jobConfig.getInstanceGroupTag());
        }
        if (jobConfig.isDisableExternalView()) {
            builder.disableExternalView();
        }
        jobIS = builder.build();
        for (int i = 0; i < numPartitions; ++i) {
            jobIS.getRecord().setListField(jobResource + "_" + i, new ArrayList<String>());
            jobIS.getRecord().setMapField(jobResource + "_" + i, new HashMap<String, String>());
        }
        jobIS.setRebalancerClassName(JobRebalancer.class.getName());
        admin.setResourceIdealState(this._manager.getClusterName(), jobResource, jobIS);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private boolean scheduleWorkflowIfReady(String workflow, WorkflowConfig workflowCfg, WorkflowContext workflowCtx, ClusterDataCache cache) {
        if (workflowCfg == null || workflowCfg.getScheduleConfig() == null) {
            return true;
        }
        ScheduleConfig scheduleConfig = workflowCfg.getScheduleConfig();
        Date startTime = scheduleConfig.getStartTime();
        long currentTime = new Date().getTime();
        long delayFromStart = startTime.getTime() - currentTime;
        if (delayFromStart <= 0L) {
            if (scheduleConfig.isRecurring()) {
                WorkflowContext lastWorkflowCtx;
                if (!workflowCfg.getTargetState().equals((Object)TargetState.START)) {
                    if (!LOG.isDebugEnabled()) return false;
                    LOG.debug("Skip scheduling since the workflow has not been started " + workflow);
                    return false;
                }
                String lastScheduled = workflowCtx.getLastScheduledSingleWorkflow();
                if (lastScheduled != null && (lastWorkflowCtx = cache.getWorkflowContext(lastScheduled)) != null && lastWorkflowCtx.getFinishTime() == -1L) {
                    LOG.info("Skip scheduling since last schedule has not completed yet " + lastScheduled);
                    return false;
                }
                long period = scheduleConfig.getRecurrenceUnit().toMillis(scheduleConfig.getRecurrenceInterval());
                long offsetMultiplier = -delayFromStart / period;
                long timeToSchedule = period * offsetMultiplier + startTime.getTime();
                SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd'T'HHmmss");
                df.setTimeZone(TimeZone.getTimeZone("UTC"));
                String newWorkflowName = workflow + "_" + df.format(new Date(timeToSchedule));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Ready to start workflow " + newWorkflowName);
                }
                if (!newWorkflowName.equals(lastScheduled)) {
                    Workflow clonedWf = WorkflowRebalancer.cloneWorkflow(this._manager, workflow, newWorkflowName, new Date(timeToSchedule));
                    TaskDriver driver = new TaskDriver(this._manager);
                    try {
                        driver.start(clonedWf);
                    }
                    catch (Exception e) {
                        LOG.error("Failed to schedule cloned workflow " + newWorkflowName, (Throwable)e);
                        this._clusterStatusMonitor.updateWorkflowCounters(clonedWf.getWorkflowConfig(), TaskState.FAILED);
                    }
                    workflowCtx.setLastScheduledSingleWorkflow(newWorkflowName);
                }
                _rebalanceScheduler.scheduleRebalance(this._manager, workflow, timeToSchedule + period);
                return false;
            }
            long scheduledTime = _rebalanceScheduler.getRebalanceTime(workflow);
            if (scheduledTime <= 0L || currentTime <= scheduledTime) return true;
            _rebalanceScheduler.removeScheduledRebalance(workflow);
            return true;
        }
        _rebalanceScheduler.scheduleRebalance(this._manager, workflow, startTime.getTime());
        return false;
    }

    public static Workflow cloneWorkflow(HelixManager manager, String origWorkflowName, String newWorkflowName, Date newStartTime) {
        PropertyKey.Builder keyBuilder;
        HelixDataAccessor accessor = manager.getHelixDataAccessor();
        Map resourceConfigMap = accessor.getChildValuesMap((keyBuilder = accessor.keyBuilder()).resourceConfigs());
        if (!resourceConfigMap.containsKey(origWorkflowName)) {
            LOG.error("No such workflow named " + origWorkflowName);
            return null;
        }
        if (resourceConfigMap.containsKey(newWorkflowName)) {
            LOG.error("Workflow with name " + newWorkflowName + " already exists!");
            return null;
        }
        Map<String, String> workflowConfigsMap = ((HelixProperty)resourceConfigMap.get(origWorkflowName)).getRecord().getSimpleFields();
        WorkflowConfig.Builder workflowConfigBlder = WorkflowConfig.Builder.fromMap(workflowConfigsMap);
        if (newStartTime != null) {
            ScheduleConfig scheduleConfig = ScheduleConfig.oneTimeDelayedStart(newStartTime);
            workflowConfigBlder.setScheduleConfig(scheduleConfig);
        }
        workflowConfigBlder.setTerminable(true);
        WorkflowConfig workflowConfig = workflowConfigBlder.build();
        JobDag jobDag = workflowConfig.getJobDag();
        Map<String, Set<String>> parentsToChildren = jobDag.getParentsToChildren();
        Workflow.Builder workflowBuilder = new Workflow.Builder(newWorkflowName);
        workflowBuilder.setWorkflowConfig(workflowConfig);
        Set<String> namespacedJobs = jobDag.getAllNodes();
        for (String namespacedJob : namespacedJobs) {
            if (!resourceConfigMap.containsKey(namespacedJob)) continue;
            String job = TaskUtil.getDenamespacedJobName(origWorkflowName, namespacedJob);
            HelixProperty jobConfig = (HelixProperty)resourceConfigMap.get(namespacedJob);
            Map<String, String> jobSimpleFields = jobConfig.getRecord().getSimpleFields();
            JobConfig.Builder jobCfgBuilder = JobConfig.Builder.fromMap(jobSimpleFields);
            jobCfgBuilder.setWorkflow(newWorkflowName);
            Map<String, Map<String, String>> rawTaskConfigMap = jobConfig.getRecord().getMapFields();
            LinkedList taskConfigs = Lists.newLinkedList();
            for (Map<String, String> rawTaskConfig : rawTaskConfigMap.values()) {
                TaskConfig taskConfig = TaskConfig.Builder.from(rawTaskConfig);
                taskConfigs.add(taskConfig);
            }
            jobCfgBuilder.addTaskConfigs(taskConfigs);
            workflowBuilder.addJob(job, jobCfgBuilder);
            Set<String> children = parentsToChildren.get(namespacedJob);
            if (children == null) continue;
            for (String namespacedChild : children) {
                String child = TaskUtil.getDenamespacedJobName(origWorkflowName, namespacedChild);
                workflowBuilder.addParentChildDependency(job, child);
            }
        }
        return workflowBuilder.build();
    }

    private void cleanupWorkflow(String workflow, WorkflowConfig workflowcfg) {
        LOG.info("Cleaning up workflow: " + workflow);
        if (workflowcfg.isTerminable() || workflowcfg.getTargetState() == TargetState.DELETE) {
            Set<String> jobs = workflowcfg.getJobDag().getAllNodes();
            _rebalanceScheduler.removeScheduledRebalance(workflow);
            for (String job : jobs) {
                _rebalanceScheduler.removeScheduledRebalance(job);
            }
            if (!TaskUtil.removeWorkflow(this._manager.getHelixDataAccessor(), this._manager.getHelixPropertyStore(), workflow, jobs)) {
                LOG.warn("Failed to clean up workflow " + workflow);
            }
        } else {
            LOG.info("Did not clean up workflow " + workflow + " because neither the workflow is non-terminable nor is set to DELETE.");
        }
    }

    private void purgeExpiredJobs(String workflow, WorkflowConfig workflowConfig, WorkflowContext workflowContext) {
        long purgeInterval = workflowConfig.getJobPurgeInterval();
        long currentTime = System.currentTimeMillis();
        if (purgeInterval > 0L && workflowContext.getLastJobPurgeTime() + purgeInterval <= currentTime) {
            Set<String> expiredJobs = TaskUtil.getExpiredJobs(this._manager.getHelixDataAccessor(), this._manager.getHelixPropertyStore(), workflowConfig, workflowContext);
            if (expiredJobs.isEmpty()) {
                LOG.info("No job to purge for the queue " + workflow);
            } else {
                LOG.info("Purge jobs " + expiredJobs + " from queue " + workflow);
                HashSet<String> failedJobRemovals = new HashSet<String>();
                for (String job : expiredJobs) {
                    if (!TaskUtil.removeJob(this._manager.getHelixDataAccessor(), this._manager.getHelixPropertyStore(), job)) {
                        failedJobRemovals.add(job);
                        LOG.warn("Failed to clean up expired and completed jobs from workflow " + workflow);
                    }
                    _rebalanceScheduler.removeScheduledRebalance(job);
                }
                expiredJobs.removeAll(failedJobRemovals);
                if (!TaskUtil.removeJobsFromDag(this._manager.getHelixDataAccessor(), workflow, expiredJobs, true)) {
                    LOG.warn("Error occurred while trying to remove jobs + " + expiredJobs + " from the workflow " + workflow);
                }
                workflowContext.removeJobStates(expiredJobs);
                workflowContext.removeJobStartTime(expiredJobs);
            }
            workflowContext.setLastJobPurgeTime(currentTime);
        }
        this.setNextJobPurgeTime(workflow, currentTime, purgeInterval);
    }

    private void setNextJobPurgeTime(String workflow, long currentTime, long purgeInterval) {
        long nextPurgeTime = currentTime + purgeInterval;
        long currentScheduledTime = _rebalanceScheduler.getRebalanceTime(workflow);
        if (currentScheduledTime == -1L || currentScheduledTime > nextPurgeTime) {
            _rebalanceScheduler.scheduleRebalance(this._manager, workflow, nextPurgeTime);
        }
    }

    @Override
    public IdealState computeNewIdealState(String resourceName, IdealState currentIdealState, CurrentStateOutput currentStateOutput, ClusterDataCache clusterData) {
        return currentIdealState;
    }
}

