/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.raptor.metadata;

import com.facebook.presto.raptor.NodeSupplier;
import com.facebook.presto.raptor.RaptorColumnHandle;
import com.facebook.presto.raptor.RaptorErrorCode;
import com.facebook.presto.raptor.metadata.AssignmentLimiter;
import com.facebook.presto.raptor.metadata.BucketNode;
import com.facebook.presto.raptor.metadata.BucketShards;
import com.facebook.presto.raptor.metadata.ColumnInfo;
import com.facebook.presto.raptor.metadata.ForMetadata;
import com.facebook.presto.raptor.metadata.IndexInserter;
import com.facebook.presto.raptor.metadata.MetadataConfig;
import com.facebook.presto.raptor.metadata.MetadataDao;
import com.facebook.presto.raptor.metadata.NodeSize;
import com.facebook.presto.raptor.metadata.SchemaDaoUtil;
import com.facebook.presto.raptor.metadata.ShardDao;
import com.facebook.presto.raptor.metadata.ShardInfo;
import com.facebook.presto.raptor.metadata.ShardIterator;
import com.facebook.presto.raptor.metadata.ShardManager;
import com.facebook.presto.raptor.metadata.ShardMetadata;
import com.facebook.presto.raptor.metadata.ShardPredicate;
import com.facebook.presto.raptor.util.ArrayUtil;
import com.facebook.presto.raptor.util.DaoSupplier;
import com.facebook.presto.raptor.util.DatabaseUtil;
import com.facebook.presto.raptor.util.UuidUtil;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.Node;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.predicate.TupleDomain;
import com.facebook.presto.spi.type.Type;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Ticker;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ExecutionError;
import com.google.common.util.concurrent.UncheckedExecutionException;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import java.sql.Connection;
import java.sql.JDBCType;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.StringJoiner;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.h2.jdbc.JdbcConnection;
import org.skife.jdbi.v2.Handle;
import org.skife.jdbi.v2.IDBI;
import org.skife.jdbi.v2.Query;
import org.skife.jdbi.v2.ResultIterator;
import org.skife.jdbi.v2.exceptions.DBIException;
import org.skife.jdbi.v2.tweak.HandleConsumer;
import org.skife.jdbi.v2.tweak.ResultSetMapper;
import org.skife.jdbi.v2.util.ByteArrayMapper;

public class DatabaseShardManager
implements ShardManager {
    private static final Logger log = Logger.get(DatabaseShardManager.class);
    private static final String INDEX_TABLE_PREFIX = "x_shards_t";
    private final IDBI dbi;
    private final DaoSupplier<ShardDao> shardDaoSupplier;
    private final ShardDao dao;
    private final NodeSupplier nodeSupplier;
    private final AssignmentLimiter assignmentLimiter;
    private final Ticker ticker;
    private final Duration startupGracePeriod;
    private final long startTime;
    private final LoadingCache<String, Integer> nodeIdCache = CacheBuilder.newBuilder().maximumSize(10000L).build((CacheLoader)new CacheLoader<String, Integer>(){

        public Integer load(String nodeIdentifier) {
            return DatabaseShardManager.this.loadNodeId(nodeIdentifier);
        }
    });
    private final LoadingCache<Long, Map<Integer, String>> bucketAssignmentsCache = CacheBuilder.newBuilder().expireAfterWrite(1L, TimeUnit.SECONDS).build((CacheLoader)new CacheLoader<Long, Map<Integer, String>>(){

        public Map<Integer, String> load(Long distributionId) {
            return DatabaseShardManager.this.loadBucketAssignments(distributionId);
        }
    });

    @Inject
    public DatabaseShardManager(@ForMetadata IDBI dbi, DaoSupplier<ShardDao> shardDaoSupplier, NodeSupplier nodeSupplier, AssignmentLimiter assignmentLimiter, Ticker ticker, MetadataConfig config) {
        this(dbi, shardDaoSupplier, nodeSupplier, assignmentLimiter, ticker, config.getStartupGracePeriod());
    }

    public DatabaseShardManager(IDBI dbi, DaoSupplier<ShardDao> shardDaoSupplier, NodeSupplier nodeSupplier, AssignmentLimiter assignmentLimiter, Ticker ticker, Duration startupGracePeriod) {
        this.dbi = Objects.requireNonNull(dbi, "dbi is null");
        this.shardDaoSupplier = Objects.requireNonNull(shardDaoSupplier, "shardDaoSupplier is null");
        this.dao = shardDaoSupplier.onDemand();
        this.nodeSupplier = Objects.requireNonNull(nodeSupplier, "nodeSupplier is null");
        this.assignmentLimiter = Objects.requireNonNull(assignmentLimiter, "assignmentLimiter is null");
        this.ticker = Objects.requireNonNull(ticker, "ticker is null");
        this.startupGracePeriod = Objects.requireNonNull(startupGracePeriod, "startupGracePeriod is null");
        this.startTime = ticker.read();
        SchemaDaoUtil.createTablesWithRetry(dbi);
    }

    @Override
    public void createTable(long tableId, List<ColumnInfo> columns, boolean bucketed) {
        StringJoiner tableColumns = new StringJoiner(",\n  ", "  ", ",\n").setEmptyValue("");
        for (ColumnInfo column : columns) {
            String columnType = DatabaseShardManager.sqlColumnType(column.getType());
            if (columnType == null) continue;
            tableColumns.add(DatabaseShardManager.minColumn(column.getColumnId()) + " " + columnType);
            tableColumns.add(DatabaseShardManager.maxColumn(column.getColumnId()) + " " + columnType);
        }
        String sql = bucketed ? "CREATE TABLE " + DatabaseShardManager.shardIndexTable(tableId) + " (\n  shard_id BIGINT NOT NULL,\n  shard_uuid BINARY(16) NOT NULL,\n  bucket_number INT NOT NULL\n," + tableColumns + "  PRIMARY KEY (bucket_number, shard_uuid),\n  UNIQUE (shard_id),\n  UNIQUE (shard_uuid)\n)" : "CREATE TABLE " + DatabaseShardManager.shardIndexTable(tableId) + " (\n  shard_id BIGINT NOT NULL PRIMARY KEY,\n  shard_uuid BINARY(16) NOT NULL,\n  node_ids VARBINARY(128) NOT NULL,\n" + tableColumns + "  UNIQUE (shard_uuid)\n)";
        try (Handle handle = this.dbi.open();){
            handle.execute(sql, new Object[0]);
        }
        catch (DBIException e) {
            throw DatabaseUtil.metadataError(e);
        }
    }

    @Override
    public void dropTable(long tableId) {
        DatabaseUtil.runTransaction(this.dbi, (handle, status) -> {
            DatabaseShardManager.lockTable(handle, tableId);
            ShardDao shardDao = this.shardDaoSupplier.attach(handle);
            shardDao.insertDeletedShards(tableId);
            shardDao.dropShardNodes(tableId);
            shardDao.dropShards(tableId);
            MetadataDao dao = (MetadataDao)handle.attach(MetadataDao.class);
            dao.dropColumns(tableId);
            dao.dropTable(tableId);
            return null;
        });
        try (Handle handle2 = this.dbi.open();){
            handle2.execute("DROP TABLE " + DatabaseShardManager.shardIndexTable(tableId), new Object[0]);
        }
        catch (DBIException e) {
            log.warn((Throwable)e, "Failed to drop index table %s", new Object[]{DatabaseShardManager.shardIndexTable(tableId)});
        }
    }

    @Override
    public void addColumn(long tableId, ColumnInfo column) {
        String columnType = DatabaseShardManager.sqlColumnType(column.getType());
        if (columnType == null) {
            return;
        }
        String sql = String.format("ALTER TABLE %s ADD COLUMN (%s %s, %s %s)", DatabaseShardManager.shardIndexTable(tableId), DatabaseShardManager.minColumn(column.getColumnId()), columnType, DatabaseShardManager.maxColumn(column.getColumnId()), columnType);
        try (Handle handle = this.dbi.open();){
            handle.execute(sql, new Object[0]);
        }
        catch (DBIException e) {
            throw DatabaseUtil.metadataError(e);
        }
    }

    @Override
    public void commitShards(long transactionId, long tableId, List<ColumnInfo> columns, Collection<ShardInfo> shards, Optional<String> externalBatchId) {
        if (externalBatchId.isPresent() && this.dao.externalBatchExists(externalBatchId.get())) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_EXTERNAL_BATCH_ALREADY_EXISTS, "External batch already exists: " + externalBatchId.get());
        }
        Map<String, Integer> nodeIds = this.toNodeIdMap(shards);
        this.runCommit(transactionId, handle -> {
            externalBatchId.ifPresent(this.shardDaoSupplier.attach(handle)::insertExternalBatch);
            DatabaseShardManager.lockTable(handle, tableId);
            DatabaseShardManager.insertShardsAndIndex(tableId, columns, shards, nodeIds, handle);
        });
    }

    @Override
    public void replaceShardUuids(long transactionId, long tableId, List<ColumnInfo> columns, Set<UUID> oldShardUuids, Collection<ShardInfo> newShards) {
        Map<String, Integer> nodeIds = this.toNodeIdMap(newShards);
        this.runCommit(transactionId, handle -> {
            DatabaseShardManager.lockTable(handle, tableId);
            for (List shards : Iterables.partition((Iterable)newShards, (int)1000)) {
                DatabaseShardManager.insertShardsAndIndex(tableId, columns, shards, nodeIds, handle);
            }
            for (List uuids : Iterables.partition((Iterable)oldShardUuids, (int)1000)) {
                this.deleteShardsAndIndex(tableId, (Set<UUID>)ImmutableSet.copyOf((Collection)uuids), handle);
            }
        });
    }

    private void runCommit(long transactionId, HandleConsumer callback) {
        int maxAttempts = 5;
        for (int attempt = 1; attempt <= maxAttempts; ++attempt) {
            try {
                this.dbi.useTransaction((handle, status) -> {
                    ShardDao dao = this.shardDaoSupplier.attach(handle);
                    if (DatabaseShardManager.commitTransaction(dao, transactionId)) {
                        callback.useHandle(handle);
                        dao.deleteCreatedShards(transactionId);
                    }
                });
                return;
            }
            catch (DBIException e) {
                Throwables.propagateIfInstanceOf((Throwable)e.getCause(), PrestoException.class);
                if (attempt == maxAttempts) {
                    throw DatabaseUtil.metadataError(e);
                }
                log.warn((Throwable)e, "Failed to commit shards on attempt %d, will retry.", new Object[]{attempt});
                try {
                    TimeUnit.SECONDS.sleep(Math.multiplyExact(attempt, 2));
                    continue;
                }
                catch (InterruptedException ie) {
                    throw DatabaseUtil.metadataError(ie);
                }
            }
        }
    }

    private static boolean commitTransaction(ShardDao dao, long transactionId) {
        if (dao.finalizeTransaction(transactionId, true) != 1) {
            if (Boolean.TRUE.equals(dao.transactionSuccessful(transactionId))) {
                return false;
            }
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.TRANSACTION_CONFLICT, "Transaction commit failed. Please retry the operation.");
        }
        return true;
    }

    private void deleteShardsAndIndex(long tableId, Set<UUID> shardUuids, Handle handle) throws SQLException {
        String args = Joiner.on((String)",").join(Collections.nCopies(shardUuids.size(), "?"));
        ImmutableSet.Builder shardIdSet = ImmutableSet.builder();
        String selectShardNodes = String.format("SELECT shard_id FROM %s WHERE shard_uuid IN (%s)", DatabaseShardManager.shardIndexTable(tableId), args);
        try (PreparedStatement statement = handle.getConnection().prepareStatement(selectShardNodes);){
            DatabaseShardManager.bindUuids(statement, shardUuids);
            try (ResultSet rs = statement.executeQuery();){
                while (rs.next()) {
                    shardIdSet.add((Object)rs.getLong("shard_id"));
                }
            }
        }
        ImmutableSet shardIds = shardIdSet.build();
        if (shardIds.size() != shardUuids.size()) {
            throw DatabaseShardManager.transactionConflict();
        }
        ShardDao dao = this.shardDaoSupplier.attach(handle);
        dao.insertDeletedShards(shardUuids);
        String where = " WHERE shard_id IN (" + args + ")";
        String deleteFromShardNodes = "DELETE FROM shard_nodes " + where;
        String deleteFromShards = "DELETE FROM shards " + where;
        String deleteFromShardIndex = "DELETE FROM " + DatabaseShardManager.shardIndexTable(tableId) + where;
        try (PreparedStatement statement = handle.getConnection().prepareStatement(deleteFromShardNodes);){
            DatabaseShardManager.bindLongs(statement, (Iterable<Long>)shardIds);
            statement.executeUpdate();
        }
        for (String sql : Arrays.asList(deleteFromShards, deleteFromShardIndex)) {
            PreparedStatement statement = handle.getConnection().prepareStatement(sql);
            Throwable throwable = null;
            try {
                DatabaseShardManager.bindLongs(statement, (Iterable<Long>)shardIds);
                if (statement.executeUpdate() == shardIds.size()) continue;
                throw DatabaseShardManager.transactionConflict();
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (statement == null) continue;
                if (throwable != null) {
                    try {
                        statement.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                statement.close();
            }
        }
    }

    private static void bindUuids(PreparedStatement statement, Iterable<UUID> uuids) throws SQLException {
        int i = 1;
        for (UUID uuid : uuids) {
            statement.setBytes(i, UuidUtil.uuidToBytes(uuid));
            ++i;
        }
    }

    private static void bindLongs(PreparedStatement statement, Iterable<Long> values) throws SQLException {
        int i = 1;
        for (long value : values) {
            statement.setLong(i, value);
            ++i;
        }
    }

    private static void insertShardsAndIndex(long tableId, List<ColumnInfo> columns, Collection<ShardInfo> shards, Map<String, Integer> nodeIds, Handle handle) throws SQLException {
        if (shards.isEmpty()) {
            return;
        }
        boolean bucketed = shards.iterator().next().getBucketNumber().isPresent();
        Connection connection = handle.getConnection();
        try (IndexInserter indexInserter = new IndexInserter(connection, tableId, columns);){
            for (List batch : Iterables.partition(shards, (int)DatabaseShardManager.batchSize(connection))) {
                List<Long> shardIds = DatabaseShardManager.insertShards(connection, tableId, batch);
                if (!bucketed) {
                    DatabaseShardManager.insertShardNodes(connection, nodeIds, shardIds, batch);
                }
                for (int i = 0; i < batch.size(); ++i) {
                    ShardInfo shard = (ShardInfo)batch.get(i);
                    Set<Integer> shardNodes = shard.getNodeIdentifiers().stream().map(nodeIds::get).collect(Collectors.toSet());
                    indexInserter.insert(shardIds.get(i), shard.getShardUuid(), shard.getBucketNumber(), shardNodes, shard.getColumnStats());
                }
                indexInserter.execute();
            }
        }
    }

    private static int batchSize(Connection connection) {
        return connection instanceof JdbcConnection ? 1 : 1000;
    }

    private Map<String, Integer> toNodeIdMap(Collection<ShardInfo> shards) {
        Set identifiers = shards.stream().map(ShardInfo::getNodeIdentifiers).flatMap(Collection::stream).collect(Collectors.toSet());
        return Maps.toMap(identifiers, this::getOrCreateNodeId);
    }

    @Override
    public Set<ShardMetadata> getNodeShards(String nodeIdentifier) {
        return this.dao.getNodeShards(nodeIdentifier);
    }

    @Override
    public ResultIterator<BucketShards> getShardNodes(long tableId, TupleDomain<RaptorColumnHandle> effectivePredicate) {
        return new ShardIterator(tableId, false, Optional.empty(), effectivePredicate, this.dbi);
    }

    @Override
    public ResultIterator<BucketShards> getShardNodesBucketed(long tableId, boolean merged, Map<Integer, String> bucketToNode, TupleDomain<RaptorColumnHandle> effectivePredicate) {
        return new ShardIterator(tableId, merged, Optional.of(bucketToNode), effectivePredicate, this.dbi);
    }

    @Override
    public void assignShard(long tableId, UUID shardUuid, String nodeIdentifier, boolean gracePeriod) {
        if (gracePeriod && this.nanosSince(this.startTime).compareTo(this.startupGracePeriod) < 0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.SERVER_STARTING_UP, "Cannot reassign shards while server is starting");
        }
        int nodeId = this.getOrCreateNodeId(nodeIdentifier);
        DatabaseUtil.runTransaction(this.dbi, (handle, status) -> {
            ShardDao dao = this.shardDaoSupplier.attach(handle);
            HashSet<Integer> nodes = new HashSet<Integer>(DatabaseShardManager.fetchLockedNodeIds(handle, tableId, shardUuid));
            if (nodes.add(nodeId)) {
                DatabaseShardManager.updateNodeIds(handle, tableId, shardUuid, nodes);
                dao.insertShardNode(shardUuid, nodeId);
            }
            return null;
        });
    }

    @Override
    public void unassignShard(long tableId, UUID shardUuid, String nodeIdentifier) {
        int nodeId = this.getOrCreateNodeId(nodeIdentifier);
        DatabaseUtil.runTransaction(this.dbi, (handle, status) -> {
            ShardDao dao = this.shardDaoSupplier.attach(handle);
            HashSet<Integer> nodes = new HashSet<Integer>(DatabaseShardManager.fetchLockedNodeIds(handle, tableId, shardUuid));
            if (nodes.remove(nodeId)) {
                DatabaseShardManager.updateNodeIds(handle, tableId, shardUuid, nodes);
                dao.deleteShardNode(shardUuid, nodeId);
            }
            return null;
        });
    }

    @Override
    public Map<String, Long> getNodeBytes() {
        return this.dao.getNodeSizes().stream().collect(Collectors.toMap(NodeSize::getNodeIdentifier, NodeSize::getSizeInBytes));
    }

    @Override
    public long beginTransaction() {
        return this.dao.insertTransaction();
    }

    @Override
    public void rollbackTransaction(long transactionId) {
        this.dao.finalizeTransaction(transactionId, false);
    }

    @Override
    public void createBuckets(long distributionId, int bucketCount) {
        Iterator<String> nodeIterator = DatabaseShardManager.cyclingShuffledIterator(this.getNodeIdentifiers());
        ArrayList<Integer> bucketNumbers = new ArrayList<Integer>();
        ArrayList<Integer> nodeIds = new ArrayList<Integer>();
        for (int bucket = 0; bucket < bucketCount; ++bucket) {
            bucketNumbers.add(bucket);
            nodeIds.add(this.getOrCreateNodeId(nodeIterator.next()));
        }
        DatabaseUtil.runIgnoringConstraintViolation(() -> this.dao.insertBuckets(distributionId, bucketNumbers, nodeIds));
    }

    @Override
    public Map<Integer, String> getBucketAssignments(long distributionId) {
        try {
            return (Map)this.bucketAssignmentsCache.getUnchecked((Object)distributionId);
        }
        catch (ExecutionError | UncheckedExecutionException e) {
            throw Throwables.propagate((Throwable)e.getCause());
        }
    }

    private Map<Integer, String> loadBucketAssignments(long distributionId) {
        Set<String> nodeIds = this.getNodeIdentifiers();
        Iterator<String> nodeIterator = DatabaseShardManager.cyclingShuffledIterator(nodeIds);
        ImmutableMap.Builder assignments = ImmutableMap.builder();
        for (BucketNode bucketNode : this.dao.getBucketNodes(distributionId)) {
            int bucket = bucketNode.getBucketNumber();
            String nodeId = bucketNode.getNodeIdentifier();
            if (!nodeIds.contains(nodeId)) {
                if (this.nanosSince(this.startTime).compareTo(this.startupGracePeriod) < 0) {
                    throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.SERVER_STARTING_UP, "Cannot reassign buckets while server is starting");
                }
                this.assignmentLimiter.checkAssignFrom(nodeId);
                String oldNodeId = nodeId;
                nodeId = nodeIterator.next();
                this.dao.updateBucketNode(distributionId, bucket, this.getOrCreateNodeId(nodeId));
                log.info("Reassigned bucket %s for distribution ID %s from %s to %s", new Object[]{bucket, distributionId, oldNodeId, nodeId});
            }
            assignments.put((Object)bucket, (Object)nodeId);
        }
        return assignments.build();
    }

    private Set<String> getNodeIdentifiers() {
        Set<String> nodeIds = this.nodeSupplier.getWorkerNodes().stream().map(Node::getNodeIdentifier).collect(Collectors.toSet());
        if (nodeIds.isEmpty()) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NO_NODES_AVAILABLE, "No nodes available for bucket assignments");
        }
        return nodeIds;
    }

    private int getOrCreateNodeId(String nodeIdentifier) {
        try {
            return (Integer)this.nodeIdCache.getUnchecked((Object)nodeIdentifier);
        }
        catch (ExecutionError | UncheckedExecutionException e) {
            throw Throwables.propagate((Throwable)e.getCause());
        }
    }

    private int loadNodeId(String nodeIdentifier) {
        Integer id = this.dao.getNodeId(nodeIdentifier);
        if (id != null) {
            return id;
        }
        DatabaseUtil.runIgnoringConstraintViolation(() -> this.dao.insertNode(nodeIdentifier));
        id = this.dao.getNodeId(nodeIdentifier);
        if (id == null) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INTERNAL_ERROR, "node does not exist after insert");
        }
        return id;
    }

    private Duration nanosSince(long nanos) {
        return new Duration((double)(this.ticker.read() - nanos), TimeUnit.NANOSECONDS);
    }

    private static List<Long> insertShards(Connection connection, long tableId, List<ShardInfo> shards) throws SQLException {
        String sql = "INSERT INTO shards (shard_uuid, table_id, create_time, row_count, compressed_size, uncompressed_size, bucket_number)\nVALUES (?, ?, CURRENT_TIMESTAMP, ?, ?, ?, ?)";
        try (PreparedStatement statement = connection.prepareStatement(sql, 1);){
            for (ShardInfo shard : shards) {
                statement.setBytes(1, UuidUtil.uuidToBytes(shard.getShardUuid()));
                statement.setLong(2, tableId);
                statement.setLong(3, shard.getRowCount());
                statement.setLong(4, shard.getCompressedSize());
                statement.setLong(5, shard.getUncompressedSize());
                DatabaseUtil.bindOptionalInt(statement, 6, shard.getBucketNumber());
                statement.addBatch();
            }
            statement.executeBatch();
            ImmutableList.Builder builder = ImmutableList.builder();
            try (ResultSet keys = statement.getGeneratedKeys();){
                while (keys.next()) {
                    builder.add((Object)keys.getLong(1));
                }
            }
            ImmutableList shardIds = builder.build();
            if (shardIds.size() != shards.size()) {
                throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Wrong number of generated keys for inserted shards");
            }
            var9_9 = shardIds;
            return var9_9;
        }
    }

    private static void insertShardNodes(Connection connection, Map<String, Integer> nodeIds, List<Long> shardIds, List<ShardInfo> shards) throws SQLException {
        Preconditions.checkArgument((shardIds.size() == shards.size() ? 1 : 0) != 0, (Object)"lists are not the same size");
        String sql = "INSERT INTO shard_nodes (shard_id, node_id) VALUES (?, ?)";
        try (PreparedStatement statement = connection.prepareStatement(sql);){
            for (int i = 0; i < shards.size(); ++i) {
                for (String identifier : shards.get(i).getNodeIdentifiers()) {
                    statement.setLong(1, shardIds.get(i));
                    statement.setInt(2, nodeIds.get(identifier));
                    statement.addBatch();
                }
            }
            statement.executeBatch();
        }
    }

    private static Collection<Integer> fetchLockedNodeIds(Handle handle, long tableId, UUID shardUuid) {
        String sql = String.format("SELECT node_ids FROM %s WHERE shard_uuid = ? FOR UPDATE", DatabaseShardManager.shardIndexTable(tableId));
        byte[] nodeArray = (byte[])((Query)handle.createQuery(sql).bind(0, UuidUtil.uuidToBytes(shardUuid))).map((ResultSetMapper)ByteArrayMapper.FIRST).first();
        return ArrayUtil.intArrayFromBytes(nodeArray);
    }

    private static void updateNodeIds(Handle handle, long tableId, UUID shardUuid, Set<Integer> nodeIds) {
        String sql = String.format("UPDATE %s SET node_ids = ? WHERE shard_uuid = ?", DatabaseShardManager.shardIndexTable(tableId));
        handle.execute(sql, new Object[]{ArrayUtil.intArrayToBytes(nodeIds), UuidUtil.uuidToBytes(shardUuid)});
    }

    private static void lockTable(Handle handle, long tableId) {
        if (((MetadataDao)handle.attach(MetadataDao.class)).getLockedTableId(tableId) == null) {
            throw DatabaseShardManager.transactionConflict();
        }
    }

    private static PrestoException transactionConflict() {
        return new PrestoException((ErrorCodeSupplier)StandardErrorCode.TRANSACTION_CONFLICT, "Table was updated by a different transaction. Please retry the operation.");
    }

    public static String shardIndexTable(long tableId) {
        return INDEX_TABLE_PREFIX + tableId;
    }

    public static String minColumn(long columnId) {
        Preconditions.checkArgument((columnId >= 0L ? 1 : 0) != 0, (String)"invalid columnId %s", (Object[])new Object[]{columnId});
        return String.format("c%s_min", columnId);
    }

    public static String maxColumn(long columnId) {
        Preconditions.checkArgument((columnId >= 0L ? 1 : 0) != 0, (String)"invalid columnId %s", (Object[])new Object[]{columnId});
        return String.format("c%s_max", columnId);
    }

    private static String sqlColumnType(Type type) {
        JDBCType jdbcType = ShardPredicate.jdbcType(type);
        if (jdbcType != null) {
            switch (jdbcType) {
                case BOOLEAN: {
                    return "boolean";
                }
                case BIGINT: {
                    return "bigint";
                }
                case DOUBLE: {
                    return "double";
                }
                case INTEGER: {
                    return "int";
                }
                case VARBINARY: {
                    return String.format("varbinary(%s)", 100);
                }
            }
        }
        return null;
    }

    private static <T> Iterator<T> cyclingShuffledIterator(Collection<T> collection) {
        ArrayList<T> list = new ArrayList<T>(collection);
        Collections.shuffle(list);
        return Iterables.cycle(list).iterator();
    }
}

