/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.raptor.storage;

import com.facebook.presto.raptor.metadata.ColumnInfo;
import com.facebook.presto.raptor.metadata.DatabaseShardManager;
import com.facebook.presto.raptor.metadata.ForMetadata;
import com.facebook.presto.raptor.metadata.MetadataDao;
import com.facebook.presto.raptor.metadata.ShardInfo;
import com.facebook.presto.raptor.metadata.ShardManager;
import com.facebook.presto.raptor.metadata.ShardMetadata;
import com.facebook.presto.raptor.metadata.TableColumn;
import com.facebook.presto.raptor.metadata.TableMetadata;
import com.facebook.presto.raptor.storage.CompactionSet;
import com.facebook.presto.raptor.storage.CompactionSetCreator;
import com.facebook.presto.raptor.storage.FileCompactionSetCreator;
import com.facebook.presto.raptor.storage.ShardCompactor;
import com.facebook.presto.raptor.storage.StorageManagerConfig;
import com.facebook.presto.raptor.storage.TemporalCompactionSetCreator;
import com.facebook.presto.raptor.util.DatabaseUtil;
import com.facebook.presto.spi.NodeManager;
import com.facebook.presto.spi.block.SortOrder;
import com.facebook.presto.spi.type.DateType;
import com.facebook.presto.spi.type.TimestampType;
import com.facebook.presto.spi.type.Type;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import io.airlift.stats.CounterStat;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import org.skife.jdbi.v2.IDBI;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

public class ShardCompactionManager {
    private static final Logger log = Logger.get(ShardCompactionManager.class);
    private static final double FILL_FACTOR = 0.75;
    private static final int MAX_PENDING_COMPACTIONS = 500;
    private final ScheduledExecutorService compactionDiscoveryService = Executors.newScheduledThreadPool(1, Threads.daemonThreadsNamed((String)"shard-compaction-discovery"));
    private final ExecutorService compactionDriverService = Executors.newFixedThreadPool(1, Threads.daemonThreadsNamed((String)"shard-compaction-driver"));
    private final ExecutorService compactionService;
    private final AtomicBoolean discoveryStarted = new AtomicBoolean();
    private final AtomicBoolean compactionStarted = new AtomicBoolean();
    private final AtomicBoolean shutdown = new AtomicBoolean();
    private final Set<UUID> shardsInProgress = Sets.newConcurrentHashSet();
    private final BlockingQueue<CompactionSet> compactionQueue = new LinkedBlockingQueue<CompactionSet>();
    private final MetadataDao metadataDao;
    private final ShardCompactor compactor;
    private final ShardManager shardManager;
    private final String currentNodeIdentifier;
    private final boolean compactionEnabled;
    private final Duration compactionDiscoveryInterval;
    private final DataSize maxShardSize;
    private final long maxShardRows;
    private final IDBI dbi;
    private final CounterStat compactionSuccessCount = new CounterStat();
    private final CounterStat compactionFailureCount = new CounterStat();

    @Inject
    public ShardCompactionManager(@ForMetadata IDBI dbi, NodeManager nodeManager, ShardManager shardManager, ShardCompactor compactor, StorageManagerConfig config) {
        this(dbi, nodeManager.getCurrentNode().getNodeIdentifier(), shardManager, compactor, config.getCompactionInterval(), config.getMaxShardSize(), config.getMaxShardRows(), config.getCompactionThreads(), config.isCompactionEnabled());
    }

    public ShardCompactionManager(IDBI dbi, String currentNodeIdentifier, ShardManager shardManager, ShardCompactor compactor, Duration compactionDiscoveryInterval, DataSize maxShardSize, long maxShardRows, int compactionThreads, boolean compactionEnabled) {
        this.dbi = Objects.requireNonNull(dbi, "dbi is null");
        this.metadataDao = DatabaseUtil.onDemandDao(dbi, MetadataDao.class);
        this.currentNodeIdentifier = Objects.requireNonNull(currentNodeIdentifier, "currentNodeIdentifier is null");
        this.shardManager = Objects.requireNonNull(shardManager, "shardManager is null");
        this.compactor = Objects.requireNonNull(compactor, "compactor is null");
        this.compactionDiscoveryInterval = Objects.requireNonNull(compactionDiscoveryInterval, "compactionDiscoveryInterval is null");
        Preconditions.checkArgument((maxShardSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxShardSize must be > 0");
        this.maxShardSize = Objects.requireNonNull(maxShardSize, "maxShardSize is null");
        Preconditions.checkArgument((maxShardRows > 0L ? 1 : 0) != 0, (Object)"maxShardRows must be > 0");
        this.maxShardRows = maxShardRows;
        this.compactionEnabled = compactionEnabled;
        if (compactionEnabled) {
            Preconditions.checkArgument((compactionThreads > 0 ? 1 : 0) != 0, (Object)"compactionThreads must be > 0");
            this.compactionService = Executors.newFixedThreadPool(compactionThreads, Threads.daemonThreadsNamed((String)"shard-compactor-%s"));
        } else {
            this.compactionService = null;
        }
    }

    @PostConstruct
    public void start() {
        if (!this.compactionEnabled) {
            return;
        }
        if (!this.discoveryStarted.getAndSet(true)) {
            this.startDiscovery();
        }
        if (!this.compactionStarted.getAndSet(true)) {
            this.compactionDriverService.submit(new ShardCompactionDriver());
        }
    }

    @PreDestroy
    public void shutdown() {
        if (!this.compactionEnabled) {
            return;
        }
        this.shutdown.set(true);
        this.compactionDiscoveryService.shutdown();
        this.compactionDriverService.shutdown();
        this.compactionService.shutdown();
    }

    private void startDiscovery() {
        this.compactionDiscoveryService.scheduleWithFixedDelay(() -> {
            try {
                long interval = (long)this.compactionDiscoveryInterval.convertTo(TimeUnit.SECONDS).getValue();
                TimeUnit.SECONDS.sleep(ThreadLocalRandom.current().nextLong(1L, interval));
                this.discoverShards();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (Throwable t) {
                log.error(t, "Error discovering shards to compact");
            }
        }, 0L, this.compactionDiscoveryInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void discoverShards() {
        if (this.shardsInProgress.size() >= 500) {
            return;
        }
        Set<ShardMetadata> allShards = this.shardManager.getNodeShards(this.currentNodeIdentifier);
        ImmutableListMultimap tableShards = Multimaps.index(allShards, ShardMetadata::getTableId);
        for (Map.Entry entry : Multimaps.asMap((ListMultimap)tableShards).entrySet()) {
            Optional<Long> temporalColumnId;
            long tableId = (Long)entry.getKey();
            if (!this.metadataDao.isCompactionEnabled(tableId) || (temporalColumnId = Optional.ofNullable(this.metadataDao.getTemporalColumnId(tableId))).isPresent() && !this.isValidTemporalColumn(tableId, temporalColumnId.get())) continue;
            CompactionSetCreator compactionSetCreator = this.getCompactionSetCreator(tableId, temporalColumnId);
            Set<ShardMetadata> shards = this.getFilteredShards((List)entry.getValue(), tableId, temporalColumnId);
            this.addToCompactionQueue(compactionSetCreator, tableId, shards);
        }
    }

    private Set<ShardMetadata> getFilteredShards(List<ShardMetadata> shardMetadatas, long tableId, Optional<Long> temporalColumnId) {
        Set<ShardMetadata> shards = shardMetadatas.stream().filter(this::needsCompaction).filter(shard -> !this.shardsInProgress.contains(shard.getShardUuid())).collect(Collectors.toSet());
        if (temporalColumnId.isPresent()) {
            shards = this.filterShardsWithTemporalMetadata(shards, tableId, temporalColumnId.get());
        }
        return shards;
    }

    private CompactionSetCreator getCompactionSetCreator(long tableId, Optional<Long> temporalColumnId) {
        if (!temporalColumnId.isPresent()) {
            return new FileCompactionSetCreator(this.maxShardSize, this.maxShardRows);
        }
        Preconditions.checkState((boolean)this.isValidTemporalColumn(tableId, temporalColumnId.get()), (Object)"invalid temporal column type");
        Type type = this.metadataDao.getTableColumn(tableId, temporalColumnId.get()).getDataType();
        return new TemporalCompactionSetCreator(this.maxShardSize, this.maxShardRows, type);
    }

    private boolean isValidTemporalColumn(long tableId, long temporalColumnId) {
        Type type = this.metadataDao.getTableColumn(tableId, temporalColumnId).getDataType();
        if (!type.equals(DateType.DATE) && !type.equals(TimestampType.TIMESTAMP)) {
            log.warn("Temporal column type of table ID %s set incorrectly to %s", new Object[]{tableId, type});
            return false;
        }
        return true;
    }

    @VisibleForTesting
    Set<ShardMetadata> filterShardsWithTemporalMetadata(Iterable<ShardMetadata> allShards, long tableId, long temporalColumnId) {
        ImmutableMap shardsById = Maps.uniqueIndex(allShards, ShardMetadata::getShardId);
        String minColumn = DatabaseShardManager.minColumn(temporalColumnId);
        String maxColumn = DatabaseShardManager.maxColumn(temporalColumnId);
        ImmutableSet.Builder temporalShards = ImmutableSet.builder();
        try (Connection connection = this.dbi.open().getConnection();){
            for (List shards : Iterables.partition(allShards, (int)1000)) {
                String args = Joiner.on((String)",").join(Collections.nCopies(shards.size(), "?"));
                String sql = String.format("SELECT shard_id, %s, %s FROM %s WHERE shard_id IN (%s)", minColumn, maxColumn, DatabaseShardManager.shardIndexTable(tableId), args);
                PreparedStatement statement = connection.prepareStatement(sql);
                Throwable throwable = null;
                try {
                    for (int i = 0; i < shards.size(); ++i) {
                        statement.setLong(i + 1, ((ShardMetadata)shards.get(i)).getShardId());
                    }
                    ResultSet resultSet = statement.executeQuery();
                    Throwable throwable2 = null;
                    try {
                        while (resultSet.next()) {
                            ShardMetadata shard;
                            long rangeStart = resultSet.getLong(minColumn);
                            if (resultSet.wasNull()) continue;
                            long rangeEnd = resultSet.getLong(maxColumn);
                            if (resultSet.wasNull()) continue;
                            long shardId = resultSet.getLong("shard_id");
                            if (resultSet.wasNull() || (shard = (ShardMetadata)shardsById.get(shardId)) == null) continue;
                            temporalShards.add((Object)shard.withTimeRange(rangeStart, rangeEnd));
                        }
                    }
                    catch (Throwable throwable3) {
                        throwable2 = throwable3;
                        throw throwable3;
                    }
                    finally {
                        if (resultSet == null) continue;
                        if (throwable2 != null) {
                            try {
                                resultSet.close();
                            }
                            catch (Throwable throwable4) {
                                throwable2.addSuppressed(throwable4);
                            }
                            continue;
                        }
                        resultSet.close();
                    }
                }
                catch (Throwable throwable5) {
                    throwable = throwable5;
                    throw throwable5;
                }
                finally {
                    if (statement == null) continue;
                    if (throwable != null) {
                        try {
                            statement.close();
                        }
                        catch (Throwable throwable6) {
                            throwable.addSuppressed(throwable6);
                        }
                        continue;
                    }
                    statement.close();
                }
            }
        }
        catch (SQLException e) {
            throw DatabaseUtil.metadataError(e);
        }
        return temporalShards.build();
    }

    private void addToCompactionQueue(CompactionSetCreator compactionSetCreator, long tableId, Set<ShardMetadata> shardsToCompact) {
        for (CompactionSet compactionSet : compactionSetCreator.createCompactionSets(tableId, shardsToCompact)) {
            if (compactionSet.getShardsToCompact().size() <= 1) continue;
            compactionSet.getShardsToCompact().stream().map(ShardMetadata::getShardUuid).forEach(this.shardsInProgress::add);
            this.compactionQueue.add(compactionSet);
        }
    }

    private boolean needsCompaction(ShardMetadata shard) {
        if ((double)shard.getUncompressedSize() < 0.75 * (double)this.maxShardSize.toBytes()) {
            return true;
        }
        return (double)shard.getRowCount() < 0.75 * (double)this.maxShardRows;
    }

    private TableMetadata getTableMetadata(long tableId) {
        List<TableColumn> sortColumns = this.metadataDao.listSortColumns(tableId);
        List<Long> sortColumnIds = sortColumns.stream().map(TableColumn::getColumnId).collect(Collectors.toList());
        List<ColumnInfo> columns = this.metadataDao.listTableColumns(tableId).stream().map(TableColumn::toColumnInfo).collect(Collectors.toList());
        return new TableMetadata(tableId, columns, sortColumnIds);
    }

    @Managed
    public int getShardsInProgress() {
        return this.shardsInProgress.size();
    }

    @Managed
    @Nested
    public CounterStat getCompactionSuccessCount() {
        return this.compactionSuccessCount;
    }

    @Managed
    @Nested
    public CounterStat getCompactionFailureCount() {
        return this.compactionFailureCount;
    }

    private class CompactionJob
    implements Runnable {
        private final CompactionSet compactionSet;

        public CompactionJob(CompactionSet compactionSet) {
            this.compactionSet = Objects.requireNonNull(compactionSet, "compactionSet is null");
        }

        @Override
        public void run() {
            Set<ShardMetadata> shards = this.compactionSet.getShardsToCompact();
            OptionalInt bucketNumber = shards.iterator().next().getBucketNumber();
            for (ShardMetadata shard : shards) {
                Verify.verify((boolean)bucketNumber.equals(shard.getBucketNumber()), (String)"mismatched bucket numbers", (Object[])new Object[0]);
            }
            Set<UUID> shardUuids = shards.stream().map(ShardMetadata::getShardUuid).collect(Collectors.toSet());
            try {
                this.compactShards(this.compactionSet.getTableId(), bucketNumber, shardUuids);
            }
            catch (IOException e) {
                throw Throwables.propagate((Throwable)e);
            }
            finally {
                ShardCompactionManager.this.shardsInProgress.removeAll(shardUuids);
            }
        }

        private void compactShards(long tableId, OptionalInt bucketNumber, Set<UUID> shardUuids) throws IOException {
            long transactionId = ShardCompactionManager.this.shardManager.beginTransaction();
            try {
                this.compactShards(transactionId, bucketNumber, tableId, shardUuids);
            }
            catch (Throwable e) {
                ShardCompactionManager.this.shardManager.rollbackTransaction(transactionId);
                throw e;
            }
        }

        private void compactShards(long transactionId, OptionalInt bucketNumber, long tableId, Set<UUID> shardUuids) throws IOException {
            TableMetadata metadata = ShardCompactionManager.this.getTableMetadata(tableId);
            List<ShardInfo> newShards = this.performCompaction(transactionId, bucketNumber, shardUuids, metadata);
            log.info("Compacted shards %s into %s", new Object[]{shardUuids, newShards.stream().map(ShardInfo::getShardUuid).collect(Collectors.toList())});
            ShardCompactionManager.this.shardManager.replaceShardUuids(transactionId, tableId, metadata.getColumns(), shardUuids, newShards);
        }

        private List<ShardInfo> performCompaction(long transactionId, OptionalInt bucketNumber, Set<UUID> shardUuids, TableMetadata tableMetadata) throws IOException {
            if (tableMetadata.getSortColumnIds().isEmpty()) {
                return ShardCompactionManager.this.compactor.compact(transactionId, bucketNumber, shardUuids, tableMetadata.getColumns());
            }
            return ShardCompactionManager.this.compactor.compactSorted(transactionId, bucketNumber, shardUuids, tableMetadata.getColumns(), tableMetadata.getSortColumnIds(), Collections.nCopies(tableMetadata.getSortColumnIds().size(), SortOrder.ASC_NULLS_FIRST));
        }
    }

    private class ShardCompactionDriver
    implements Runnable {
        private ShardCompactionDriver() {
        }

        @Override
        public void run() {
            while (!Thread.currentThread().isInterrupted() && !ShardCompactionManager.this.shutdown.get()) {
                try {
                    CompactionSet compactionSet = (CompactionSet)ShardCompactionManager.this.compactionQueue.take();
                    CompletableFuture.runAsync(new CompactionJob(compactionSet), ShardCompactionManager.this.compactionService).whenComplete((none, throwable) -> {
                        if (throwable == null) {
                            ShardCompactionManager.this.compactionSuccessCount.update(1L);
                        } else {
                            log.warn(throwable, "Error in compaction");
                            ShardCompactionManager.this.compactionFailureCount.update(1L);
                        }
                    });
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }
}

