/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.raptor.backup;

import com.facebook.airlift.concurrent.Threads;
import com.facebook.airlift.log.Logger;
import com.facebook.presto.raptor.RaptorErrorCode;
import com.facebook.presto.raptor.backup.BackupConfig;
import com.facebook.presto.raptor.backup.BackupStore;
import com.facebook.presto.raptor.filesystem.LocalOrcDataEnvironment;
import com.facebook.presto.raptor.storage.BackupStats;
import com.facebook.presto.raptor.storage.OrcDataEnvironment;
import com.facebook.presto.raptor.storage.StorageService;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.google.common.base.Preconditions;
import com.google.common.io.Files;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RawLocalFileSystem;
import org.weakref.jmx.Flatten;
import org.weakref.jmx.Managed;

public class BackupManager {
    private static final Logger log = Logger.get(BackupManager.class);
    private final Optional<BackupStore> backupStore;
    private final StorageService storageService;
    private final ExecutorService executorService;
    private final Optional<RawLocalFileSystem> localFileSystem;
    private final AtomicInteger pendingBackups = new AtomicInteger();
    private final BackupStats stats = new BackupStats();

    @Inject
    public BackupManager(Optional<BackupStore> backupStore, StorageService storageService, OrcDataEnvironment environment, BackupConfig config) {
        this(backupStore, storageService, environment, config.getBackupThreads());
    }

    public BackupManager(Optional<BackupStore> backupStore, StorageService storageService, OrcDataEnvironment environment, int backupThreads) {
        Preconditions.checkArgument((backupThreads > 0 ? 1 : 0) != 0, (Object)"backupThreads must be > 0");
        this.backupStore = Objects.requireNonNull(backupStore, "backupStore is null");
        this.storageService = Objects.requireNonNull(storageService, "storageService is null");
        this.executorService = Executors.newFixedThreadPool(backupThreads, Threads.daemonThreadsNamed((String)"background-shard-backup-%s"));
        this.localFileSystem = LocalOrcDataEnvironment.tryGetLocalFileSystem(Objects.requireNonNull(environment, "environment is null"));
        Preconditions.checkState((!backupStore.isPresent() || this.localFileSystem.isPresent() ? 1 : 0) != 0, (Object)"cannot support backup for remote file system");
    }

    @PreDestroy
    public void shutdown() {
        this.executorService.shutdownNow();
    }

    public CompletableFuture<?> submit(UUID uuid, Path source) {
        Objects.requireNonNull(uuid, "uuid is null");
        Objects.requireNonNull(source, "source is null");
        if (!this.backupStore.isPresent()) {
            return CompletableFuture.completedFuture(null);
        }
        this.pendingBackups.incrementAndGet();
        CompletableFuture<Void> future = CompletableFuture.runAsync(new BackgroundBackup(uuid, this.localFileSystem.get().pathToFile(source)), this.executorService);
        future.whenComplete((none, throwable) -> this.pendingBackups.decrementAndGet());
        return future;
    }

    @Managed
    public int getPendingBackupCount() {
        return this.pendingBackups.get();
    }

    @Managed
    @Flatten
    public BackupStats getStats() {
        return this.stats;
    }

    private static boolean filesEqual(File file1, File file2) {
        try {
            return Files.equal((File)file1, (File)file2);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private class BackgroundBackup
    implements Runnable {
        private final UUID uuid;
        private final File source;
        private final long queuedTime = System.nanoTime();

        public BackgroundBackup(UUID uuid, File source) {
            this.uuid = Objects.requireNonNull(uuid, "uuid is null");
            this.source = Objects.requireNonNull(source, "source is null");
        }

        @Override
        public void run() {
            try {
                BackupManager.this.stats.addQueuedTime(Duration.nanosSince((long)this.queuedTime));
                long start = System.nanoTime();
                ((BackupStore)BackupManager.this.backupStore.get()).backupShard(this.uuid, this.source);
                BackupManager.this.stats.addCopyShardDataRate(new DataSize((double)this.source.length(), DataSize.Unit.BYTE), Duration.nanosSince((long)start));
                File restored = new File(((RawLocalFileSystem)BackupManager.this.localFileSystem.get()).pathToFile(BackupManager.this.storageService.getStagingFile(this.uuid)) + ".validate");
                ((BackupStore)BackupManager.this.backupStore.get()).restoreShard(this.uuid, restored);
                if (!BackupManager.filesEqual(this.source, restored)) {
                    BackupManager.this.stats.incrementBackupCorruption();
                    File quarantineBase = ((RawLocalFileSystem)BackupManager.this.localFileSystem.get()).pathToFile(BackupManager.this.storageService.getQuarantineFile(this.uuid));
                    File quarantineOriginal = new File(quarantineBase.getPath() + ".original");
                    File quarantineRestored = new File(quarantineBase.getPath() + ".restored");
                    log.error("Backup is corrupt after write. Quarantining local file: %s", new Object[]{quarantineBase});
                    if (!this.source.renameTo(quarantineOriginal) || !restored.renameTo(quarantineRestored)) {
                        log.warn("Quarantine of corrupt backup shard failed: %s", new Object[]{this.uuid});
                    }
                    throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_BACKUP_CORRUPTION, "Backup is corrupt after write: " + this.uuid);
                }
                if (!restored.delete()) {
                    log.warn("Failed to delete staging file: %s", new Object[]{restored});
                }
                BackupManager.this.stats.incrementBackupSuccess();
            }
            catch (Throwable t) {
                BackupManager.this.stats.incrementBackupFailure();
                throw t;
            }
        }
    }
}

