/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.raptor.filesystem;

import com.facebook.airlift.log.Logger;
import com.facebook.presto.raptor.RaptorErrorCode;
import com.facebook.presto.raptor.filesystem.LocalOrcDataEnvironment;
import com.facebook.presto.raptor.storage.OrcDataEnvironment;
import com.facebook.presto.raptor.storage.StorageManagerConfig;
import com.facebook.presto.raptor.storage.StorageService;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.net.URI;
import java.nio.file.Files;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Pattern;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RawLocalFileSystem;

public class LocalFileStorageService
implements StorageService {
    private static final Logger log = Logger.get(LocalFileStorageService.class);
    private static final Pattern HEX_DIRECTORY = Pattern.compile("[0-9a-f]{2}");
    private static final String FILE_EXTENSION = ".orc";
    private final RawLocalFileSystem localFileSystem;
    private final File baseStorageDir;
    private final File baseStagingDir;
    private final File baseQuarantineDir;

    @Inject
    public LocalFileStorageService(OrcDataEnvironment environment, StorageManagerConfig storageManagerConfig) {
        this(environment, storageManagerConfig.getDataDirectory());
    }

    public LocalFileStorageService(OrcDataEnvironment environment, URI dataDirectory) {
        Optional<RawLocalFileSystem> fileSystem = LocalOrcDataEnvironment.tryGetLocalFileSystem(Objects.requireNonNull(environment, "environment is null"));
        Preconditions.checkState((boolean)fileSystem.isPresent(), (Object)"LocalFileStorageService has to have local file system");
        Preconditions.checkState((boolean)dataDirectory.isAbsolute(), (Object)"dataDirectory URI is not absolute");
        Preconditions.checkState((boolean)dataDirectory.getScheme().equals("file"), (Object)"dataDirectory URI is not pointing to local file system");
        this.localFileSystem = fileSystem.get();
        File baseDataDir = Objects.requireNonNull(this.localFileSystem.pathToFile(new Path(dataDirectory)), "dataDirectory is null");
        this.baseStorageDir = new File(baseDataDir, "storage");
        this.baseStagingDir = new File(baseDataDir, "staging");
        this.baseQuarantineDir = new File(baseDataDir, "quarantine");
    }

    @Override
    @PostConstruct
    public void start() {
        this.deleteStagingFilesAsync();
        this.createDirectory(new Path(this.baseStagingDir.toURI()));
        this.createDirectory(new Path(this.baseStorageDir.toURI()));
        this.createDirectory(new Path(this.baseQuarantineDir.toURI()));
    }

    @Override
    public long getAvailableBytes() {
        return this.baseStorageDir.getUsableSpace();
    }

    @PreDestroy
    public void stop() throws IOException {
        LocalFileStorageService.deleteDirectory(this.baseStagingDir);
    }

    @Override
    public Path getStorageFile(UUID shardUuid) {
        return new Path(LocalFileStorageService.getFileSystemPath(this.baseStorageDir, shardUuid).toString());
    }

    @Override
    public Path getStagingFile(UUID shardUuid) {
        String name = LocalFileStorageService.getFileSystemPath(new File("/"), shardUuid).getName();
        return new Path(this.baseStagingDir.toString(), name);
    }

    @Override
    public Path getQuarantineFile(UUID shardUuid) {
        String name = LocalFileStorageService.getFileSystemPath(new File("/"), shardUuid).getName();
        return new Path(this.baseQuarantineDir.toString(), name);
    }

    @Override
    public Set<UUID> getStorageShards() {
        ImmutableSet.Builder shards = ImmutableSet.builder();
        for (File level1 : LocalFileStorageService.listFiles(this.baseStorageDir, LocalFileStorageService::isHexDirectory)) {
            for (File level2 : LocalFileStorageService.listFiles(level1, LocalFileStorageService::isHexDirectory)) {
                for (File file : LocalFileStorageService.listFiles(level2, path -> true)) {
                    if (!file.isFile()) continue;
                    LocalFileStorageService.uuidFromFileName(file.getName()).ifPresent(arg_0 -> ((ImmutableSet.Builder)shards).add(arg_0));
                }
            }
        }
        return shards.build();
    }

    @Override
    public void createParents(Path file) {
        this.createDirectory(file.getParent());
    }

    @Override
    public void promoteFromStagingToStorage(UUID shardUuid) {
        Path stagingFile = this.getStagingFile(shardUuid);
        Path storageFile = this.getStorageFile(shardUuid);
        this.createParents(storageFile);
        try {
            this.localFileSystem.rename(stagingFile, storageFile);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to move shard file", (Throwable)e);
        }
    }

    public static File getFileSystemPath(File base, UUID shardUuid) {
        String uuid = shardUuid.toString().toLowerCase(Locale.ENGLISH);
        return base.toPath().resolve(uuid.substring(0, 2)).resolve(uuid.substring(2, 4)).resolve(uuid + FILE_EXTENSION).toFile();
    }

    private void deleteStagingFilesAsync() {
        List<File> files = LocalFileStorageService.listFiles(this.baseStagingDir, null);
        if (!files.isEmpty()) {
            new Thread(() -> {
                for (File file : files) {
                    try {
                        Files.deleteIfExists(file.toPath());
                    }
                    catch (IOException e) {
                        log.warn((Throwable)e, "Failed to delete file: %s", new Object[]{file.getAbsolutePath()});
                    }
                }
            }, "background-staging-delete").start();
        }
    }

    private static void deleteDirectory(File dir) throws IOException {
        if (!dir.exists()) {
            return;
        }
        File[] files = dir.listFiles();
        if (files == null) {
            throw new IOException("Failed to list directory: " + dir);
        }
        for (File file : files) {
            Files.deleteIfExists(file.toPath());
        }
        Files.deleteIfExists(dir.toPath());
    }

    private void createDirectory(Path file) {
        File directory = this.localFileSystem.pathToFile(file);
        if (!directory.mkdirs() && !directory.isDirectory()) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed creating directories: " + directory);
        }
    }

    private static List<File> listFiles(File dir, FileFilter filter) {
        Object[] files = dir.listFiles(filter);
        if (files == null) {
            return ImmutableList.of();
        }
        return ImmutableList.copyOf((Object[])files);
    }

    private static boolean isHexDirectory(File file) {
        return file.isDirectory() && HEX_DIRECTORY.matcher(file.getName()).matches();
    }

    private static Optional<UUID> uuidFromFileName(String name) {
        if (name.endsWith(FILE_EXTENSION)) {
            name = name.substring(0, name.length() - FILE_EXTENSION.length());
            return LocalFileStorageService.uuidFromString(name);
        }
        return Optional.empty();
    }

    private static Optional<UUID> uuidFromString(String value) {
        try {
            return Optional.of(UUID.fromString(value));
        }
        catch (IllegalArgumentException e) {
            return Optional.empty();
        }
    }
}

