/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.timeline;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.HoodieTimeline;
import org.apache.hudi.common.table.timeline.HoodieDefaultTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.util.FileIOUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class HoodieActiveTimeline
extends HoodieDefaultTimeline {
    public static final SimpleDateFormat COMMIT_FORMATTER = new SimpleDateFormat("yyyyMMddHHmmss");
    public static final Set<String> VALID_EXTENSIONS_IN_ACTIVE_TIMELINE = new HashSet<String>(Arrays.asList(".commit", ".inflight", ".commit.requested", ".deltacommit", ".deltacommit.inflight", ".deltacommit.requested", ".savepoint", ".savepoint.inflight", ".clean", ".clean.requested", ".clean.inflight", INFLIGHT_COMPACTION_EXTENSION, REQUESTED_COMPACTION_EXTENSION, ".restore.inflight", ".restore"));
    private static final Logger LOG = LogManager.getLogger(HoodieActiveTimeline.class);
    protected HoodieTableMetaClient metaClient;
    private static AtomicReference<String> lastInstantTime = new AtomicReference<String>(String.valueOf(Integer.MIN_VALUE));

    public static String createNewInstantTime() {
        return lastInstantTime.updateAndGet(oldVal -> {
            String newCommitTime = null;
            while (HoodieTimeline.compareTimestamps(newCommitTime = COMMIT_FORMATTER.format(new Date()), oldVal, LESSER_OR_EQUAL)) {
            }
            return newCommitTime;
        });
    }

    protected HoodieActiveTimeline(HoodieTableMetaClient metaClient, Set<String> includedExtensions) {
        this(metaClient, includedExtensions, true);
    }

    protected HoodieActiveTimeline(HoodieTableMetaClient metaClient, Set<String> includedExtensions, boolean applyLayoutFilters) {
        try {
            this.setInstants(metaClient.scanHoodieInstantsFromFileSystem(includedExtensions, applyLayoutFilters));
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to scan metadata", e);
        }
        this.metaClient = metaClient;
        this.details = this::getInstantDetails;
        LOG.info((Object)("Loaded instants " + this.getInstants().collect(Collectors.toList())));
    }

    public HoodieActiveTimeline(HoodieTableMetaClient metaClient) {
        this(metaClient, (Set<String>)new ImmutableSet.Builder().addAll(VALID_EXTENSIONS_IN_ACTIVE_TIMELINE).build());
    }

    public HoodieActiveTimeline(HoodieTableMetaClient metaClient, boolean applyLayoutFilter) {
        this(metaClient, (Set<String>)new ImmutableSet.Builder().addAll(VALID_EXTENSIONS_IN_ACTIVE_TIMELINE).build(), applyLayoutFilter);
    }

    public HoodieActiveTimeline() {
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
    }

    public HoodieTimeline getCommitsTimeline() {
        return this.getTimelineOfActions(Sets.newHashSet((Object[])new String[]{"commit", "deltacommit"}));
    }

    @Override
    public HoodieTimeline getCommitsAndCompactionTimeline() {
        return this.getTimelineOfActions(Sets.newHashSet((Object[])new String[]{"commit", "deltacommit", "compaction"}));
    }

    public HoodieTimeline getAllCommitsTimeline() {
        return this.getTimelineOfActions(Sets.newHashSet((Object[])new String[]{"commit", "deltacommit", "clean", "compaction", "savepoint", "rollback"}));
    }

    public HoodieTimeline getCommitTimeline() {
        return this.getTimelineOfActions(Sets.newHashSet((Object[])new String[]{"commit"}));
    }

    public HoodieTimeline getDeltaCommitTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("deltacommit"), this::getInstantDetails);
    }

    public HoodieTimeline getTimelineOfActions(Set<String> actions) {
        return new HoodieDefaultTimeline(this.getInstants().filter((? super T s) -> actions.contains(s.getAction())), this::getInstantDetails);
    }

    public HoodieTimeline getCleanerTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("clean"), this::getInstantDetails);
    }

    public HoodieTimeline getRollbackTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("rollback"), this::getInstantDetails);
    }

    public HoodieTimeline getSavePointTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("savepoint"), this::getInstantDetails);
    }

    public HoodieTimeline getRestoreTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("restore"), this::getInstantDetails);
    }

    protected Stream<HoodieInstant> filterInstantsByAction(String action) {
        return this.getInstants().filter((? super T s) -> s.getAction().equals(action));
    }

    public void createNewInstant(HoodieInstant instant) {
        LOG.info((Object)("Creating a new instant " + instant));
        this.createFileInMetaPath(instant.getFileName(), Option.empty(), false);
    }

    public void saveAsComplete(HoodieInstant instant, Option<byte[]> data) {
        LOG.info((Object)("Marking instant complete " + instant));
        Preconditions.checkArgument((boolean)instant.isInflight(), (Object)("Could not mark an already completed instant as complete again " + instant));
        this.transitionState(instant, HoodieTimeline.getCompletedInstant(instant), data);
        LOG.info((Object)("Completed " + instant));
    }

    public HoodieInstant revertToInflight(HoodieInstant instant) {
        LOG.info((Object)("Reverting instant to inflight " + instant));
        HoodieInstant inflight = HoodieTimeline.getInflightInstant(instant, this.metaClient.getTableType());
        this.revertCompleteToInflight(instant, inflight);
        LOG.info((Object)("Reverted " + instant + " to inflight " + inflight));
        return inflight;
    }

    public void deleteInflight(HoodieInstant instant) {
        Preconditions.checkArgument((boolean)instant.isInflight());
        this.deleteInstantFile(instant);
    }

    public void deletePending(HoodieInstant instant) {
        Preconditions.checkArgument((!instant.isCompleted() ? 1 : 0) != 0);
        this.deleteInstantFile(instant);
    }

    public void deleteCompactionRequested(HoodieInstant instant) {
        Preconditions.checkArgument((boolean)instant.isRequested());
        Preconditions.checkArgument((instant.getAction() == "compaction" ? 1 : 0) != 0);
        this.deleteInstantFile(instant);
    }

    private void deleteInstantFile(HoodieInstant instant) {
        LOG.info((Object)("Deleting instant " + instant));
        Path inFlightCommitFilePath = new Path(this.metaClient.getMetaPath(), instant.getFileName());
        try {
            boolean result = this.metaClient.getFs().delete(inFlightCommitFilePath, false);
            if (!result) {
                throw new HoodieIOException("Could not delete instant " + instant);
            }
            LOG.info((Object)("Removed instant " + instant));
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not remove inflight commit " + inFlightCommitFilePath, e);
        }
    }

    @Override
    public Option<byte[]> getInstantDetails(HoodieInstant instant) {
        Path detailPath = new Path(this.metaClient.getMetaPath(), instant.getFileName());
        return this.readDataFromPath(detailPath);
    }

    public Option<byte[]> readCleanerInfoAsBytes(HoodieInstant instant) {
        return this.readDataFromPath(new Path(this.metaClient.getMetaPath(), instant.getFileName()));
    }

    public Option<byte[]> readCompactionPlanAsBytes(HoodieInstant instant) {
        try {
            return this.readDataFromPath(new Path(this.metaClient.getMetaAuxiliaryPath(), instant.getFileName()));
        }
        catch (HoodieIOException e) {
            if (e.getIOException() instanceof FileNotFoundException) {
                return this.readDataFromPath(new Path(this.metaClient.getMetaPath(), instant.getFileName()));
            }
            throw e;
        }
    }

    public HoodieInstant revertCompactionInflightToRequested(HoodieInstant inflightInstant) {
        Preconditions.checkArgument((boolean)inflightInstant.getAction().equals("compaction"));
        Preconditions.checkArgument((boolean)inflightInstant.isInflight());
        HoodieInstant requestedInstant = new HoodieInstant(HoodieInstant.State.REQUESTED, "compaction", inflightInstant.getTimestamp());
        if (this.metaClient.getTimelineLayoutVersion().isNullVersion()) {
            this.transitionState(inflightInstant, requestedInstant, Option.empty());
        } else {
            this.deleteInflight(inflightInstant);
        }
        return requestedInstant;
    }

    public HoodieInstant transitionCompactionRequestedToInflight(HoodieInstant requestedInstant) {
        Preconditions.checkArgument((boolean)requestedInstant.getAction().equals("compaction"));
        Preconditions.checkArgument((boolean)requestedInstant.isRequested());
        HoodieInstant inflightInstant = new HoodieInstant(HoodieInstant.State.INFLIGHT, "compaction", requestedInstant.getTimestamp());
        this.transitionState(requestedInstant, inflightInstant, Option.empty());
        return inflightInstant;
    }

    public HoodieInstant transitionCompactionInflightToComplete(HoodieInstant inflightInstant, Option<byte[]> data) {
        Preconditions.checkArgument((boolean)inflightInstant.getAction().equals("compaction"));
        Preconditions.checkArgument((boolean)inflightInstant.isInflight());
        HoodieInstant commitInstant = new HoodieInstant(HoodieInstant.State.COMPLETED, "commit", inflightInstant.getTimestamp());
        this.transitionState(inflightInstant, commitInstant, data);
        return commitInstant;
    }

    private void createFileInAuxiliaryFolder(HoodieInstant instant, Option<byte[]> data) {
        Path fullPath = new Path(this.metaClient.getMetaAuxiliaryPath(), instant.getFileName());
        this.createFileInPath(fullPath, data);
    }

    public HoodieInstant transitionCleanInflightToComplete(HoodieInstant inflightInstant, Option<byte[]> data) {
        Preconditions.checkArgument((boolean)inflightInstant.getAction().equals("clean"));
        Preconditions.checkArgument((boolean)inflightInstant.isInflight());
        HoodieInstant commitInstant = new HoodieInstant(HoodieInstant.State.COMPLETED, "clean", inflightInstant.getTimestamp());
        this.transitionState(inflightInstant, commitInstant, data);
        return commitInstant;
    }

    public HoodieInstant transitionCleanRequestedToInflight(HoodieInstant requestedInstant, Option<byte[]> data) {
        Preconditions.checkArgument((boolean)requestedInstant.getAction().equals("clean"));
        Preconditions.checkArgument((boolean)requestedInstant.isRequested());
        HoodieInstant inflight = new HoodieInstant(HoodieInstant.State.INFLIGHT, "clean", requestedInstant.getTimestamp());
        this.transitionState(requestedInstant, inflight, data);
        return inflight;
    }

    private void transitionState(HoodieInstant fromInstant, HoodieInstant toInstant, Option<byte[]> data) {
        Preconditions.checkArgument((boolean)fromInstant.getTimestamp().equals(toInstant.getTimestamp()));
        try {
            if (this.metaClient.getTimelineLayoutVersion().isNullVersion()) {
                this.createFileInMetaPath(fromInstant.getFileName(), data, false);
                Path fromInstantPath = new Path(this.metaClient.getMetaPath(), fromInstant.getFileName());
                Path toInstantPath = new Path(this.metaClient.getMetaPath(), toInstant.getFileName());
                boolean success = this.metaClient.getFs().rename(fromInstantPath, toInstantPath);
                if (!success) {
                    throw new HoodieIOException("Could not rename " + fromInstantPath + " to " + toInstantPath);
                }
            } else {
                LOG.info((Object)("Checking for file exists ?" + new Path(this.metaClient.getMetaPath(), fromInstant.getFileName())));
                Preconditions.checkArgument((boolean)this.metaClient.getFs().exists(new Path(this.metaClient.getMetaPath(), fromInstant.getFileName())));
                this.createImmutableFileInPath(new Path(this.metaClient.getMetaPath(), toInstant.getFileName()), data);
                LOG.info((Object)("Create new file for toInstant ?" + new Path(this.metaClient.getMetaPath(), toInstant.getFileName())));
            }
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not complete " + fromInstant, e);
        }
    }

    private void revertCompleteToInflight(HoodieInstant completed, HoodieInstant inflight) {
        Preconditions.checkArgument((boolean)completed.getTimestamp().equals(inflight.getTimestamp()));
        Path inFlightCommitFilePath = new Path(this.metaClient.getMetaPath(), inflight.getFileName());
        Path commitFilePath = new Path(this.metaClient.getMetaPath(), completed.getFileName());
        try {
            if (this.metaClient.getTimelineLayoutVersion().isNullVersion()) {
                boolean success;
                if (!this.metaClient.getFs().exists(inFlightCommitFilePath) && !(success = this.metaClient.getFs().rename(commitFilePath, inFlightCommitFilePath))) {
                    throw new HoodieIOException("Could not rename " + commitFilePath + " to " + inFlightCommitFilePath);
                }
            } else {
                Path requestedInstantFilePath = new Path(this.metaClient.getMetaPath(), new HoodieInstant(HoodieInstant.State.REQUESTED, inflight.getAction(), inflight.getTimestamp()).getFileName());
                if (!this.metaClient.getFs().exists(requestedInstantFilePath)) {
                    this.metaClient.getFs().create(requestedInstantFilePath, false).close();
                }
                if (!this.metaClient.getFs().exists(inFlightCommitFilePath)) {
                    this.metaClient.getFs().create(inFlightCommitFilePath, false).close();
                }
                boolean success = this.metaClient.getFs().delete(commitFilePath, false);
                Preconditions.checkArgument((boolean)success, (Object)"State Reverting failed");
            }
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not complete revert " + completed, e);
        }
    }

    public void transitionRequestedToInflight(HoodieInstant requested, Option<byte[]> content) {
        HoodieInstant inflight = new HoodieInstant(HoodieInstant.State.INFLIGHT, requested.getAction(), requested.getTimestamp());
        Preconditions.checkArgument((boolean)requested.isRequested(), (Object)("Instant " + requested + " in wrong state"));
        this.transitionState(requested, inflight, content);
    }

    public void saveToCompactionRequested(HoodieInstant instant, Option<byte[]> content) {
        this.saveToCompactionRequested(instant, content, false);
    }

    public void saveToCompactionRequested(HoodieInstant instant, Option<byte[]> content, boolean overwrite) {
        Preconditions.checkArgument((boolean)instant.getAction().equals("compaction"));
        this.createFileInAuxiliaryFolder(instant, content);
        this.createFileInMetaPath(instant.getFileName(), content, overwrite);
    }

    public void saveToCleanRequested(HoodieInstant instant, Option<byte[]> content) {
        Preconditions.checkArgument((boolean)instant.getAction().equals("clean"));
        Preconditions.checkArgument((boolean)instant.getState().equals((Object)HoodieInstant.State.REQUESTED));
        this.createFileInMetaPath(instant.getFileName(), content, false);
    }

    private void createFileInMetaPath(String filename, Option<byte[]> content, boolean allowOverwrite) {
        Path fullPath = new Path(this.metaClient.getMetaPath(), filename);
        if (allowOverwrite || this.metaClient.getTimelineLayoutVersion().isNullVersion()) {
            this.createFileInPath(fullPath, content);
        } else {
            this.createImmutableFileInPath(fullPath, content);
        }
    }

    private void createFileInPath(Path fullPath, Option<byte[]> content) {
        try {
            if (!this.metaClient.getFs().exists(fullPath)) {
                if (this.metaClient.getFs().createNewFile(fullPath)) {
                    LOG.info((Object)("Created a new file in meta path: " + fullPath));
                } else {
                    throw new HoodieIOException("Failed to create file " + fullPath);
                }
            }
            if (content.isPresent()) {
                FSDataOutputStream fsout = this.metaClient.getFs().create(fullPath, true);
                fsout.write(content.get());
                fsout.close();
            }
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to create file " + fullPath, e);
        }
    }

    private void createImmutableFileInPath(Path fullPath, Option<byte[]> content) {
        FSDataOutputStream fsout = null;
        try {
            fsout = this.metaClient.getFs().create(fullPath, false);
            if (content.isPresent()) {
                fsout.write(content.get());
            }
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to create file " + fullPath, e);
        }
        finally {
            try {
                if (null != fsout) {
                    fsout.close();
                }
            }
            catch (IOException e) {
                throw new HoodieIOException("Failed to close file " + fullPath, e);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Option<byte[]> readDataFromPath(Path detailPath) {
        try (FSDataInputStream is = this.metaClient.getFs().open(detailPath);){
            Option<byte[]> option = Option.of(FileIOUtils.readAsByteArray((InputStream)is));
            return option;
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not read commit details from " + detailPath, e);
        }
    }

    public HoodieActiveTimeline reload() {
        return new HoodieActiveTimeline(this.metaClient);
    }
}

