/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.controller.stages;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.concurrent.Callable;
import org.apache.helix.HelixManager;
import org.apache.helix.controller.LogUtil;
import org.apache.helix.controller.pipeline.AbstractBaseStage;
import org.apache.helix.controller.pipeline.StageException;
import org.apache.helix.controller.rebalancer.AbstractRebalancer;
import org.apache.helix.controller.rebalancer.AutoRebalancer;
import org.apache.helix.controller.rebalancer.CustomRebalancer;
import org.apache.helix.controller.rebalancer.MaintenanceRebalancer;
import org.apache.helix.controller.rebalancer.Rebalancer;
import org.apache.helix.controller.rebalancer.SemiAutoRebalancer;
import org.apache.helix.controller.rebalancer.internal.MappingCalculator;
import org.apache.helix.controller.stages.AttributeName;
import org.apache.helix.controller.stages.BestPossibleStateOutput;
import org.apache.helix.controller.stages.ClusterDataCache;
import org.apache.helix.controller.stages.ClusterEvent;
import org.apache.helix.controller.stages.CurrentStateOutput;
import org.apache.helix.model.IdealState;
import org.apache.helix.model.InstanceConfig;
import org.apache.helix.model.Partition;
import org.apache.helix.model.Resource;
import org.apache.helix.model.ResourceAssignment;
import org.apache.helix.model.StateModelDefinition;
import org.apache.helix.monitoring.mbeans.ClusterStatusMonitor;
import org.apache.helix.task.JobContext;
import org.apache.helix.task.JobRebalancer;
import org.apache.helix.task.TaskDriver;
import org.apache.helix.task.TaskRebalancer;
import org.apache.helix.util.HelixUtil;
import org.apache.helix.util.StatusUpdateUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BestPossibleStateCalcStage
extends AbstractBaseStage {
    private static final Logger logger = LoggerFactory.getLogger((String)BestPossibleStateCalcStage.class.getName());
    private final StatusUpdateUtil _statusUpdateUtil = new StatusUpdateUtil();

    @Override
    public void process(ClusterEvent event) throws Exception {
        this._eventId = event.getEventId();
        CurrentStateOutput currentStateOutput = (CurrentStateOutput)event.getAttribute(AttributeName.CURRENT_STATE.name());
        final Map resourceMap = (Map)event.getAttribute(AttributeName.RESOURCES_TO_REBALANCE.name());
        final ClusterStatusMonitor clusterStatusMonitor = (ClusterStatusMonitor)event.getAttribute(AttributeName.clusterStatusMonitor.name());
        ClusterDataCache cache = (ClusterDataCache)event.getAttribute(AttributeName.ClusterDataCache.name());
        if (currentStateOutput == null || resourceMap == null || cache == null) {
            throw new StageException("Missing attributes in event:" + event + ". Requires CURRENT_STATE|RESOURCES|DataCache");
        }
        cache.resetActiveTaskCount(currentStateOutput);
        this.validateOfflineInstancesLimit(cache, (HelixManager)event.getAttribute(AttributeName.helixmanager.name()), clusterStatusMonitor);
        final BestPossibleStateOutput bestPossibleStateOutput = this.compute(event, resourceMap, currentStateOutput);
        event.addAttribute(AttributeName.BEST_POSSIBLE_STATE.name(), bestPossibleStateOutput);
        if (!cache.isTaskCache()) {
            final Map<String, InstanceConfig> instanceConfigMap = cache.getInstanceConfigMap();
            final Map<String, StateModelDefinition> stateModelDefMap = cache.getStateModelDefMap();
            BestPossibleStateCalcStage.asyncExecute(cache.getAsyncTasksThreadPool(), new Callable<Object>(){

                @Override
                public Object call() {
                    try {
                        if (clusterStatusMonitor != null) {
                            clusterStatusMonitor.setPerInstanceResourceStatus(bestPossibleStateOutput, instanceConfigMap, resourceMap, stateModelDefMap);
                        }
                    }
                    catch (Exception e) {
                        LogUtil.logError(logger, BestPossibleStateCalcStage.this._eventId, "Could not update cluster status metrics!", e);
                    }
                    return null;
                }
            });
        }
    }

    private BestPossibleStateOutput compute(ClusterEvent event, Map<String, Resource> resourceMap, CurrentStateOutput currentStateOutput) {
        ClusterDataCache cache = (ClusterDataCache)event.getAttribute(AttributeName.ClusterDataCache.name());
        BestPossibleStateOutput output = new BestPossibleStateOutput();
        PriorityQueue<ResourcePriority> resourcePriorityQueue = new PriorityQueue<ResourcePriority>();
        TaskDriver taskDriver = null;
        HelixManager helixManager = (HelixManager)event.getAttribute(AttributeName.helixmanager.name());
        if (helixManager != null) {
            taskDriver = new TaskDriver(helixManager);
        }
        for (Resource resource : resourceMap.values()) {
            resourcePriorityQueue.add(new ResourcePriority(resource, cache.getIdealState(resource.getResourceName()), taskDriver));
        }
        ArrayList<String> failureResources = new ArrayList<String>();
        Iterator itr = resourcePriorityQueue.iterator();
        while (itr.hasNext()) {
            Resource resource = ((ResourcePriority)itr.next()).getResource();
            if (this.computeResourceBestPossibleState(event, cache, currentStateOutput, resource, output)) continue;
            failureResources.add(resource.getResourceName());
            LogUtil.logWarn(logger, this._eventId, "Failed to calculate best possible states for " + resource.getResourceName());
        }
        if (!cache.isTaskCache()) {
            ClusterStatusMonitor clusterStatusMonitor = (ClusterStatusMonitor)event.getAttribute(AttributeName.clusterStatusMonitor.name());
            this.updateRebalanceStatus(!failureResources.isEmpty(), helixManager, cache, clusterStatusMonitor, "Failed to calculate best possible states for " + failureResources.size() + " resources.");
        }
        return output;
    }

    private void updateRebalanceStatus(final boolean hasFailure, HelixManager helixManager, ClusterDataCache cache, final ClusterStatusMonitor clusterStatusMonitor, String errorMessage) {
        BestPossibleStateCalcStage.asyncExecute(cache.getAsyncTasksThreadPool(), new Callable<Object>(){

            @Override
            public Object call() {
                try {
                    if (clusterStatusMonitor != null) {
                        clusterStatusMonitor.setRebalanceFailureGauge(hasFailure);
                    }
                }
                catch (Exception e) {
                    LogUtil.logError(logger, BestPossibleStateCalcStage.this._eventId, "Could not update cluster status!", e);
                }
                return null;
            }
        });
    }

    private void validateOfflineInstancesLimit(ClusterDataCache cache, HelixManager manager, ClusterStatusMonitor clusterStatusMonitor) {
        int offlineCount;
        int maxOfflineInstancesAllowed = cache.getClusterConfig().getMaxOfflineInstancesAllowed();
        if (maxOfflineInstancesAllowed >= 0 && (offlineCount = cache.getAllInstances().size() - cache.getEnabledLiveInstances().size()) > maxOfflineInstancesAllowed) {
            String errMsg = String.format("Offline Instances count %d greater than allowed count %d. Stop rebalance pipeline and pause the cluster %s", offlineCount, maxOfflineInstancesAllowed, cache.getClusterName());
            if (manager != null) {
                if (manager.getHelixDataAccessor().getProperty(manager.getHelixDataAccessor().keyBuilder().maintenance()) == null) {
                    manager.getClusterManagmentTool().enableMaintenanceMode(manager.getClusterName(), true, errMsg);
                }
            } else {
                LogUtil.logError(logger, this._eventId, "Failed to pause cluster, HelixManager is not set!");
            }
            if (!cache.isTaskCache()) {
                this.updateRebalanceStatus(true, manager, cache, clusterStatusMonitor, errMsg);
            }
        }
    }

    private boolean computeResourceBestPossibleState(ClusterEvent event, ClusterDataCache cache, CurrentStateOutput currentStateOutput, Resource resource, BestPossibleStateOutput output) {
        String resourceName = resource.getResourceName();
        LogUtil.logDebug(logger, this._eventId, "Processing resource:" + resourceName);
        IdealState idealState = cache.getIdealState(resourceName);
        if (idealState == null) {
            LogUtil.logInfo(logger, this._eventId, "resource:" + resourceName + " does not exist anymore");
            idealState = new IdealState(resourceName);
            idealState.setStateModelDefRef(resource.getStateModelDefRef());
        }
        Rebalancer rebalancer = this.getRebalancer(idealState, resourceName, cache.isMaintenanceModeEnabled());
        MappingCalculator mappingCalculator = this.getMappingCalculator(rebalancer, resourceName);
        if (rebalancer == null || mappingCalculator == null) {
            LogUtil.logError(logger, this._eventId, "Error computing assignment for resource " + resourceName + ". no rebalancer found. rebalancer: " + rebalancer + " mappingCaculator: " + mappingCalculator);
        }
        if (rebalancer != null && mappingCalculator != null) {
            if (rebalancer instanceof TaskRebalancer) {
                TaskRebalancer taskRebalancer = (TaskRebalancer)TaskRebalancer.class.cast(rebalancer);
                taskRebalancer.setClusterStatusMonitor((ClusterStatusMonitor)event.getAttribute(AttributeName.clusterStatusMonitor.name()));
            }
            ResourceAssignment partitionStateAssignment = null;
            try {
                HelixManager manager = (HelixManager)event.getAttribute(AttributeName.helixmanager.name());
                rebalancer.init(manager);
                idealState = rebalancer.computeNewIdealState(resourceName, idealState, currentStateOutput, cache);
                output.setPreferenceLists(resourceName, idealState.getPreferenceLists());
                partitionStateAssignment = mappingCalculator.computeBestPossiblePartitionState(cache, idealState, resource, currentStateOutput);
                for (Partition partition : resource.getPartitions()) {
                    Map<String, String> newStateMap = partitionStateAssignment.getReplicaMap(partition);
                    output.setState(resourceName, partition, newStateMap);
                }
                return this.checkBestPossibleStateCalculation(idealState);
            }
            catch (Exception e) {
                LogUtil.logError(logger, this._eventId, "Error computing assignment for resource " + resourceName + ". Skipping.", e);
                StringBuilder sb = new StringBuilder();
                sb.append(String.format("HelixManager is null : %s\n", event.getAttribute("helixmanager") == null));
                sb.append(String.format("Rebalancer is null : %s\n", rebalancer == null));
                sb.append(String.format("Calculated idealState is null : %s\n", idealState == null));
                sb.append(String.format("MappingCaculator is null : %s\n", mappingCalculator == null));
                sb.append(String.format("PartitionAssignment is null : %s\n", partitionStateAssignment == null));
                sb.append(String.format("Output is null : %s\n", output == null));
                LogUtil.logError(logger, this._eventId, sb.toString());
            }
        }
        return false;
    }

    private boolean checkBestPossibleStateCalculation(IdealState idealState) {
        if (idealState.getRebalanceMode() == IdealState.RebalanceMode.FULL_AUTO && !idealState.getReplicas().equals("0")) {
            Map<String, List<String>> preferenceLists = idealState.getPreferenceLists();
            if (preferenceLists == null || preferenceLists.isEmpty()) {
                return false;
            }
            int emptyListCount = 0;
            for (List<String> preferenceList : preferenceLists.values()) {
                if (!preferenceList.isEmpty()) continue;
                ++emptyListCount;
            }
            return emptyListCount != preferenceLists.values().size();
        }
        return true;
    }

    private Rebalancer getRebalancer(IdealState idealState, String resourceName, boolean isMaintenanceModeEnabled) {
        Rebalancer customizedRebalancer = null;
        String rebalancerClassName = idealState.getRebalancerClassName();
        if (rebalancerClassName != null) {
            if (logger.isDebugEnabled()) {
                LogUtil.logDebug(logger, this._eventId, "resource " + resourceName + " use idealStateRebalancer " + rebalancerClassName);
            }
            try {
                customizedRebalancer = (Rebalancer)Rebalancer.class.cast(HelixUtil.loadClass(this.getClass(), rebalancerClassName).newInstance());
            }
            catch (Exception e) {
                LogUtil.logError(logger, this._eventId, "Exception while invoking custom rebalancer class:" + rebalancerClassName, e);
            }
        }
        AbstractRebalancer rebalancer = null;
        switch (idealState.getRebalanceMode()) {
            case FULL_AUTO: {
                if (isMaintenanceModeEnabled) {
                    rebalancer = new MaintenanceRebalancer();
                    break;
                }
                if (customizedRebalancer != null) {
                    rebalancer = customizedRebalancer;
                    break;
                }
                rebalancer = new AutoRebalancer();
                break;
            }
            case SEMI_AUTO: {
                rebalancer = new SemiAutoRebalancer();
                break;
            }
            case CUSTOMIZED: {
                rebalancer = new CustomRebalancer();
                break;
            }
            case USER_DEFINED: 
            case TASK: {
                rebalancer = customizedRebalancer;
                break;
            }
            default: {
                LogUtil.logError(logger, this._eventId, "Fail to find the rebalancer, invalid rebalance mode " + (Object)((Object)idealState.getRebalanceMode()));
            }
        }
        return rebalancer;
    }

    private MappingCalculator getMappingCalculator(Rebalancer rebalancer, String resourceName) {
        MappingCalculator mappingCalculator = null;
        if (rebalancer != null) {
            try {
                mappingCalculator = (MappingCalculator)MappingCalculator.class.cast(rebalancer);
            }
            catch (ClassCastException e) {
                LogUtil.logWarn(logger, this._eventId, "Rebalancer does not have a mapping calculator, defaulting to SEMI_AUTO, resource: " + resourceName);
            }
        }
        if (mappingCalculator == null) {
            mappingCalculator = new SemiAutoRebalancer();
        }
        return mappingCalculator;
    }

    class ResourcePriority
    implements Comparable<ResourcePriority> {
        final Resource _resource;
        Long _priority = Long.MAX_VALUE;

        Resource getResource() {
            return this._resource;
        }

        public ResourcePriority(Resource resource, IdealState idealState, TaskDriver taskDriver) {
            JobContext jobContext;
            this._resource = resource;
            if (taskDriver != null && idealState != null && idealState.getRebalancerClassName() != null && idealState.getRebalancerClassName().equals(JobRebalancer.class.getName()) && (jobContext = taskDriver.getJobContext(resource.getResourceName())) != null && jobContext.getStartTime() != -1L) {
                this._priority = jobContext.getStartTime();
            }
        }

        @Override
        public int compareTo(ResourcePriority otherJob) {
            return this._priority.compareTo(otherJob._priority);
        }
    }
}

