/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.raptor.metadata;

import com.facebook.airlift.log.Logger;
import com.facebook.airlift.stats.CounterStat;
import com.facebook.presto.raptor.NodeSupplier;
import com.facebook.presto.raptor.RaptorColumnHandle;
import com.facebook.presto.raptor.RaptorErrorCode;
import com.facebook.presto.raptor.metadata.AssignmentLimiter;
import com.facebook.presto.raptor.metadata.BucketNode;
import com.facebook.presto.raptor.metadata.BucketReassigner;
import com.facebook.presto.raptor.metadata.BucketShards;
import com.facebook.presto.raptor.metadata.ColumnInfo;
import com.facebook.presto.raptor.metadata.DeltaInfoPair;
import com.facebook.presto.raptor.metadata.Distribution;
import com.facebook.presto.raptor.metadata.ForMetadata;
import com.facebook.presto.raptor.metadata.IndexInserter;
import com.facebook.presto.raptor.metadata.MetadataConfig;
import com.facebook.presto.raptor.metadata.MetadataDao;
import com.facebook.presto.raptor.metadata.NodeSize;
import com.facebook.presto.raptor.metadata.ShardDao;
import com.facebook.presto.raptor.metadata.ShardInfo;
import com.facebook.presto.raptor.metadata.ShardIterator;
import com.facebook.presto.raptor.metadata.ShardManager;
import com.facebook.presto.raptor.metadata.ShardMetadata;
import com.facebook.presto.raptor.metadata.ShardsAndIndexDeleter;
import com.facebook.presto.raptor.metadata.ShardsAndIndexUpdater;
import com.facebook.presto.raptor.storage.ColumnIndexStatsUtils;
import com.facebook.presto.raptor.storage.organization.ShardOrganizerDao;
import com.facebook.presto.raptor.util.ArrayUtil;
import com.facebook.presto.raptor.util.DaoSupplier;
import com.facebook.presto.raptor.util.DatabaseUtil;
import com.facebook.presto.raptor.util.UuidUtil;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.Node;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.predicate.TupleDomain;
import com.facebook.presto.spi.type.Type;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Ticker;
import com.google.common.base.Verify;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.UncheckedExecutionException;
import io.airlift.units.Duration;
import java.sql.Connection;
import java.sql.JDBCType;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.StringJoiner;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.h2.jdbc.JdbcConnection;
import org.skife.jdbi.v2.Handle;
import org.skife.jdbi.v2.IDBI;
import org.skife.jdbi.v2.Query;
import org.skife.jdbi.v2.ResultIterator;
import org.skife.jdbi.v2.exceptions.DBIException;
import org.skife.jdbi.v2.tweak.HandleConsumer;
import org.skife.jdbi.v2.tweak.ResultSetMapper;
import org.skife.jdbi.v2.util.ByteArrayMapper;
import org.weakref.jmx.Flatten;
import org.weakref.jmx.Managed;

public class DatabaseShardManager
implements ShardManager {
    private static final Logger log = Logger.get(DatabaseShardManager.class);
    private static final String INDEX_TABLE_PREFIX = "x_shards_t";
    private static final int MAX_ADD_COLUMN_ATTEMPTS = 100;
    private final DeltaDeleteStats deltaDeleteStats = new DeltaDeleteStats();
    private final IDBI dbi;
    private final DaoSupplier<ShardDao> shardDaoSupplier;
    private final ShardDao dao;
    private final NodeSupplier nodeSupplier;
    private final AssignmentLimiter assignmentLimiter;
    private final Ticker ticker;
    private final Duration startupGracePeriod;
    private final long startTime;
    private final LoadingCache<String, Integer> nodeIdCache = CacheBuilder.newBuilder().maximumSize(10000L).build(CacheLoader.from(this::loadNodeId));
    private final LoadingCache<Long, List<String>> bucketAssignmentsCache = CacheBuilder.newBuilder().expireAfterWrite(1L, TimeUnit.SECONDS).build(CacheLoader.from(this::loadBucketAssignments));

    @Inject
    public DatabaseShardManager(@ForMetadata IDBI dbi, DaoSupplier<ShardDao> shardDaoSupplier, NodeSupplier nodeSupplier, AssignmentLimiter assignmentLimiter, Ticker ticker, MetadataConfig config) {
        this(dbi, shardDaoSupplier, nodeSupplier, assignmentLimiter, ticker, config.getStartupGracePeriod());
    }

    public DatabaseShardManager(IDBI dbi, DaoSupplier<ShardDao> shardDaoSupplier, NodeSupplier nodeSupplier, AssignmentLimiter assignmentLimiter, Ticker ticker, Duration startupGracePeriod) {
        this.dbi = Objects.requireNonNull(dbi, "dbi is null");
        this.shardDaoSupplier = Objects.requireNonNull(shardDaoSupplier, "shardDaoSupplier is null");
        this.dao = shardDaoSupplier.onDemand();
        this.nodeSupplier = Objects.requireNonNull(nodeSupplier, "nodeSupplier is null");
        this.assignmentLimiter = Objects.requireNonNull(assignmentLimiter, "assignmentLimiter is null");
        this.ticker = Objects.requireNonNull(ticker, "ticker is null");
        this.startupGracePeriod = Objects.requireNonNull(startupGracePeriod, "startupGracePeriod is null");
        this.startTime = ticker.read();
    }

    @Override
    public void createTable(long tableId, List<ColumnInfo> columns, boolean bucketed, OptionalLong temporalColumnId, boolean tableSupportsDeltaDelete) {
        StringJoiner tableColumns = new StringJoiner(",\n  ", "  ", ",\n").setEmptyValue("");
        for (ColumnInfo column : columns) {
            String columnType = DatabaseShardManager.sqlColumnType(column.getType());
            if (columnType == null) continue;
            tableColumns.add(DatabaseShardManager.minColumn(column.getColumnId()) + " " + columnType);
            tableColumns.add(DatabaseShardManager.maxColumn(column.getColumnId()) + " " + columnType);
        }
        StringJoiner coveringIndexColumns = new StringJoiner(", ");
        temporalColumnId.ifPresent(id -> coveringIndexColumns.add(DatabaseShardManager.maxColumn(id)));
        temporalColumnId.ifPresent(id -> coveringIndexColumns.add(DatabaseShardManager.minColumn(id)));
        if (bucketed) {
            coveringIndexColumns.add("bucket_number");
        } else {
            coveringIndexColumns.add("node_ids");
        }
        coveringIndexColumns.add("shard_id").add("shard_uuid");
        String sql = "CREATE TABLE " + DatabaseShardManager.shardIndexTable(tableId) + " (\n  shard_id BIGINT NOT NULL,\n  shard_uuid BINARY(16) NOT NULL,\n" + (tableSupportsDeltaDelete ? "  delta_shard_uuid BINARY(16) DEFAULT NULL,\n" : "") + (bucketed ? "  bucket_number INT NOT NULL\n," : "  node_ids VARBINARY(128) NOT NULL,\n") + tableColumns + (bucketed ? "  PRIMARY KEY (bucket_number, shard_uuid),\n" : "  PRIMARY KEY (node_ids, shard_uuid),\n") + "  UNIQUE (shard_id),\n  UNIQUE (shard_uuid),\n  UNIQUE (" + coveringIndexColumns + ")\n)";
        try (Handle handle = this.dbi.open();){
            handle.execute(sql, new Object[0]);
        }
        catch (DBIException e) {
            throw DatabaseUtil.metadataError(e);
        }
    }

    @Override
    public void dropTable(long tableId) {
        DatabaseUtil.runTransaction(this.dbi, (handle, status) -> {
            DatabaseShardManager.lockTable(handle, tableId);
            ShardDao shardDao = this.shardDaoSupplier.attach(handle);
            shardDao.insertDeletedShards(tableId);
            shardDao.dropShardNodes(tableId);
            shardDao.dropShards(tableId);
            ((ShardOrganizerDao)handle.attach(ShardOrganizerDao.class)).dropOrganizerJobs(tableId);
            MetadataDao dao = (MetadataDao)handle.attach(MetadataDao.class);
            dao.dropColumns(tableId);
            dao.dropTable(tableId);
            return null;
        });
        try (Handle handle2 = this.dbi.open();){
            handle2.execute("DROP TABLE " + DatabaseShardManager.shardIndexTable(tableId), new Object[0]);
        }
        catch (DBIException e) {
            log.warn((Throwable)e, "Failed to drop index table %s", new Object[]{DatabaseShardManager.shardIndexTable(tableId)});
        }
    }

    @Override
    public void addColumn(long tableId, ColumnInfo column) {
        String columnType = DatabaseShardManager.sqlColumnType(column.getType());
        if (columnType == null) {
            return;
        }
        String sql = String.format("ALTER TABLE %s ADD COLUMN (%s %s, %s %s)", DatabaseShardManager.shardIndexTable(tableId), DatabaseShardManager.minColumn(column.getColumnId()), columnType, DatabaseShardManager.maxColumn(column.getColumnId()), columnType);
        int attempts = 0;
        while (true) {
            ++attempts;
            try {
                Handle handle = this.dbi.open();
                Throwable throwable = null;
                try {
                    handle.execute(sql, new Object[0]);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (handle == null) continue;
                    if (throwable != null) {
                        try {
                            handle.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    handle.close();
                }
            }
            catch (DBIException e) {
                if (DatabaseUtil.isSyntaxOrAccessError((Exception)((Object)e))) {
                    return;
                }
                if (attempts >= 100) {
                    throw DatabaseUtil.metadataError(e);
                }
                log.warn((Throwable)e, "Failed to alter table on attempt %s, will retry. SQL: %s", new Object[]{attempts, sql});
                try {
                    TimeUnit.SECONDS.sleep(3L);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw DatabaseUtil.metadataError(ie);
                }
            }
        }
    }

    @Override
    public void commitShards(long transactionId, long tableId, List<ColumnInfo> columns, Collection<ShardInfo> shards, Optional<String> externalBatchId, long updateTime) {
        if (externalBatchId.isPresent() && this.dao.externalBatchExists(externalBatchId.get())) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_EXTERNAL_BATCH_ALREADY_EXISTS, "External batch already exists: " + externalBatchId.get());
        }
        Map<String, Integer> nodeIds = this.toNodeIdMap(shards);
        this.runCommit(transactionId, handle -> {
            externalBatchId.ifPresent(this.shardDaoSupplier.attach(handle)::insertExternalBatch);
            DatabaseShardManager.lockTable(handle, tableId);
            DatabaseShardManager.insertShardsAndIndex(tableId, columns, shards, nodeIds, handle);
            ShardStats stats = DatabaseShardManager.shardStats(shards);
            this.updateStatsAndVersion(handle, tableId, shards.size(), 0L, stats.getRowCount(), stats.getCompressedSize(), stats.getUncompressedSize(), OptionalLong.of(updateTime));
        });
    }

    @Override
    public void replaceShardUuids(long transactionId, long tableId, List<ColumnInfo> columns, Set<UUID> oldShardUuids, Collection<ShardInfo> newShards, OptionalLong updateTime) {
        Map<String, Integer> nodeIds = this.toNodeIdMap(newShards);
        this.runCommit(transactionId, handle -> {
            DatabaseShardManager.lockTable(handle, tableId);
            if (!updateTime.isPresent() && ((MetadataDao)handle.attach(MetadataDao.class)).isMaintenanceBlockedLocked(tableId)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.TRANSACTION_CONFLICT, "Maintenance is blocked for table");
            }
            ShardStats newStats = DatabaseShardManager.shardStats(newShards);
            long rowCount = newStats.getRowCount();
            long compressedSize = newStats.getCompressedSize();
            long uncompressedSize = newStats.getUncompressedSize();
            for (List shards : Iterables.partition((Iterable)newShards, (int)1000)) {
                DatabaseShardManager.insertShardsAndIndex(tableId, columns, shards, nodeIds, handle);
            }
            for (List uuids : Iterables.partition((Iterable)oldShardUuids, (int)1000)) {
                ShardStats stats = this.deleteShardsAndIndex(tableId, (Set<UUID>)ImmutableSet.copyOf((Collection)uuids), handle);
                rowCount -= stats.getRowCount();
                compressedSize -= stats.getCompressedSize();
                uncompressedSize -= stats.getUncompressedSize();
            }
            long shardCount = newShards.size() - oldShardUuids.size();
            if (!oldShardUuids.isEmpty() || !newShards.isEmpty()) {
                MetadataDao metadata = (MetadataDao)handle.attach(MetadataDao.class);
                metadata.updateTableStats(tableId, shardCount, 0L, rowCount, compressedSize, uncompressedSize);
                updateTime.ifPresent(time -> metadata.updateTableVersion(tableId, time));
            }
        });
    }

    private void runCommit(long transactionId, HandleConsumer callback) {
        int maxAttempts = 5;
        for (int attempt = 1; attempt <= maxAttempts; ++attempt) {
            try {
                this.dbi.useTransaction((handle, status) -> {
                    ShardDao dao = this.shardDaoSupplier.attach(handle);
                    if (DatabaseShardManager.commitTransaction(dao, transactionId)) {
                        callback.useHandle(handle);
                        dao.deleteCreatedShards(transactionId);
                    }
                });
                return;
            }
            catch (DBIException e) {
                if (DatabaseUtil.isTransactionCacheFullError((Exception)((Object)e))) {
                    throw DatabaseUtil.metadataError(e, "Transaction too large");
                }
                if (e.getCause() != null) {
                    Throwables.throwIfInstanceOf((Throwable)e.getCause(), PrestoException.class);
                }
                if (attempt == maxAttempts) {
                    throw DatabaseUtil.metadataError(e);
                }
                log.warn((Throwable)e, "Failed to commit shards on attempt %d, will retry.", new Object[]{attempt});
                try {
                    TimeUnit.SECONDS.sleep(Math.multiplyExact(attempt, 2));
                    continue;
                }
                catch (InterruptedException ie) {
                    throw DatabaseUtil.metadataError(ie);
                }
            }
        }
    }

    private static boolean commitTransaction(ShardDao dao, long transactionId) {
        if (dao.finalizeTransaction(transactionId, true) != 1) {
            if (Boolean.TRUE.equals(dao.transactionSuccessful(transactionId))) {
                return false;
            }
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.TRANSACTION_CONFLICT, "Transaction commit failed. Please retry the operation.");
        }
        return true;
    }

    private ShardStats deleteShardsAndIndex(long tableId, Set<UUID> shardUuids, Handle handle) throws SQLException {
        String args = Joiner.on((String)",").join(Collections.nCopies(shardUuids.size(), "?"));
        ImmutableSet.Builder shardIdSet = ImmutableSet.builder();
        long rowCount = 0L;
        long compressedSize = 0L;
        long uncompressedSize = 0L;
        String selectShards = String.format("SELECT shard_id, row_count, compressed_size, uncompressed_size\nFROM shards\nWHERE shard_uuid IN (%s)", args);
        try (PreparedStatement statement = handle.getConnection().prepareStatement(selectShards);){
            DatabaseShardManager.bindUuids(statement, shardUuids);
            try (ResultSet rs = statement.executeQuery();){
                while (rs.next()) {
                    shardIdSet.add((Object)rs.getLong("shard_id"));
                    rowCount += rs.getLong("row_count");
                    compressedSize += rs.getLong("compressed_size");
                    uncompressedSize += rs.getLong("uncompressed_size");
                }
            }
        }
        ImmutableSet shardIds = shardIdSet.build();
        if (shardIds.size() != shardUuids.size()) {
            throw DatabaseShardManager.transactionConflict();
        }
        ShardDao dao = this.shardDaoSupplier.attach(handle);
        dao.insertDeletedShards(shardUuids);
        String where = " WHERE shard_id IN (" + args + ")";
        String deleteFromShardNodes = "DELETE FROM shard_nodes " + where;
        String deleteFromShards = "DELETE FROM shards " + where;
        String deleteFromShardIndex = "DELETE FROM " + DatabaseShardManager.shardIndexTable(tableId) + where;
        try (PreparedStatement statement = handle.getConnection().prepareStatement(deleteFromShardNodes);){
            DatabaseShardManager.bindLongs(statement, (Iterable<Long>)shardIds);
            statement.executeUpdate();
        }
        for (String sql : Arrays.asList(deleteFromShards, deleteFromShardIndex)) {
            PreparedStatement statement = handle.getConnection().prepareStatement(sql);
            Throwable throwable = null;
            try {
                DatabaseShardManager.bindLongs(statement, (Iterable<Long>)shardIds);
                if (statement.executeUpdate() == shardIds.size()) continue;
                throw DatabaseShardManager.transactionConflict();
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (statement == null) continue;
                if (throwable != null) {
                    try {
                        statement.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                statement.close();
            }
        }
        return new ShardStats(rowCount, compressedSize, uncompressedSize);
    }

    private static void bindUuids(PreparedStatement statement, Iterable<UUID> uuids) throws SQLException {
        int i = 1;
        for (UUID uuid : uuids) {
            statement.setBytes(i, UuidUtil.uuidToBytes(uuid));
            ++i;
        }
    }

    private static void bindLongs(PreparedStatement statement, Iterable<Long> values) throws SQLException {
        int i = 1;
        for (long value : values) {
            statement.setLong(i, value);
            ++i;
        }
    }

    private static void insertShardsAndIndex(long tableId, List<ColumnInfo> columns, Collection<ShardInfo> shards, Map<String, Integer> nodeIds, Handle handle) throws SQLException {
        if (shards.isEmpty()) {
            return;
        }
        boolean bucketed = shards.iterator().next().getBucketNumber().isPresent();
        Connection connection = handle.getConnection();
        try (IndexInserter indexInserter = new IndexInserter(connection, tableId, columns);){
            for (List batch : Iterables.partition(shards, (int)DatabaseShardManager.batchSize(connection))) {
                List<Long> shardIds = DatabaseShardManager.insertShards(connection, tableId, batch);
                if (!bucketed) {
                    DatabaseShardManager.insertShardNodes(connection, nodeIds, shardIds, batch);
                }
                for (int i = 0; i < batch.size(); ++i) {
                    ShardInfo shard = (ShardInfo)batch.get(i);
                    Set<Integer> shardNodes = shard.getNodeIdentifiers().stream().map(nodeIds::get).collect(Collectors.toSet());
                    indexInserter.insert(shardIds.get(i), shard.getShardUuid(), shard.getBucketNumber(), shardNodes, shard.getColumnStats());
                }
                indexInserter.execute();
            }
        }
    }

    private static int batchSize(Connection connection) {
        return connection instanceof JdbcConnection ? 1 : 1000;
    }

    @Override
    public void replaceShardUuids(long transactionId, long tableId, List<ColumnInfo> columns, Map<UUID, Optional<UUID>> oldShardAndDeltaUuids, Collection<ShardInfo> newShards, OptionalLong updateTime, boolean tableSupportsDeltaDelete) {
        Map<String, Integer> nodeIds = this.toNodeIdMap(newShards);
        this.runCommit(transactionId, handle -> {
            DatabaseShardManager.lockTable(handle, tableId);
            if (!updateTime.isPresent() && ((MetadataDao)handle.attach(MetadataDao.class)).isMaintenanceBlockedLocked(tableId)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.TRANSACTION_CONFLICT, "Maintenance is blocked for table");
            }
            DatabaseShardManager.insertShardsAndIndex(tableId, columns, newShards, nodeIds, handle, false);
            ShardStats newStats = DatabaseShardManager.shardStats(newShards);
            long rowCount = newStats.getRowCount();
            long compressedSize = newStats.getCompressedSize();
            long uncompressedSize = newStats.getUncompressedSize();
            Set oldDeltaUuidSet = (Set)oldShardAndDeltaUuids.values().stream().filter(Optional::isPresent).map(Optional::get).collect(ImmutableSet.toImmutableSet());
            ShardStats stats = this.deleteShardsAndIndex(tableId, oldShardAndDeltaUuids, oldDeltaUuidSet, handle, tableSupportsDeltaDelete);
            rowCount -= stats.getRowCount();
            compressedSize -= stats.getCompressedSize();
            uncompressedSize -= stats.getUncompressedSize();
            long deltaCountChange = -oldDeltaUuidSet.size();
            long shardCountChange = newShards.size() - oldShardAndDeltaUuids.size();
            if (!oldShardAndDeltaUuids.isEmpty() || !newShards.isEmpty()) {
                this.updateStatsAndVersion(handle, tableId, shardCountChange, deltaCountChange, rowCount, compressedSize, uncompressedSize, updateTime);
            }
        });
    }

    @Override
    public void replaceDeltaUuids(long transactionId, long tableId, List<ColumnInfo> columns, Map<UUID, DeltaInfoPair> shardMap, OptionalLong updateTime) {
        this.runCommit(transactionId, handle -> {
            DatabaseShardManager.lockTable(handle, tableId);
            HashSet<ShardInfo> newDeltas = new HashSet<ShardInfo>();
            HashSet<UUID> oldDeltaUuids = new HashSet<UUID>();
            HashMap<UUID, Optional<UUID>> shardsMapToDelete = new HashMap<UUID, Optional<UUID>>();
            HashMap<UUID, DeltaUuidPair> shardsToUpdate = new HashMap<UUID, DeltaUuidPair>();
            for (Map.Entry entry : shardMap.entrySet()) {
                UUID uuid = (UUID)entry.getKey();
                DeltaInfoPair deltaInfoPair = (DeltaInfoPair)entry.getValue();
                if (deltaInfoPair.getNewDeltaDeleteShard().isPresent()) {
                    newDeltas.add(deltaInfoPair.getNewDeltaDeleteShard().get());
                    shardsToUpdate.put(uuid, new DeltaUuidPair(deltaInfoPair.getOldDeltaDeleteShard(), deltaInfoPair.getNewDeltaDeleteShard().get().getShardUuid()));
                } else {
                    shardsMapToDelete.put(uuid, deltaInfoPair.getOldDeltaDeleteShard());
                }
                if (!deltaInfoPair.getOldDeltaDeleteShard().isPresent()) continue;
                oldDeltaUuids.add(deltaInfoPair.getOldDeltaDeleteShard().get());
            }
            Map<String, Integer> nodeIds = this.toNodeIdMap(newDeltas);
            DatabaseShardManager.insertShardsAndIndex(tableId, columns, newDeltas, nodeIds, handle, true);
            ShardStats newStats = DatabaseShardManager.shardStats(newDeltas);
            long rowCount = -newStats.getRowCount();
            long compressedSize = newStats.getCompressedSize();
            long uncompressedSize = newStats.getUncompressedSize();
            ShardStats stats = this.deleteShardsAndIndex(tableId, shardsMapToDelete, oldDeltaUuids, handle, true);
            rowCount -= stats.getRowCount();
            compressedSize -= stats.getCompressedSize();
            uncompressedSize -= stats.getUncompressedSize();
            this.updateShardsAndIndex(tableId, shardsToUpdate, handle);
            int shardCountChange = -shardsMapToDelete.size();
            int deltaCountChange = newDeltas.size() - oldDeltaUuids.size();
            if (!newDeltas.isEmpty() || !oldDeltaUuids.isEmpty() || shardsToUpdate.isEmpty() || !shardsMapToDelete.isEmpty()) {
                this.updateStatsAndVersion(handle, tableId, shardCountChange, deltaCountChange, rowCount, compressedSize, uncompressedSize, updateTime);
            }
            this.deltaDeleteStats.deletedShards.update((long)shardsMapToDelete.size());
            this.deltaDeleteStats.updatedShards.update((long)shardsToUpdate.size());
        });
    }

    private void updateStatsAndVersion(Handle handle, long tableId, long shardCountChange, long deltaCountChange, long rowCount, long compressedSize, long uncompressedSize, OptionalLong updateTime) {
        MetadataDao metadata = (MetadataDao)handle.attach(MetadataDao.class);
        metadata.updateTableStats(tableId, shardCountChange, deltaCountChange, rowCount, compressedSize, uncompressedSize);
        updateTime.ifPresent(time -> metadata.updateTableVersion(tableId, time));
    }

    private ShardStats deleteShardsAndIndex(long tableId, Map<UUID, Optional<UUID>> oldShardUuidsMap, Set<UUID> oldDeltaUuidSet, Handle handle, boolean tableSupportsDeltaDelete) throws SQLException {
        if (tableSupportsDeltaDelete) {
            ShardStats shardStats = this.deleteShardsAndIndexWithDelta(tableId, oldShardUuidsMap, handle);
            long rowCount = shardStats.getRowCount();
            long compressedSize = shardStats.getCompressedSize();
            long uncompressedSize = shardStats.getUncompressedSize();
            ShardStats deltaStats = this.deleteShardsAndIndexSimple(tableId, oldDeltaUuidSet, handle, true);
            return new ShardStats(rowCount -= deltaStats.getRowCount(), compressedSize += deltaStats.getCompressedSize(), uncompressedSize += deltaStats.getUncompressedSize());
        }
        return this.deleteShardsAndIndexSimple(tableId, oldShardUuidsMap.keySet(), handle, false);
    }

    private ShardStats deleteShardsAndIndexSimple(long tableId, Set<UUID> shardUuids, Handle handle, boolean isDelta) throws SQLException {
        if (shardUuids.isEmpty()) {
            return new ShardStats(0L, 0L, 0L);
        }
        long rowCount = 0L;
        long compressedSize = 0L;
        long uncompressedSize = 0L;
        for (List uuids : Iterables.partition(shardUuids, (int)1000)) {
            String args = Joiner.on((String)",").join(Collections.nCopies(uuids.size(), "?"));
            ImmutableSet.Builder shardIdSet = ImmutableSet.builder();
            String selectShards = String.format("SELECT shard_id, row_count, compressed_size, uncompressed_size\nFROM shards\nWHERE shard_uuid IN (%s)", args);
            try (PreparedStatement statement = handle.getConnection().prepareStatement(selectShards);){
                DatabaseShardManager.bindUuids(statement, uuids);
                try (ResultSet rs = statement.executeQuery();){
                    while (rs.next()) {
                        shardIdSet.add((Object)rs.getLong("shard_id"));
                        rowCount += rs.getLong("row_count");
                        compressedSize += rs.getLong("compressed_size");
                        uncompressedSize += rs.getLong("uncompressed_size");
                    }
                }
            }
            ImmutableSet shardIds = shardIdSet.build();
            if (shardIds.size() != uuids.size()) {
                throw DatabaseShardManager.transactionConflict();
            }
            ShardDao dao = this.shardDaoSupplier.attach(handle);
            dao.insertDeletedShards(uuids);
            String where = " WHERE shard_id IN (" + args + ")";
            String deleteFromShardNodes = "DELETE FROM shard_nodes " + where;
            String deleteFromShards = "DELETE FROM shards " + where;
            String deleteFromShardIndex = "DELETE FROM " + DatabaseShardManager.shardIndexTable(tableId) + where;
            try (PreparedStatement statement = handle.getConnection().prepareStatement(deleteFromShardNodes);){
                DatabaseShardManager.bindLongs(statement, (Iterable<Long>)shardIds);
                statement.executeUpdate();
            }
            Iterator iterator = (isDelta ? ImmutableList.of((Object)deleteFromShards) : Arrays.asList(deleteFromShards, deleteFromShardIndex)).iterator();
            while (iterator.hasNext()) {
                String sql = (String)iterator.next();
                PreparedStatement statement = handle.getConnection().prepareStatement(sql);
                Throwable throwable = null;
                try {
                    DatabaseShardManager.bindLongs(statement, (Iterable<Long>)shardIds);
                    if (statement.executeUpdate() == shardIds.size()) continue;
                    throw DatabaseShardManager.transactionConflict();
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (statement == null) continue;
                    if (throwable != null) {
                        try {
                            statement.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    statement.close();
                }
            }
        }
        return new ShardStats(rowCount, compressedSize, uncompressedSize);
    }

    private ShardStats deleteShardsAndIndexWithDelta(long tableId, Map<UUID, Optional<UUID>> oldShardUuidsMap, Handle handle) throws SQLException {
        if (oldShardUuidsMap.isEmpty()) {
            return new ShardStats(0L, 0L, 0L);
        }
        String args = Joiner.on((String)",").join(Collections.nCopies(oldShardUuidsMap.size(), "?"));
        ImmutableMap.Builder shardUuidToIdBuilder = ImmutableMap.builder();
        long rowCount = 0L;
        long compressedSize = 0L;
        long uncompressedSize = 0L;
        String selectShards = String.format("SELECT shard_id, shard_uuid, row_count, compressed_size, uncompressed_size\nFROM shards\nWHERE shard_uuid IN (%s)", args);
        try (PreparedStatement statement = handle.getConnection().prepareStatement(selectShards);){
            DatabaseShardManager.bindUuids(statement, oldShardUuidsMap.keySet());
            try (ResultSet rs = statement.executeQuery();){
                while (rs.next()) {
                    shardUuidToIdBuilder.put((Object)UuidUtil.uuidFromBytes(rs.getBytes("shard_uuid")), (Object)rs.getLong("shard_id"));
                    rowCount += rs.getLong("row_count");
                    compressedSize += rs.getLong("compressed_size");
                    uncompressedSize += rs.getLong("uncompressed_size");
                }
            }
        }
        ImmutableMap shardUuidToId = shardUuidToIdBuilder.build();
        if (shardUuidToId.size() != oldShardUuidsMap.size()) {
            throw DatabaseShardManager.transactionConflict();
        }
        ShardDao dao = this.shardDaoSupplier.attach(handle);
        dao.insertDeletedShards(oldShardUuidsMap.keySet());
        String where = " WHERE shard_id IN (" + args + ")";
        String deleteFromShardNodes = "DELETE FROM shard_nodes " + where;
        try (PreparedStatement statement = handle.getConnection().prepareStatement(deleteFromShardNodes);){
            DatabaseShardManager.bindLongs(statement, shardUuidToId.values());
            statement.executeUpdate();
        }
        Connection connection = handle.getConnection();
        int updatedCount = 0;
        try (ShardsAndIndexDeleter shardsAndIndexDeleter = new ShardsAndIndexDeleter(connection, tableId);){
            for (List batch : Iterables.partition(oldShardUuidsMap.keySet(), (int)DatabaseShardManager.batchSize(connection))) {
                for (UUID uuid : batch) {
                    Optional<UUID> deltaUuid = oldShardUuidsMap.get(uuid);
                    shardsAndIndexDeleter.delete((Long)shardUuidToId.get(uuid), deltaUuid);
                }
                updatedCount += shardsAndIndexDeleter.execute();
            }
        }
        if (updatedCount != oldShardUuidsMap.size()) {
            throw DatabaseShardManager.transactionConflict();
        }
        return new ShardStats(rowCount, compressedSize, uncompressedSize);
    }

    private static void insertShardsAndIndex(long tableId, List<ColumnInfo> columns, Collection<ShardInfo> shards, Map<String, Integer> nodeIds, Handle handle, boolean isDelta) throws SQLException {
        if (shards.isEmpty()) {
            return;
        }
        boolean bucketed = shards.iterator().next().getBucketNumber().isPresent();
        Connection connection = handle.getConnection();
        try (IndexInserter indexInserter = new IndexInserter(connection, tableId, columns);){
            for (List batch : Iterables.partition(shards, (int)DatabaseShardManager.batchSize(connection))) {
                List<Long> shardIds = DatabaseShardManager.insertShards(connection, tableId, batch, isDelta);
                if (!bucketed) {
                    DatabaseShardManager.insertShardNodes(connection, nodeIds, shardIds, batch);
                }
                if (isDelta) continue;
                for (int i = 0; i < batch.size(); ++i) {
                    ShardInfo shard = (ShardInfo)batch.get(i);
                    Set<Integer> shardNodes = shard.getNodeIdentifiers().stream().map(nodeIds::get).collect(Collectors.toSet());
                    indexInserter.insert(shardIds.get(i), shard.getShardUuid(), shard.getBucketNumber(), shardNodes, shard.getColumnStats());
                }
                indexInserter.execute();
            }
        }
    }

    private void updateShardsAndIndex(long tableId, Map<UUID, DeltaUuidPair> toUpdateShard, Handle handle) throws SQLException {
        Throwable throwable;
        if (toUpdateShard.isEmpty()) {
            return;
        }
        String args = Joiner.on((String)",").join(Collections.nCopies(toUpdateShard.size(), "?"));
        ImmutableMap.Builder shardMapBuilder = ImmutableMap.builder();
        String selectShards = String.format("SELECT shard_id, shard_uuid\nFROM shards\nWHERE shard_uuid IN (%s)", args);
        try (PreparedStatement statement = handle.getConnection().prepareStatement(selectShards);){
            DatabaseShardManager.bindUuids(statement, toUpdateShard.keySet());
            throwable = null;
            try (ResultSet rs = statement.executeQuery();){
                while (rs.next()) {
                    shardMapBuilder.put((Object)rs.getLong("shard_id"), (Object)UuidUtil.uuidFromBytes(rs.getBytes("shard_uuid")));
                }
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
        }
        ImmutableMap shardIdToUuid = shardMapBuilder.build();
        if (toUpdateShard.size() != shardIdToUuid.size()) {
            throw DatabaseShardManager.transactionConflict();
        }
        int updatedCount = 0;
        throwable = null;
        try (ShardsAndIndexUpdater shardsAndIndexUpdater = new ShardsAndIndexUpdater(handle.getConnection(), tableId);){
            for (List batch : Iterables.partition(shardIdToUuid.keySet(), (int)DatabaseShardManager.batchSize(handle.getConnection()))) {
                Iterator iterator = batch.iterator();
                while (iterator.hasNext()) {
                    long shardId = (Long)iterator.next();
                    shardsAndIndexUpdater.update(shardId, toUpdateShard.get(shardIdToUuid.get(shardId)).getOldDeltaUuid(), toUpdateShard.get(shardIdToUuid.get(shardId)).getNewDeltaUuid());
                }
                updatedCount += shardsAndIndexUpdater.execute();
            }
        }
        catch (Throwable throwable3) {
            throwable = throwable3;
            throw throwable3;
        }
        if (updatedCount != shardIdToUuid.size()) {
            log.error("updatedCount is not equal to shardIdToUuid size");
            throw DatabaseShardManager.transactionConflict();
        }
    }

    private static List<Long> insertShards(Connection connection, long tableId, List<ShardInfo> shards, boolean isDelta) throws SQLException {
        String sql = "INSERT INTO shards (shard_uuid, table_id, is_delta, delta_uuid, create_time, row_count, compressed_size, uncompressed_size, xxhash64, bucket_number)\nVALUES (?, ?, ?, ?, CURRENT_TIMESTAMP, ?, ?, ?, ?, ?)";
        try (PreparedStatement statement = connection.prepareStatement(sql, 1);){
            for (ShardInfo shard : shards) {
                statement.setBytes(1, UuidUtil.uuidToBytes(shard.getShardUuid()));
                statement.setLong(2, tableId);
                statement.setBoolean(3, isDelta);
                statement.setNull(4, -2);
                statement.setLong(5, shard.getRowCount());
                statement.setLong(6, shard.getCompressedSize());
                statement.setLong(7, shard.getUncompressedSize());
                statement.setLong(8, shard.getXxhash64());
                DatabaseUtil.bindOptionalInt(statement, 9, shard.getBucketNumber());
                statement.addBatch();
            }
            statement.executeBatch();
            ImmutableList.Builder builder = ImmutableList.builder();
            try (ResultSet keys = statement.getGeneratedKeys();){
                while (keys.next()) {
                    builder.add((Object)keys.getLong(1));
                }
            }
            ImmutableList shardIds = builder.build();
            if (shardIds.size() != shards.size()) {
                throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Wrong number of generated keys for inserted shards");
            }
            var10_10 = shardIds;
            return var10_10;
        }
    }

    private Map<String, Integer> toNodeIdMap(Collection<ShardInfo> shards) {
        Set identifiers = shards.stream().map(ShardInfo::getNodeIdentifiers).flatMap(Collection::stream).collect(Collectors.toSet());
        return Maps.toMap(identifiers, this::getOrCreateNodeId);
    }

    @Override
    public ShardMetadata getShard(UUID shardUuid) {
        return this.dao.getShard(shardUuid);
    }

    @Override
    public Set<ShardMetadata> getNodeShardsAndDeltas(String nodeIdentifier) {
        return this.dao.getNodeShardsAndDeltas(nodeIdentifier, null);
    }

    @Override
    public Set<ShardMetadata> getNodeShards(String nodeIdentifier) {
        return this.dao.getNodeShards(nodeIdentifier, null);
    }

    @Override
    public Set<ShardMetadata> getNodeShards(String nodeIdentifier, long tableId) {
        return this.dao.getNodeShards(nodeIdentifier, tableId);
    }

    @Override
    public ResultIterator<BucketShards> getShardNodes(long tableId, TupleDomain<RaptorColumnHandle> effectivePredicate, boolean tableSupportsDeltaDelete) {
        return new ShardIterator(tableId, false, tableSupportsDeltaDelete, Optional.empty(), effectivePredicate, this.dbi);
    }

    @Override
    public ResultIterator<BucketShards> getShardNodesBucketed(long tableId, boolean merged, List<String> bucketToNode, TupleDomain<RaptorColumnHandle> effectivePredicate, boolean tableSupportsDeltaDelete) {
        return new ShardIterator(tableId, merged, tableSupportsDeltaDelete, Optional.of(bucketToNode), effectivePredicate, this.dbi);
    }

    @Override
    public void replaceShardAssignment(long tableId, UUID shardUuid, Optional<UUID> deltaUuid, String nodeIdentifier, boolean gracePeriod) {
        if (gracePeriod && this.nanosSince(this.startTime).compareTo(this.startupGracePeriod) < 0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.SERVER_STARTING_UP, "Cannot reassign shards while server is starting");
        }
        int nodeId = this.getOrCreateNodeId(nodeIdentifier);
        DatabaseUtil.runTransaction(this.dbi, (handle, status) -> {
            ShardDao dao = this.shardDaoSupplier.attach(handle);
            HashSet<Integer> oldAssignments = new HashSet<Integer>(DatabaseShardManager.fetchLockedNodeIds(handle, tableId, shardUuid));
            DatabaseShardManager.updateNodeIds(handle, tableId, shardUuid, (Set<Integer>)ImmutableSet.of((Object)nodeId));
            dao.deleteShardNodes(shardUuid, oldAssignments);
            dao.insertShardNode(shardUuid, nodeId);
            if (deltaUuid.isPresent()) {
                dao.deleteShardNodes((UUID)deltaUuid.get(), oldAssignments);
                dao.insertShardNode((UUID)deltaUuid.get(), nodeId);
            }
            return null;
        });
    }

    @Override
    public Map<String, Long> getNodeBytes() {
        return this.dao.getNodeSizes().stream().collect(Collectors.toMap(NodeSize::getNodeIdentifier, NodeSize::getSizeInBytes));
    }

    @Override
    public long beginTransaction() {
        return this.dao.insertTransaction();
    }

    @Override
    public void rollbackTransaction(long transactionId) {
        this.dao.finalizeTransaction(transactionId, false);
    }

    @Override
    public void createBuckets(long distributionId, int bucketCount) {
        Iterator<String> nodeIterator = DatabaseShardManager.cyclingShuffledIterator(this.getNodeIdentifiers());
        ArrayList<Integer> bucketNumbers = new ArrayList<Integer>();
        ArrayList<Integer> nodeIds = new ArrayList<Integer>();
        for (int bucket = 0; bucket < bucketCount; ++bucket) {
            bucketNumbers.add(bucket);
            nodeIds.add(this.getOrCreateNodeId(nodeIterator.next()));
        }
        DatabaseUtil.runIgnoringConstraintViolation(() -> this.dao.insertBuckets(distributionId, bucketNumbers, nodeIds));
    }

    @Override
    public List<String> getBucketAssignments(long distributionId) {
        try {
            return (List)this.bucketAssignmentsCache.getUnchecked((Object)distributionId);
        }
        catch (UncheckedExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), PrestoException.class);
            throw e;
        }
    }

    @Override
    public void updateBucketAssignment(long distributionId, int bucketNumber, String nodeId) {
        this.dao.updateBucketNode(distributionId, bucketNumber, this.getOrCreateNodeId(nodeId));
    }

    @Override
    public List<Distribution> getDistributions() {
        return this.dao.listActiveDistributions();
    }

    @Override
    public long getDistributionSizeInBytes(long distributionId) {
        return this.dao.getDistributionSizeBytes(distributionId);
    }

    @Override
    public List<BucketNode> getBucketNodes(long distibutionId) {
        return this.dao.getBucketNodes(distibutionId);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public Set<UUID> getExistingShardUuids(long tableId, Set<UUID> shardUuids) {
        try (Handle handle = this.dbi.open();){
            String args = Joiner.on((String)",").join(Collections.nCopies(shardUuids.size(), "?"));
            String selectShards = String.format("SELECT shard_uuid FROM %s WHERE shard_uuid IN (%s)", DatabaseShardManager.shardIndexTable(tableId), args);
            ImmutableSet.Builder existingShards = ImmutableSet.builder();
            try (PreparedStatement statement = handle.getConnection().prepareStatement(selectShards);){
                DatabaseShardManager.bindUuids(statement, shardUuids);
                try (ResultSet rs = statement.executeQuery();){
                    while (rs.next()) {
                        existingShards.add((Object)UuidUtil.uuidFromBytes(rs.getBytes("shard_uuid")));
                    }
                }
            }
            ImmutableSet immutableSet = existingShards.build();
            return immutableSet;
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    private List<BucketNode> getBuckets(long distributionId) {
        return this.dao.getBucketNodes(distributionId);
    }

    private List<String> loadBucketAssignments(long distributionId) {
        Set<String> nodeIds = this.getNodeIdentifiers();
        List<BucketNode> bucketNodes = this.getBuckets(distributionId);
        BucketReassigner reassigner = new BucketReassigner(nodeIds, bucketNodes);
        ArrayList<Object> assignments = new ArrayList<Object>(Collections.nCopies(bucketNodes.size(), null));
        PrestoException limiterException = null;
        HashSet<String> offlineNodes = new HashSet<String>();
        for (BucketNode bucketNode : bucketNodes) {
            int bucket = bucketNode.getBucketNumber();
            String nodeId = bucketNode.getNodeIdentifier();
            if (!nodeIds.contains(nodeId)) {
                block6: {
                    if (this.nanosSince(this.startTime).compareTo(this.startupGracePeriod) < 0) {
                        throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.SERVER_STARTING_UP, "Cannot reassign buckets while server is starting");
                    }
                    try {
                        if (!offlineNodes.add(nodeId)) break block6;
                        this.assignmentLimiter.checkAssignFrom(nodeId);
                    }
                    catch (PrestoException e) {
                        if (limiterException != null) continue;
                        limiterException = e;
                        continue;
                    }
                }
                String oldNodeId = nodeId;
                nodeId = reassigner.getNextReassignmentDestination();
                this.dao.updateBucketNode(distributionId, bucket, this.getOrCreateNodeId(nodeId));
                log.info("Reassigned bucket %s for distribution ID %s from %s to %s", new Object[]{bucket, distributionId, oldNodeId, nodeId});
            }
            Verify.verify((assignments.set(bucket, nodeId) == null ? 1 : 0) != 0, (String)"Duplicate bucket", (Object[])new Object[0]);
        }
        if (limiterException != null) {
            throw limiterException;
        }
        return ImmutableList.copyOf(assignments);
    }

    private Set<String> getNodeIdentifiers() {
        Set<String> nodeIds = this.nodeSupplier.getWorkerNodes().stream().map(Node::getNodeIdentifier).collect(Collectors.toSet());
        if (nodeIds.isEmpty()) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NO_NODES_AVAILABLE, "No nodes available for bucket assignments");
        }
        return nodeIds;
    }

    private int getOrCreateNodeId(String nodeIdentifier) {
        try {
            return (Integer)this.nodeIdCache.getUnchecked((Object)nodeIdentifier);
        }
        catch (UncheckedExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), PrestoException.class);
            throw e;
        }
    }

    private int loadNodeId(String nodeIdentifier) {
        Integer id = this.dao.getNodeId(nodeIdentifier);
        if (id != null) {
            return id;
        }
        DatabaseUtil.runIgnoringConstraintViolation(() -> this.dao.insertNode(nodeIdentifier));
        id = this.dao.getNodeId(nodeIdentifier);
        if (id == null) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "node does not exist after insert");
        }
        return id;
    }

    private Duration nanosSince(long nanos) {
        return new Duration((double)(this.ticker.read() - nanos), TimeUnit.NANOSECONDS);
    }

    private static List<Long> insertShards(Connection connection, long tableId, List<ShardInfo> shards) throws SQLException {
        String sql = "INSERT INTO shards (shard_uuid, table_id, create_time, row_count, compressed_size, uncompressed_size, xxhash64, bucket_number)\nVALUES (?, ?, CURRENT_TIMESTAMP, ?, ?, ?, ?, ?)";
        try (PreparedStatement statement = connection.prepareStatement(sql, 1);){
            for (ShardInfo shard : shards) {
                statement.setBytes(1, UuidUtil.uuidToBytes(shard.getShardUuid()));
                statement.setLong(2, tableId);
                statement.setLong(3, shard.getRowCount());
                statement.setLong(4, shard.getCompressedSize());
                statement.setLong(5, shard.getUncompressedSize());
                statement.setLong(6, shard.getXxhash64());
                DatabaseUtil.bindOptionalInt(statement, 7, shard.getBucketNumber());
                statement.addBatch();
            }
            statement.executeBatch();
            ImmutableList.Builder builder = ImmutableList.builder();
            try (ResultSet keys = statement.getGeneratedKeys();){
                while (keys.next()) {
                    builder.add((Object)keys.getLong(1));
                }
            }
            ImmutableList shardIds = builder.build();
            if (shardIds.size() != shards.size()) {
                throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Wrong number of generated keys for inserted shards");
            }
            var9_9 = shardIds;
            return var9_9;
        }
    }

    private static void insertShardNodes(Connection connection, Map<String, Integer> nodeIds, List<Long> shardIds, List<ShardInfo> shards) throws SQLException {
        Preconditions.checkArgument((shardIds.size() == shards.size() ? 1 : 0) != 0, (Object)"lists are not the same size");
        String sql = "INSERT INTO shard_nodes (shard_id, node_id) VALUES (?, ?)";
        try (PreparedStatement statement = connection.prepareStatement(sql);){
            for (int i = 0; i < shards.size(); ++i) {
                for (String identifier : shards.get(i).getNodeIdentifiers()) {
                    statement.setLong(1, shardIds.get(i));
                    statement.setInt(2, nodeIds.get(identifier));
                    statement.addBatch();
                }
            }
            statement.executeBatch();
        }
    }

    private static Collection<Integer> fetchLockedNodeIds(Handle handle, long tableId, UUID shardUuid) {
        String sql = String.format("SELECT node_ids FROM %s WHERE shard_uuid = ? FOR UPDATE", DatabaseShardManager.shardIndexTable(tableId));
        byte[] nodeArray = (byte[])((Query)handle.createQuery(sql).bind(0, UuidUtil.uuidToBytes(shardUuid))).map((ResultSetMapper)ByteArrayMapper.FIRST).first();
        return ArrayUtil.intArrayFromBytes(nodeArray);
    }

    private static void updateNodeIds(Handle handle, long tableId, UUID shardUuid, Set<Integer> nodeIds) {
        String sql = String.format("UPDATE %s SET node_ids = ? WHERE shard_uuid = ?", DatabaseShardManager.shardIndexTable(tableId));
        handle.execute(sql, new Object[]{ArrayUtil.intArrayToBytes(nodeIds), UuidUtil.uuidToBytes(shardUuid)});
    }

    private static void lockTable(Handle handle, long tableId) {
        if (((MetadataDao)handle.attach(MetadataDao.class)).getLockedTableId(tableId) == null) {
            throw DatabaseShardManager.transactionConflict();
        }
    }

    public static PrestoException transactionConflict() {
        return new PrestoException((ErrorCodeSupplier)StandardErrorCode.TRANSACTION_CONFLICT, "Table was updated by a different transaction. Please retry the operation.");
    }

    public static String shardIndexTable(long tableId) {
        return INDEX_TABLE_PREFIX + tableId;
    }

    public static String minColumn(long columnId) {
        Preconditions.checkArgument((columnId >= 0L ? 1 : 0) != 0, (String)"invalid columnId %s", (long)columnId);
        return String.format("c%s_min", columnId);
    }

    public static String maxColumn(long columnId) {
        Preconditions.checkArgument((columnId >= 0L ? 1 : 0) != 0, (String)"invalid columnId %s", (long)columnId);
        return String.format("c%s_max", columnId);
    }

    private static String sqlColumnType(Type type) {
        JDBCType jdbcType = ColumnIndexStatsUtils.jdbcType(type);
        if (jdbcType != null) {
            switch (jdbcType) {
                case BOOLEAN: {
                    return "boolean";
                }
                case BIGINT: {
                    return "bigint";
                }
                case DOUBLE: {
                    return "double";
                }
                case INTEGER: {
                    return "int";
                }
                case VARBINARY: {
                    return String.format("varbinary(%s)", 100);
                }
            }
        }
        return null;
    }

    private static <T> Iterator<T> cyclingShuffledIterator(Collection<T> collection) {
        ArrayList<T> list = new ArrayList<T>(collection);
        Collections.shuffle(list);
        return Iterables.cycle(list).iterator();
    }

    private static ShardStats shardStats(Collection<ShardInfo> shards) {
        return new ShardStats(shards.stream().mapToLong(ShardInfo::getRowCount).sum(), shards.stream().mapToLong(ShardInfo::getCompressedSize).sum(), shards.stream().mapToLong(ShardInfo::getUncompressedSize).sum());
    }

    @Managed
    @Flatten
    public DeltaDeleteStats getDeltaDeleteStats() {
        return this.deltaDeleteStats;
    }

    private static class DeltaDeleteStats {
        private final CounterStat updatedShards = new CounterStat();
        private final CounterStat deletedShards = new CounterStat();

        private DeltaDeleteStats() {
        }
    }

    private static class DeltaUuidPair {
        private Optional<UUID> oldDeltaUuid;
        private UUID newDeltaUuid;

        public DeltaUuidPair(Optional<UUID> oldDeltaUuid, UUID newDeltaUuid) {
            this.oldDeltaUuid = oldDeltaUuid;
            this.newDeltaUuid = newDeltaUuid;
        }

        public Optional<UUID> getOldDeltaUuid() {
            return this.oldDeltaUuid;
        }

        public UUID getNewDeltaUuid() {
            return this.newDeltaUuid;
        }
    }

    private static class ShardStats {
        private final long rowCount;
        private final long compressedSize;
        private final long uncompressedSize;

        public ShardStats(long rowCount, long compressedSize, long uncompressedSize) {
            this.rowCount = rowCount;
            this.compressedSize = compressedSize;
            this.uncompressedSize = uncompressedSize;
        }

        public long getRowCount() {
            return this.rowCount;
        }

        public long getCompressedSize() {
            return this.compressedSize;
        }

        public long getUncompressedSize() {
            return this.uncompressedSize;
        }
    }
}

