/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.elasticsearch;

import com.facebook.airlift.concurrent.Threads;
import com.facebook.airlift.json.ObjectMapperProvider;
import com.facebook.airlift.log.Logger;
import com.facebook.presto.elasticsearch.ElasticsearchColumn;
import com.facebook.presto.elasticsearch.ElasticsearchConnectorConfig;
import com.facebook.presto.elasticsearch.ElasticsearchErrorCode;
import com.facebook.presto.elasticsearch.ElasticsearchTableDescription;
import com.facebook.presto.elasticsearch.ElasticsearchTableDescriptionProvider;
import com.facebook.presto.elasticsearch.RetryDriver;
import com.facebook.presto.spi.ColumnMetadata;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.SchemaTableName;
import com.facebook.presto.spi.type.BigintType;
import com.facebook.presto.spi.type.BooleanType;
import com.facebook.presto.spi.type.DoubleType;
import com.facebook.presto.spi.type.IntegerType;
import com.facebook.presto.spi.type.RowType;
import com.facebook.presto.spi.type.Type;
import com.facebook.presto.spi.type.VarbinaryType;
import com.facebook.presto.spi.type.VarcharType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.floragunn.searchguard.ssl.SearchGuardSSLPlugin;
import com.google.common.base.Strings;
import com.google.common.base.Verify;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.io.Closer;
import io.airlift.units.Duration;
import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.elasticsearch.action.admin.cluster.shards.ClusterSearchShardsRequest;
import org.elasticsearch.action.admin.cluster.shards.ClusterSearchShardsResponse;
import org.elasticsearch.action.admin.indices.get.GetIndexRequest;
import org.elasticsearch.action.admin.indices.get.GetIndexResponse;
import org.elasticsearch.action.admin.indices.mapping.get.GetMappingsRequest;
import org.elasticsearch.action.admin.indices.mapping.get.GetMappingsResponse;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.cluster.metadata.MappingMetaData;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.transport.client.PreBuiltTransportClient;

public class ElasticsearchClient {
    private static final Logger LOG = Logger.get(ElasticsearchClient.class);
    private final ExecutorService executor = Executors.newFixedThreadPool(1, Threads.daemonThreadsNamed((String)"elasticsearch-metadata-%s"));
    private final ObjectMapper objecMapper = new ObjectMapperProvider().get();
    private final ElasticsearchTableDescriptionProvider tableDescriptions;
    private final Map<String, TransportClient> clients = new HashMap<String, TransportClient>();
    private final LoadingCache<ElasticsearchTableDescription, List<ColumnMetadata>> columnMetadataCache;
    private final Duration requestTimeout;
    private final int maxAttempts;
    private final Duration maxRetryTime;

    @Inject
    public ElasticsearchClient(ElasticsearchTableDescriptionProvider descriptions, ElasticsearchConnectorConfig config) throws IOException {
        this.tableDescriptions = Objects.requireNonNull(descriptions, "description is null");
        ElasticsearchConnectorConfig configuration = Objects.requireNonNull(config, "config is null");
        this.requestTimeout = configuration.getRequestTimeout();
        this.maxAttempts = configuration.getMaxRequestRetries();
        this.maxRetryTime = configuration.getMaxRetryTime();
        for (ElasticsearchTableDescription tableDescription : this.tableDescriptions.getAllTableDescriptions()) {
            if (this.clients.containsKey(tableDescription.getClusterName())) continue;
            TransportAddress address = new TransportAddress(InetAddress.getByName(tableDescription.getHost()), tableDescription.getPort());
            TransportClient client = ElasticsearchClient.createTransportClient(config, address, Optional.of(tableDescription.getClusterName()));
            this.clients.put(tableDescription.getClusterName(), client);
        }
        this.columnMetadataCache = CacheBuilder.newBuilder().expireAfterWrite(30L, TimeUnit.MINUTES).refreshAfterWrite(15L, TimeUnit.MINUTES).maximumSize(500L).build(CacheLoader.asyncReloading((CacheLoader)CacheLoader.from(this::loadColumns), (Executor)this.executor));
    }

    @PreDestroy
    public void tearDown() {
        try (Closer closer = Closer.create();){
            closer.register(this.clients::clear);
            for (Map.Entry<String, TransportClient> entry : this.clients.entrySet()) {
                closer.register((Closeable)entry.getValue());
            }
            closer.register(this.executor::shutdown);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public List<String> listSchemas() {
        return (List)this.tableDescriptions.getAllSchemaTableNames().stream().map(SchemaTableName::getSchemaName).collect(ImmutableList.toImmutableList());
    }

    public List<SchemaTableName> listTables(Optional<String> schemaName) {
        return (List)this.tableDescriptions.getAllSchemaTableNames().stream().filter(schemaTableName -> !schemaName.isPresent() || schemaTableName.getSchemaName().equals(schemaName.get())).collect(ImmutableList.toImmutableList());
    }

    private List<ColumnMetadata> loadColumns(ElasticsearchTableDescription table) {
        if (table.getColumns().isPresent()) {
            return this.buildMetadata(table.getColumns().get());
        }
        return this.buildMetadata(this.buildColumns(table));
    }

    public List<ColumnMetadata> getColumnMetadata(ElasticsearchTableDescription tableDescription) {
        return (List)this.columnMetadataCache.getUnchecked((Object)tableDescription);
    }

    public ElasticsearchTableDescription getTable(String schemaName, String tableName) {
        Objects.requireNonNull(schemaName, "schemaName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        ElasticsearchTableDescription table = this.tableDescriptions.get(new SchemaTableName(schemaName, tableName));
        if (table == null) {
            return null;
        }
        if (table.getColumns().isPresent()) {
            return table;
        }
        return new ElasticsearchTableDescription(table.getTableName(), table.getSchemaName(), table.getHost(), table.getPort(), table.getClusterName(), table.getIndex(), table.getIndexExactMatch(), table.getType(), Optional.of(this.buildColumns(table)));
    }

    public List<String> getIndices(ElasticsearchTableDescription tableDescription) {
        if (tableDescription.getIndexExactMatch()) {
            return ImmutableList.of((Object)tableDescription.getIndex());
        }
        TransportClient client = this.clients.get(tableDescription.getClusterName());
        Verify.verify((client != null ? 1 : 0) != 0, (String)"client is null", (Object[])new Object[0]);
        String[] indices = this.getIndices(client, new GetIndexRequest());
        return (List)Arrays.stream(indices).filter(index -> index.startsWith(tableDescription.getIndex())).collect(ImmutableList.toImmutableList());
    }

    public ClusterSearchShardsResponse getSearchShards(String index, ElasticsearchTableDescription tableDescription) {
        TransportClient client = this.clients.get(tableDescription.getClusterName());
        Verify.verify((client != null ? 1 : 0) != 0, (String)"client is null", (Object[])new Object[0]);
        return this.getSearchShardsResponse(client, new ClusterSearchShardsRequest(new String[]{index}));
    }

    private String[] getIndices(TransportClient client, GetIndexRequest request) {
        try {
            return RetryDriver.retry().maxAttempts(this.maxAttempts).exponentialBackoff(this.maxRetryTime).run("getIndices", () -> ((GetIndexResponse)client.admin().indices().getIndex(request).actionGet(this.requestTimeout.toMillis())).getIndices());
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private ClusterSearchShardsResponse getSearchShardsResponse(TransportClient client, ClusterSearchShardsRequest request) {
        try {
            return RetryDriver.retry().maxAttempts(this.maxAttempts).exponentialBackoff(this.maxRetryTime).run("getSearchShardsResponse", () -> (ClusterSearchShardsResponse)client.admin().cluster().searchShards(request).actionGet(this.requestTimeout.toMillis()));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private List<ColumnMetadata> buildMetadata(List<ElasticsearchColumn> columns) {
        ArrayList<ColumnMetadata> result = new ArrayList<ColumnMetadata>();
        for (ElasticsearchColumn column : columns) {
            HashMap<String, Object> properties = new HashMap<String, Object>();
            properties.put("originalColumnName", column.getName());
            properties.put("jsonPath", column.getJsonPath());
            properties.put("jsonType", column.getJsonType());
            properties.put("isList", column.isList());
            properties.put("ordinalPosition", column.getOrdinalPosition());
            result.add(new ColumnMetadata(column.getName(), column.getType(), "", "", false, properties));
        }
        return result;
    }

    private List<ElasticsearchColumn> buildColumns(ElasticsearchTableDescription tableDescription) {
        ArrayList<ElasticsearchColumn> columns = new ArrayList<ElasticsearchColumn>();
        TransportClient client = this.clients.get(tableDescription.getClusterName());
        Verify.verify((client != null ? 1 : 0) != 0, (String)"client is null", (Object[])new Object[0]);
        for (String index : this.getIndices(tableDescription)) {
            GetMappingsRequest mappingsRequest = (GetMappingsRequest)new GetMappingsRequest().types(new String[]{tableDescription.getType()});
            if (!Strings.isNullOrEmpty((String)index)) {
                mappingsRequest.indices(new String[]{index});
            }
            ImmutableOpenMap<String, ImmutableOpenMap<String, MappingMetaData>> mappings = this.getMappings(client, mappingsRequest);
            Iterator indexIterator = mappings.keysIt();
            while (indexIterator.hasNext()) {
                JsonNode arrayNode;
                JsonNode rootNode;
                MappingMetaData mappingMetaData = (MappingMetaData)((ImmutableOpenMap)mappings.get(indexIterator.next())).get((Object)tableDescription.getType());
                try {
                    rootNode = this.objecMapper.readTree(mappingMetaData.source().uncompressed());
                }
                catch (IOException e) {
                    throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CORRUPTED_MAPPING_METADATA, (Throwable)e);
                }
                JsonNode mappingNode = rootNode.get(tableDescription.getType());
                JsonNode propertiesNode = mappingNode.get("properties");
                ArrayList<String> lists = new ArrayList<String>();
                JsonNode metaNode = mappingNode.get("_meta");
                if (metaNode != null && (arrayNode = metaNode.get("lists")) != null && arrayNode.isArray()) {
                    ArrayNode arrays = (ArrayNode)arrayNode;
                    for (int i = 0; i < arrays.size(); ++i) {
                        lists.add(arrays.get(i).textValue());
                    }
                }
                this.populateColumns(propertiesNode, lists, columns);
            }
        }
        return columns;
    }

    private ImmutableOpenMap<String, ImmutableOpenMap<String, MappingMetaData>> getMappings(TransportClient client, GetMappingsRequest request) {
        try {
            return RetryDriver.retry().maxAttempts(this.maxAttempts).exponentialBackoff(this.maxRetryTime).run("getMappings", () -> ((GetMappingsResponse)client.admin().indices().getMappings(request).actionGet(this.requestTimeout.toMillis())).getMappings());
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private List<String> getColumnMetadata(Optional<String> parent, JsonNode propertiesNode) {
        ImmutableList.Builder metadata = ImmutableList.builder();
        Iterator iterator = propertiesNode.fields();
        while (iterator.hasNext()) {
            Map.Entry entry = (Map.Entry)iterator.next();
            String key = (String)entry.getKey();
            JsonNode value = (JsonNode)entry.getValue();
            String childKey = parent.isPresent() ? (parent.get().isEmpty() ? key : parent.get().concat(".").concat(key)) : key;
            if (value.isObject()) {
                metadata.addAll(this.getColumnMetadata(Optional.of(childKey), value));
                continue;
            }
            if (value.isArray()) continue;
            metadata.add((Object)childKey.concat(":").concat(value.textValue()));
        }
        return metadata.build();
    }

    private void populateColumns(JsonNode propertiesNode, List<String> arrays, List<ElasticsearchColumn> columns) {
        FieldNestingComparator comparator = new FieldNestingComparator();
        TreeMap<String, Type> fieldsMap = new TreeMap<String, Type>(comparator);
        for (String columnMetadata : this.getColumnMetadata(Optional.empty(), propertiesNode)) {
            int delimiterIndex = columnMetadata.lastIndexOf(":");
            if (delimiterIndex == -1 || delimiterIndex == columnMetadata.length() - 1) {
                LOG.debug("Invalid column path format: %s", new Object[]{columnMetadata});
                continue;
            }
            String fieldName = columnMetadata.substring(0, delimiterIndex);
            String typeName = columnMetadata.substring(delimiterIndex + 1);
            if (!fieldName.endsWith(".type")) {
                LOG.debug("Ignoring column with no type info: %s", new Object[]{columnMetadata});
                continue;
            }
            String propertyName = fieldName.substring(0, fieldName.lastIndexOf(46));
            String nestedName = propertyName.replaceAll("properties\\.", "");
            if (nestedName.contains(".")) {
                fieldsMap.put(nestedName, ElasticsearchClient.getPrestoType(typeName));
                continue;
            }
            boolean newColumnFound = columns.stream().noneMatch(column -> column.getName().equalsIgnoreCase(nestedName));
            if (!newColumnFound) continue;
            columns.add(new ElasticsearchColumn(nestedName, ElasticsearchClient.getPrestoType(typeName), nestedName, typeName, arrays.contains(nestedName), -1));
        }
        this.processNestedFields(fieldsMap, columns, arrays);
    }

    private void processNestedFields(TreeMap<String, Type> fieldsMap, List<ElasticsearchColumn> columns, List<String> arrays) {
        if (fieldsMap.size() == 0) {
            return;
        }
        Map.Entry<String, Type> first = fieldsMap.firstEntry();
        String field = first.getKey();
        Type type = first.getValue();
        if (field.contains(".")) {
            String prefix = field.substring(0, field.lastIndexOf(46));
            ImmutableList.Builder fieldsBuilder = ImmutableList.builder();
            int size = field.split("\\.").length;
            Iterator<String> iterator = fieldsMap.navigableKeySet().iterator();
            while (iterator.hasNext()) {
                String name = iterator.next();
                if (name.split("\\.").length != size || !name.startsWith(prefix)) continue;
                Optional<String> columnName = Optional.of(name.substring(name.lastIndexOf(46) + 1));
                Type columnType = fieldsMap.get(name);
                RowType.Field column2 = new RowType.Field(columnName, columnType);
                fieldsBuilder.add((Object)column2);
                iterator.remove();
            }
            fieldsMap.put(prefix, (Type)RowType.from((List)fieldsBuilder.build()));
        } else {
            boolean newColumnFound = columns.stream().noneMatch(column -> column.getName().equalsIgnoreCase(field));
            if (newColumnFound) {
                columns.add(new ElasticsearchColumn(field, type, field, type.getDisplayName(), arrays.contains(field), -1));
            }
            fieldsMap.remove(field);
        }
        this.processNestedFields(fieldsMap, columns, arrays);
    }

    private static Type getPrestoType(String elasticsearchType) {
        switch (elasticsearchType) {
            case "double": 
            case "float": {
                return DoubleType.DOUBLE;
            }
            case "integer": {
                return IntegerType.INTEGER;
            }
            case "long": {
                return BigintType.BIGINT;
            }
            case "string": 
            case "text": 
            case "keyword": {
                return VarcharType.VARCHAR;
            }
            case "boolean": {
                return BooleanType.BOOLEAN;
            }
            case "binary": {
                return VarbinaryType.VARBINARY;
            }
        }
        throw new IllegalArgumentException("Unsupported type: " + elasticsearchType);
    }

    static TransportClient createTransportClient(ElasticsearchConnectorConfig config, TransportAddress address) {
        return ElasticsearchClient.createTransportClient(config, address, Optional.empty());
    }

    static TransportClient createTransportClient(ElasticsearchConnectorConfig config, TransportAddress address, Optional<String> clusterName) {
        TransportClient client;
        Settings.Builder builder = clusterName.isPresent() ? Settings.builder().put("cluster.name", clusterName.get()) : Settings.builder().put("client.transport.ignore_cluster_name", true);
        switch (config.getCertificateFormat()) {
            case PEM: {
                Settings settings = builder.put(new Object[]{"searchguard.ssl.transport.pemcert_filepath", config.getPemcertFilepath()}).put(new Object[]{"searchguard.ssl.transport.pemkey_filepath", config.getPemkeyFilepath()}).put("searchguard.ssl.transport.pemkey_password", config.getPemkeyPassword()).put(new Object[]{"searchguard.ssl.transport.pemtrustedcas_filepath", config.getPemtrustedcasFilepath()}).put("searchguard.ssl.transport.enforce_hostname_verification", false).build();
                client = new PreBuiltTransportClient(settings, new Class[]{SearchGuardSSLPlugin.class}).addTransportAddress(address);
                break;
            }
            case JKS: {
                Settings settings = Settings.builder().put(new Object[]{"searchguard.ssl.transport.keystore_filepath", config.getKeystoreFilepath()}).put(new Object[]{"searchguard.ssl.transport.truststore_filepath", config.getTruststoreFilepath()}).put("searchguard.ssl.transport.keystore_password", config.getKeystorePassword()).put("searchguard.ssl.transport.truststore_password", config.getTruststorePassword()).put("searchguard.ssl.transport.enforce_hostname_verification", false).build();
                client = new PreBuiltTransportClient(settings, new Class[]{SearchGuardSSLPlugin.class}).addTransportAddress(address);
                break;
            }
            default: {
                Settings settings = builder.build();
                client = new PreBuiltTransportClient(settings, new Class[0]).addTransportAddress(address);
            }
        }
        return client;
    }

    private static class FieldNestingComparator
    implements Comparator<String> {
        FieldNestingComparator() {
        }

        @Override
        public int compare(String left, String right) {
            int rightLength;
            int leftLength = left.split("\\.").length;
            if (leftLength == (rightLength = right.split("\\.").length)) {
                return left.compareTo(right);
            }
            return rightLength - leftLength;
        }
    }
}

