/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.orc;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Longs;
import io.airlift.units.DataSize;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.OptionalDouble;
import java.util.OptionalInt;
import java.util.Set;

public class DictionaryCompressionOptimizer {
    private static final double DICTIONARY_MIN_COMPRESSION_RATIO = 1.25;
    private static final int TWO_BYTES_VARINT_MIN = 128;
    private static final int THREE_BYTES_VARINT_MIN = 16384;
    private static final int FOUR_BYTES_VARINT_MIN = 0x200000;
    static final DataSize DIRECT_COLUMN_SIZE_RANGE = new DataSize(4.0, DataSize.Unit.MEGABYTE);
    static final int NUMBER_OF_NULLS_FOR_DICTIONARY_BYTE = 4;
    private final List<DictionaryColumnManager> allWriters;
    private final List<DictionaryColumnManager> directConversionCandidates = new ArrayList<DictionaryColumnManager>();
    private final int stripeMinBytes;
    private final int stripeMaxBytes;
    private final int stripeMaxRowCount;
    private final int dictionaryMemoryMaxBytesLow;
    private final int dictionaryMemoryMaxBytesHigh;
    private final int dictionaryUsefulCheckColumnSizeBytes;
    private final int dictionaryUsefulCheckPerChunkFrequency;
    private int dictionaryMemoryBytes;
    private int dictionaryUsefulCheckCounter;

    public DictionaryCompressionOptimizer(Set<? extends DictionaryColumn> writers, int stripeMinBytes, int stripeMaxBytes, int stripeMaxRowCount, int dictionaryMemoryMaxBytes, int dictionaryMemoryAlmostFullRangeBytes, int dictionaryUsefulCheckColumnSizeBytes, int dictionaryUsefulCheckPerChunkFrequency) {
        Objects.requireNonNull(writers, "writers is null");
        this.allWriters = (List)writers.stream().map(DictionaryColumnManager::new).collect(ImmutableList.toImmutableList());
        Preconditions.checkArgument((stripeMinBytes >= 0 ? 1 : 0) != 0, (Object)"stripeMinBytes is negative");
        this.stripeMinBytes = stripeMinBytes;
        Preconditions.checkArgument((stripeMaxBytes >= stripeMinBytes ? 1 : 0) != 0, (Object)"stripeMaxBytes is less than stripeMinBytes");
        this.stripeMaxBytes = stripeMaxBytes;
        Preconditions.checkArgument((stripeMaxRowCount >= 0 ? 1 : 0) != 0, (Object)"stripeMaxRowCount is negative");
        this.stripeMaxRowCount = stripeMaxRowCount;
        Preconditions.checkArgument((dictionaryMemoryMaxBytes >= 0 ? 1 : 0) != 0, (Object)"dictionaryMemoryMaxBytes is negative");
        Preconditions.checkArgument((dictionaryMemoryAlmostFullRangeBytes >= 0 ? 1 : 0) != 0, (Object)"dictionaryMemoryRangeBytes is negative");
        this.dictionaryMemoryMaxBytesHigh = dictionaryMemoryMaxBytes;
        this.dictionaryMemoryMaxBytesLow = Math.max(dictionaryMemoryMaxBytes - dictionaryMemoryAlmostFullRangeBytes, 0);
        Preconditions.checkArgument((dictionaryUsefulCheckPerChunkFrequency >= 0 ? 1 : 0) != 0, (Object)"dictionaryUsefulCheckPerChunkFrequency is negative");
        this.dictionaryUsefulCheckPerChunkFrequency = dictionaryUsefulCheckPerChunkFrequency;
        this.dictionaryUsefulCheckColumnSizeBytes = dictionaryUsefulCheckColumnSizeBytes;
        this.directConversionCandidates.addAll(this.allWriters);
    }

    public int getDictionaryMemoryBytes() {
        return this.dictionaryMemoryBytes;
    }

    public boolean isFull(long bufferedBytes) {
        if (bufferedBytes > (long)this.stripeMinBytes) {
            return this.dictionaryMemoryBytes > this.dictionaryMemoryMaxBytesLow;
        }
        return this.dictionaryMemoryBytes > this.dictionaryMemoryMaxBytesHigh;
    }

    public void reset() {
        this.directConversionCandidates.clear();
        this.directConversionCandidates.addAll(this.allWriters);
        this.dictionaryMemoryBytes = 0;
        this.allWriters.forEach(DictionaryColumnManager::reset);
    }

    public void finalOptimize(int bufferedBytes) {
        this.updateDirectConversionCandidates();
        this.convertLowCompressionStreams(true, bufferedBytes);
    }

    @VisibleForTesting
    boolean isUsefulCheckRequired(int dictionaryMemoryBytes) {
        if (dictionaryMemoryBytes < this.dictionaryUsefulCheckColumnSizeBytes) {
            return false;
        }
        ++this.dictionaryUsefulCheckCounter;
        if (this.dictionaryUsefulCheckCounter == this.dictionaryUsefulCheckPerChunkFrequency) {
            this.dictionaryUsefulCheckCounter = 0;
            return true;
        }
        return false;
    }

    public void optimize(int bufferedBytes, int stripeRowCount) {
        boolean isDictionaryAlmostFull;
        int totalDictionaryBytes = 0;
        long totalNullBytes = 0L;
        for (DictionaryColumnManager writer : this.allWriters) {
            if (writer.isDirectEncoded()) continue;
            totalDictionaryBytes += writer.getDictionaryBytes();
            totalNullBytes += writer.getDictionaryColumn().getNullValueCount();
            writer.updateHistory(stripeRowCount);
        }
        this.dictionaryMemoryBytes = totalDictionaryBytes;
        boolean bl = isDictionaryAlmostFull = (long)this.dictionaryMemoryBytes + totalNullBytes / 4L > (long)this.dictionaryMemoryMaxBytesLow;
        if (isDictionaryAlmostFull || this.isUsefulCheckRequired(this.dictionaryMemoryBytes)) {
            this.updateDirectConversionCandidates();
            bufferedBytes = this.convertLowCompressionStreams(isDictionaryAlmostFull, bufferedBytes);
        }
        if (this.dictionaryMemoryBytes <= this.dictionaryMemoryMaxBytesLow || bufferedBytes >= this.stripeMaxBytes) {
            return;
        }
        int nonDictionaryBufferedBytes = bufferedBytes;
        for (DictionaryColumnManager dictionaryWriter : this.allWriters) {
            if (dictionaryWriter.isDirectEncoded()) continue;
            nonDictionaryBufferedBytes = (int)((long)nonDictionaryBufferedBytes - dictionaryWriter.getBufferedBytes());
        }
        BufferedBytesCounter bufferedBytesCounter = new BufferedBytesCounter(bufferedBytes, nonDictionaryBufferedBytes);
        this.optimizeDictionaryColumns(stripeRowCount, bufferedBytesCounter);
    }

    private void optimizeDictionaryColumns(int stripeRowCount, BufferedBytesCounter bufferedBytesCounter) {
        while (!this.directConversionCandidates.isEmpty() && this.dictionaryMemoryBytes > this.dictionaryMemoryMaxBytesHigh && bufferedBytesCounter.getBufferedBytes() < this.stripeMaxBytes) {
            this.convertDictionaryColumn(bufferedBytesCounter, stripeRowCount, OptionalDouble.empty());
        }
        if (bufferedBytesCounter.getBufferedBytes() >= this.stripeMaxBytes) {
            return;
        }
        if (bufferedBytesCounter.getBufferedBytes() >= this.stripeMinBytes) {
            double currentCompressionRatio = this.currentCompressionRatio(bufferedBytesCounter.getNonDictionaryBufferedBytes());
            while (!this.directConversionCandidates.isEmpty() && bufferedBytesCounter.getBufferedBytes() < this.stripeMaxBytes) {
                if (this.convertDictionaryColumn(bufferedBytesCounter, stripeRowCount, OptionalDouble.of(currentCompressionRatio))) continue;
                return;
            }
        }
    }

    private boolean convertDictionaryColumn(BufferedBytesCounter bufferedBytesCounter, int stripeRowCount, OptionalDouble currentCompressionRatio) {
        DictionaryCompressionProjection projection = this.selectDictionaryColumnToConvert(bufferedBytesCounter.getNonDictionaryBufferedBytes(), stripeRowCount);
        int index = projection.getDirectConversionCandidateIndex();
        if (currentCompressionRatio.isPresent() && projection.getPredictedFileCompressionRatio() < currentCompressionRatio.getAsDouble()) {
            return false;
        }
        DictionaryColumnManager column = this.directConversionCandidates.get(index);
        int dictionaryBytes = Math.toIntExact(column.getBufferedBytes());
        OptionalInt directBytes = this.tryConvertToDirect(column, this.getMaxDirectBytes(bufferedBytesCounter.getBufferedBytes()));
        this.removeDirectConversionCandidate(index);
        if (directBytes.isPresent()) {
            bufferedBytesCounter.incrementBufferedBytes(directBytes.getAsInt() - dictionaryBytes);
            bufferedBytesCounter.incrementNonDictionaryBufferedBytes(directBytes.getAsInt());
        }
        return true;
    }

    @VisibleForTesting
    int convertLowCompressionStreams(boolean tryAllStreams, int bufferedBytes) {
        Iterator<DictionaryColumnManager> iterator = this.directConversionCandidates.iterator();
        while (iterator.hasNext()) {
            DictionaryColumnManager dictionaryWriter = iterator.next();
            if (!tryAllStreams && dictionaryWriter.getDictionaryBytes() < this.dictionaryUsefulCheckColumnSizeBytes || !(dictionaryWriter.getCompressionRatio() < 1.25)) continue;
            int columnBufferedBytes = Math.toIntExact(dictionaryWriter.getBufferedBytes());
            OptionalInt directBytes = this.tryConvertToDirect(dictionaryWriter, this.getMaxDirectBytes(bufferedBytes));
            iterator.remove();
            if (!directBytes.isPresent() || (bufferedBytes = bufferedBytes + directBytes.getAsInt() - columnBufferedBytes) < this.stripeMaxBytes) continue;
            return bufferedBytes;
        }
        return bufferedBytes;
    }

    @VisibleForTesting
    List<DictionaryColumnManager> getDirectConversionCandidates() {
        return this.directConversionCandidates;
    }

    private void updateDirectConversionCandidates() {
        this.directConversionCandidates.removeIf(DictionaryColumnManager::isDirectEncoded);
    }

    private void removeDirectConversionCandidate(int index) {
        DictionaryColumnManager last = this.directConversionCandidates.get(this.directConversionCandidates.size() - 1);
        this.directConversionCandidates.set(index, last);
        this.directConversionCandidates.remove(this.directConversionCandidates.size() - 1);
    }

    private OptionalInt tryConvertToDirect(DictionaryColumnManager dictionaryWriter, int maxDirectBytes) {
        int dictionaryBytes = dictionaryWriter.getDictionaryBytes();
        OptionalInt directBytes = dictionaryWriter.tryConvertToDirect(maxDirectBytes);
        if (directBytes.isPresent()) {
            this.dictionaryMemoryBytes -= dictionaryBytes;
        }
        return directBytes;
    }

    private double currentCompressionRatio(int totalNonDictionaryBytes) {
        long uncompressedBytes = totalNonDictionaryBytes;
        long compressedBytes = totalNonDictionaryBytes;
        for (DictionaryColumnManager column : this.allWriters) {
            if (column.isDirectEncoded()) continue;
            uncompressedBytes += column.getRawBytesEstimate();
            compressedBytes += (long)column.getDictionaryBytes();
        }
        return 1.0 * (double)uncompressedBytes / (double)compressedBytes;
    }

    private DictionaryCompressionProjection selectDictionaryColumnToConvert(int totalNonDictionaryBytes, int stripeRowCount) {
        Preconditions.checkState((!this.directConversionCandidates.isEmpty() ? 1 : 0) != 0);
        int totalNonDictionaryBytesPerRow = totalNonDictionaryBytes / stripeRowCount;
        long totalDictionaryRawBytes = 0L;
        long totalDictionaryBytes = 0L;
        long totalDictionaryIndexBytes = 0L;
        long totalDictionaryRawBytesPerRow = 0L;
        long totalDictionaryBytesPerNewRow = 0L;
        long totalDictionaryIndexBytesPerRow = 0L;
        for (DictionaryColumnManager column : this.allWriters) {
            if (column.isDirectEncoded()) continue;
            totalDictionaryRawBytes += column.getRawBytesEstimate();
            totalDictionaryBytes += (long)column.getDictionaryBytes();
            totalDictionaryIndexBytes += (long)column.getIndexBytes();
            totalDictionaryRawBytesPerRow = (long)((double)totalDictionaryRawBytesPerRow + column.getRawBytesPerRow());
            totalDictionaryBytesPerNewRow = (long)((double)totalDictionaryBytesPerNewRow + column.getDictionaryBytesPerFutureRow());
            totalDictionaryIndexBytesPerRow = (long)((double)totalDictionaryIndexBytesPerRow + column.getIndexBytesPerRow());
        }
        long totalUncompressedBytesPerRow = (long)totalNonDictionaryBytesPerRow + totalDictionaryRawBytesPerRow;
        DictionaryCompressionProjection maxProjectedCompression = null;
        for (int index = 0; index < this.directConversionCandidates.size(); ++index) {
            DictionaryColumnManager column = this.directConversionCandidates.get(index);
            long currentRawBytes = (long)totalNonDictionaryBytes + column.getRawBytesEstimate();
            long currentDictionaryBytes = totalDictionaryBytes - (long)column.getDictionaryBytes();
            long currentIndexBytes = totalDictionaryIndexBytes - (long)column.getIndexBytes();
            long currentTotalBytes = currentRawBytes + currentDictionaryBytes + currentIndexBytes;
            double rawBytesPerFutureRow = (double)totalNonDictionaryBytesPerRow + column.getRawBytesPerRow();
            double dictionaryBytesPerFutureRow = (double)totalDictionaryBytesPerNewRow - column.getDictionaryBytesPerFutureRow();
            double indexBytesPerFutureRow = (double)totalDictionaryIndexBytesPerRow - column.getIndexBytesPerRow();
            double totalBytesPerFutureRow = rawBytesPerFutureRow + dictionaryBytesPerFutureRow + indexBytesPerFutureRow;
            long rowsToDictionaryMemoryLimit = (long)((double)((long)this.dictionaryMemoryMaxBytesLow - currentDictionaryBytes) / dictionaryBytesPerFutureRow);
            long rowsToStripeMemoryLimit = (long)((double)((long)this.stripeMaxBytes - currentTotalBytes) / totalBytesPerFutureRow);
            long rowsToStripeRowLimit = this.stripeMaxRowCount - stripeRowCount;
            long rowsToLimit = Longs.min((long[])new long[]{rowsToDictionaryMemoryLimit, rowsToStripeMemoryLimit, rowsToStripeRowLimit});
            long predictedUncompressedSizeAtLimit = (long)totalNonDictionaryBytes + totalDictionaryRawBytes + totalUncompressedBytesPerRow * rowsToLimit;
            long predictedCompressedSizeAtLimit = (long)((double)currentTotalBytes + totalBytesPerFutureRow * (double)rowsToLimit);
            double predictedCompressionRatioAtLimit = 1.0 * (double)predictedUncompressedSizeAtLimit / (double)predictedCompressedSizeAtLimit;
            if (maxProjectedCompression != null && !(maxProjectedCompression.getPredictedFileCompressionRatio() < predictedCompressionRatioAtLimit)) continue;
            maxProjectedCompression = new DictionaryCompressionProjection(index, predictedCompressionRatioAtLimit);
        }
        return maxProjectedCompression;
    }

    private int getMaxDirectBytes(int bufferedBytes) {
        return Math.toIntExact(Math.min((long)this.stripeMaxBytes, (long)(this.stripeMaxBytes - bufferedBytes) + DIRECT_COLUMN_SIZE_RANGE.toBytes()));
    }

    public static int estimateIndexBytesPerValue(int dictionaryEntries) {
        if (dictionaryEntries < 128) {
            return 1;
        }
        if (dictionaryEntries < 16384) {
            return 2;
        }
        if (dictionaryEntries < 0x200000) {
            return 3;
        }
        return 4;
    }

    private static class BufferedBytesCounter {
        private int bufferedBytes;
        private int nonDictionaryBufferedBytes;

        public BufferedBytesCounter(int bufferedBytes, int nonDictionaryBufferedBytes) {
            this.bufferedBytes = bufferedBytes;
            this.nonDictionaryBufferedBytes = nonDictionaryBufferedBytes;
        }

        public int getBufferedBytes() {
            return this.bufferedBytes;
        }

        public void incrementBufferedBytes(int value) {
            this.bufferedBytes += value;
        }

        public int getNonDictionaryBufferedBytes() {
            return this.nonDictionaryBufferedBytes;
        }

        public void incrementNonDictionaryBufferedBytes(int value) {
            this.nonDictionaryBufferedBytes += value;
        }
    }

    private static class DictionaryCompressionProjection {
        private final int directConversionIndex;
        private final double predictedFileCompressionRatio;

        public DictionaryCompressionProjection(int directConversionIndex, double predictedFileCompressionRatio) {
            this.directConversionIndex = directConversionIndex;
            this.predictedFileCompressionRatio = predictedFileCompressionRatio;
        }

        public int getDirectConversionCandidateIndex() {
            return this.directConversionIndex;
        }

        public double getPredictedFileCompressionRatio() {
            return this.predictedFileCompressionRatio;
        }
    }

    @VisibleForTesting
    static class DictionaryColumnManager {
        private final DictionaryColumn dictionaryColumn;
        private int rowCount;
        private long pastValueCount;
        private int pastDictionaryEntries;
        private long pendingPastValueCount;
        private int pendingPastDictionaryEntries;

        public DictionaryColumnManager(DictionaryColumn dictionaryColumn) {
            this.dictionaryColumn = dictionaryColumn;
        }

        OptionalInt tryConvertToDirect(int maxDirectBytes) {
            return this.dictionaryColumn.tryConvertToDirect(maxDirectBytes);
        }

        void reset() {
            this.pastValueCount = 0L;
            this.pastDictionaryEntries = 0;
            this.pendingPastValueCount = 0L;
            this.pendingPastDictionaryEntries = 0;
        }

        public void updateHistory(int rowCount) {
            this.rowCount = rowCount;
            long currentValueCount = this.dictionaryColumn.getValueCount();
            if (currentValueCount - this.pendingPastValueCount >= 1024L) {
                this.pastValueCount = this.pendingPastValueCount;
                this.pastDictionaryEntries = this.pendingPastDictionaryEntries;
                this.pendingPastValueCount = currentValueCount;
                this.pendingPastDictionaryEntries = this.dictionaryColumn.getDictionaryEntries();
            }
        }

        public long getRawBytesEstimate() {
            Preconditions.checkState((!this.isDirectEncoded() ? 1 : 0) != 0);
            return this.dictionaryColumn.getRawBytesEstimate();
        }

        public double getRawBytesPerRow() {
            Preconditions.checkState((!this.isDirectEncoded() ? 1 : 0) != 0);
            return 1.0 * (double)this.getRawBytesEstimate() / (double)this.rowCount;
        }

        public int getDictionaryBytes() {
            Preconditions.checkState((!this.isDirectEncoded() ? 1 : 0) != 0);
            return this.dictionaryColumn.getDictionaryBytes();
        }

        public double getDictionaryBytesPerFutureRow() {
            Preconditions.checkState((!this.isDirectEncoded() ? 1 : 0) != 0);
            int currentDictionaryEntries = this.dictionaryColumn.getDictionaryEntries();
            long currentValueCount = this.dictionaryColumn.getValueCount();
            double dictionaryBytesPerEntry = 1.0 * (double)this.dictionaryColumn.getDictionaryBytes() / (double)currentDictionaryEntries;
            double dictionaryEntriesPerFutureValue = 1.0 * (double)(currentDictionaryEntries - this.pastDictionaryEntries) / (double)(currentValueCount - this.pastValueCount);
            return dictionaryBytesPerEntry * dictionaryEntriesPerFutureValue;
        }

        public int getIndexBytes() {
            Preconditions.checkState((!this.isDirectEncoded() ? 1 : 0) != 0);
            return this.dictionaryColumn.getIndexBytes();
        }

        public double getIndexBytesPerRow() {
            Preconditions.checkState((!this.isDirectEncoded() ? 1 : 0) != 0);
            return 1.0 * (double)this.getIndexBytes() / (double)this.rowCount;
        }

        public double getCompressionRatio() {
            Preconditions.checkState((!this.isDirectEncoded() ? 1 : 0) != 0);
            long bufferedBytes = this.getBufferedBytes();
            if (bufferedBytes == 0L) {
                return 0.0;
            }
            return 1.0 * (double)this.getRawBytesEstimate() / (double)bufferedBytes;
        }

        public long getBufferedBytes() {
            return this.dictionaryColumn.getBufferedBytes();
        }

        public boolean isDirectEncoded() {
            return this.dictionaryColumn.isDirectEncoded();
        }

        @VisibleForTesting
        public DictionaryColumn getDictionaryColumn() {
            return this.dictionaryColumn;
        }
    }

    public static interface DictionaryColumn {
        public long getValueCount();

        public long getNonNullValueCount();

        public long getNullValueCount();

        public long getRawBytesEstimate();

        public int getDictionaryEntries();

        public int getDictionaryBytes();

        public int getIndexBytes();

        public OptionalInt tryConvertToDirect(int var1);

        public long getBufferedBytes();

        public boolean isDirectEncoded();
    }
}

