/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.plugin.geospatial;

import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.GeometryCursor;
import com.esri.core.geometry.MultiPath;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.Polygon;
import com.esri.core.geometry.ogc.OGCGeometry;
import com.esri.core.geometry.ogc.OGCGeometryCollection;
import com.esri.core.geometry.ogc.OGCPoint;
import com.facebook.presto.common.block.Block;
import com.facebook.presto.geospatial.GeometryType;
import com.facebook.presto.geospatial.GeometryUtils;
import com.facebook.presto.geospatial.KdbTree;
import com.facebook.presto.geospatial.Rectangle;
import com.facebook.presto.geospatial.SphericalGeographyUtils;
import com.facebook.presto.geospatial.serde.EsriGeometrySerde;
import com.facebook.presto.geospatial.serde.JtsGeometrySerde;
import com.facebook.presto.plugin.geospatial.GeoFunctions;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.function.Description;
import com.facebook.presto.spi.function.ScalarFunction;
import com.facebook.presto.spi.function.SqlNullable;
import com.facebook.presto.spi.function.SqlType;
import com.google.common.base.Preconditions;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.util.EnumSet;

public final class SphericalGeoFunctions {
    private static final EnumSet<Geometry.Type> GEOMETRY_TYPES_FOR_SPHERICAL_GEOGRAPHY = EnumSet.of(Geometry.Type.Point, Geometry.Type.Polyline, Geometry.Type.Polygon, Geometry.Type.MultiPoint);

    private SphericalGeoFunctions() {
    }

    @Description(value="Converts a Geometry object to a SphericalGeography object")
    @ScalarFunction(value="to_spherical_geography")
    @SqlType(value="SphericalGeography")
    public static Slice toSphericalGeography(@SqlType(value="Geometry") Slice input) {
        Geometry subGeometry;
        OGCGeometry geometry;
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)input);
        if (!envelope.isEmpty()) {
            SphericalGeographyUtils.checkLatitude((double)envelope.getYMin());
            SphericalGeographyUtils.checkLatitude((double)envelope.getYMax());
            SphericalGeographyUtils.checkLongitude((double)envelope.getXMin());
            SphericalGeographyUtils.checkLongitude((double)envelope.getXMax());
        }
        if ((geometry = EsriGeometrySerde.deserialize((Slice)input)).is3D()) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Cannot convert 3D geometry to a spherical geography");
        }
        GeometryCursor cursor = geometry.getEsriGeometryCursor();
        while ((subGeometry = cursor.next()) != null) {
            if (GEOMETRY_TYPES_FOR_SPHERICAL_GEOGRAPHY.contains(subGeometry.getType())) continue;
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Cannot convert geometry of this type to spherical geography: " + subGeometry.getType());
        }
        return input;
    }

    @Description(value="Converts a SphericalGeography object to a Geometry object.")
    @ScalarFunction(value="to_geometry")
    @SqlType(value="Geometry")
    public static Slice toGeometry(@SqlType(value="SphericalGeography") Slice input) {
        return input;
    }

    @Description(value="Returns the Well-Known Text (WKT) representation of the geometry")
    @ScalarFunction(value="ST_AsText")
    @SqlType(value="varchar")
    public static Slice stSphericalAsText(@SqlType(value="SphericalGeography") Slice input) {
        return Slices.utf8Slice((String)GeometryUtils.wktFromJtsGeometry((org.locationtech.jts.geom.Geometry)JtsGeometrySerde.deserialize((Slice)input)));
    }

    @SqlNullable
    @Description(value="Returns the great-circle distance in meters between two SphericalGeography points.")
    @ScalarFunction(value="ST_Distance")
    @SqlType(value="double")
    public static Double stSphericalDistance(@SqlType(value="SphericalGeography") Slice left, @SqlType(value="SphericalGeography") Slice right) {
        return SphericalGeographyUtils.sphericalDistance((OGCGeometry)EsriGeometrySerde.deserialize((Slice)left), (OGCGeometry)EsriGeometrySerde.deserialize((Slice)right));
    }

    @SqlNullable
    @Description(value="Returns the area of a geometry on the Earth's surface using spherical model")
    @ScalarFunction(value="ST_Area")
    @SqlType(value="double")
    public static Double stSphericalArea(@SqlType(value="SphericalGeography") Slice input) {
        OGCGeometry geometry = EsriGeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        SphericalGeographyUtils.validateSphericalType((String)"ST_Area", (OGCGeometry)geometry, EnumSet.of(GeometryType.POLYGON, GeometryType.MULTI_POLYGON));
        Polygon polygon = (Polygon)geometry.getEsriGeometry();
        double sphericalExcess = 0.0;
        int numPaths = polygon.getPathCount();
        for (int i = 0; i < numPaths; ++i) {
            double sign = polygon.isExteriorRing(i) ? 1.0 : -1.0;
            sphericalExcess += sign * Math.abs(SphericalGeoFunctions.computeSphericalExcess(polygon, polygon.getPathStart(i), polygon.getPathEnd(i)));
        }
        return Math.abs(sphericalExcess * 6371010.0 * 6371010.0);
    }

    @ScalarFunction
    @Description(value="Calculates the great-circle distance between two points on the Earth's surface in kilometers")
    @SqlType(value="double")
    public static double greatCircleDistance(@SqlType(value="double") double latitude1, @SqlType(value="double") double longitude1, @SqlType(value="double") double latitude2, @SqlType(value="double") double longitude2) {
        return SphericalGeographyUtils.greatCircleDistance((double)latitude1, (double)longitude1, (double)latitude2, (double)longitude2);
    }

    @ScalarFunction
    @SqlNullable
    @Description(value="Returns an array of spatial partition IDs for a given geometry")
    @SqlType(value="array(int)")
    public static Block spatialPartitions(@SqlType(value="KdbTree") Object kdbTree, @SqlType(value="SphericalGeography") Slice geometry) {
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)geometry);
        if (envelope.isEmpty()) {
            return null;
        }
        return GeoFunctions.spatialPartitions((KdbTree)kdbTree, new Rectangle(envelope.getXMin(), envelope.getYMin(), envelope.getXMax(), envelope.getYMax()));
    }

    @ScalarFunction
    @SqlNullable
    @Description(value="Returns an array of spatial partition IDs for a geometry representing a set of points within specified distance from the input geometry")
    @SqlType(value="array(int)")
    public static Block spatialPartitions(@SqlType(value="KdbTree") Object kdbTree, @SqlType(value="SphericalGeography") Slice geometry, @SqlType(value="double") double distance) {
        if (Double.isNaN(distance)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is NaN");
        }
        if (Double.isInfinite(distance)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is infinite");
        }
        if (distance < 0.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is negative");
        }
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)geometry);
        if (envelope.isEmpty()) {
            return null;
        }
        Rectangle expandedEnvelope2D = new Rectangle(envelope.getXMin() - distance, envelope.getYMin() - distance, envelope.getXMax() + distance, envelope.getYMax() + distance);
        return GeoFunctions.spatialPartitions((KdbTree)kdbTree, expandedEnvelope2D);
    }

    @SqlNullable
    @Description(value="Returns the great-circle length in meters of a linestring or multi-linestring on Earth's surface")
    @ScalarFunction(value="ST_Length")
    @SqlType(value="double")
    public static Double stSphericalLength(@SqlType(value="SphericalGeography") Slice input) {
        OGCGeometry geometry = EsriGeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        SphericalGeographyUtils.validateSphericalType((String)"ST_Length", (OGCGeometry)geometry, EnumSet.of(GeometryType.LINE_STRING, GeometryType.MULTI_LINE_STRING));
        MultiPath lineString = (MultiPath)geometry.getEsriGeometry();
        double sum = 0.0;
        for (int path = 0; path < lineString.getPathCount(); ++path) {
            if (lineString.getPathSize(path) < 2) continue;
            int pathStart = lineString.getPathStart(path);
            Point prev = lineString.getPoint(pathStart);
            for (int i = pathStart + 1; i < lineString.getPathEnd(path); ++i) {
                Point next = lineString.getPoint(i);
                sum += SphericalGeoFunctions.greatCircleDistance(prev.getY(), prev.getX(), next.getY(), next.getX());
                prev = next;
            }
        }
        return sum * 1000.0;
    }

    @SqlNullable
    @Description(value="Returns the Point value that is the mathematical centroid of a Spherical Geography")
    @ScalarFunction(value="ST_Centroid")
    @SqlType(value="SphericalGeography")
    public static Slice stSphericalCentroid(@SqlType(value="SphericalGeography") Slice input) {
        Point centroid;
        OGCGeometry geometry = EsriGeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        SphericalGeographyUtils.validateSphericalType((String)"ST_Centroid", (OGCGeometry)geometry, EnumSet.of(GeometryType.POINT, GeometryType.MULTI_POINT));
        if (geometry instanceof OGCPoint) {
            return input;
        }
        OGCGeometryCollection geometryCollection = (OGCGeometryCollection)geometry;
        for (int i = 0; i < geometryCollection.numGeometries(); ++i) {
            OGCGeometry g = geometryCollection.geometryN(i);
            SphericalGeographyUtils.validateSphericalType((String)"ST_Centroid", (OGCGeometry)g, EnumSet.of(GeometryType.POINT));
            Point p = (Point)g.getEsriGeometry();
            SphericalGeographyUtils.checkLongitude((double)p.getX());
            SphericalGeographyUtils.checkLatitude((double)p.getY());
        }
        if (geometryCollection.numGeometries() == 1) {
            centroid = (Point)geometryCollection.geometryN(0).getEsriGeometry();
        } else {
            double x3DTotal = 0.0;
            double y3DTotal = 0.0;
            double z3DTotal = 0.0;
            for (int i = 0; i < geometryCollection.numGeometries(); ++i) {
                SphericalGeographyUtils.CartesianPoint cp = new SphericalGeographyUtils.CartesianPoint((Point)geometryCollection.geometryN(i).getEsriGeometry());
                x3DTotal += cp.getX();
                y3DTotal += cp.getY();
                z3DTotal += cp.getZ();
            }
            double centroidVectorLength = Math.sqrt(x3DTotal * x3DTotal + y3DTotal * y3DTotal + z3DTotal * z3DTotal);
            if (centroidVectorLength == 0.0) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Unexpected error. Average vector length adds to zero (%f, %f, %f)", x3DTotal, y3DTotal, z3DTotal));
            }
            centroid = new SphericalGeographyUtils.CartesianPoint(x3DTotal / centroidVectorLength, y3DTotal / centroidVectorLength, z3DTotal / centroidVectorLength).asSphericalPoint();
        }
        return EsriGeometrySerde.serialize((OGCGeometry)new OGCPoint(centroid, geometryCollection.getEsriSpatialReference()));
    }

    private static double computeSphericalExcess(Polygon polygon, int start, int end) {
        if (polygon.getPoint(end - 1).equals((Object)polygon.getPoint(start))) {
            --end;
        }
        if (end - start < 3) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Polygon is not valid: a loop contains less then 3 vertices.");
        }
        Point point = new Point();
        polygon.getPoint(end - 1, point);
        SphericalExcessCalculator calculator = new SphericalExcessCalculator(point);
        for (int i = start; i < end; ++i) {
            polygon.getPoint(i, point);
            calculator.add(point);
        }
        return calculator.computeSphericalExcess();
    }

    private static class SphericalExcessCalculator {
        private static final double TWO_PI = Math.PI * 2;
        private static final double THREE_PI = Math.PI * 3;
        private double sphericalExcess;
        private double courseDelta;
        private boolean firstPoint;
        private double firstInitialBearing;
        private double previousFinalBearing;
        private double previousPhi;
        private double previousCos;
        private double previousSin;
        private double previousTan;
        private double previousLongitude;
        private boolean done;

        public SphericalExcessCalculator(Point endPoint) {
            this.previousPhi = Math.toRadians(endPoint.getY());
            this.previousSin = Math.sin(this.previousPhi);
            this.previousCos = Math.cos(this.previousPhi);
            this.previousTan = Math.tan(this.previousPhi / 2.0);
            this.previousLongitude = Math.toRadians(endPoint.getX());
            this.firstPoint = true;
        }

        private void add(Point point) throws IllegalStateException {
            Preconditions.checkState((!this.done ? 1 : 0) != 0, (Object)"Computation of spherical excess is complete");
            double phi = Math.toRadians(point.getY());
            double tan = Math.tan(phi / 2.0);
            double longitude = Math.toRadians(point.getX());
            if (longitude == this.previousLongitude && phi == this.previousPhi) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Polygon is not valid: it has two identical consecutive vertices");
            }
            double deltaLongitude = longitude - this.previousLongitude;
            this.sphericalExcess += 2.0 * Math.atan2(Math.tan(deltaLongitude / 2.0) * (this.previousTan + tan), 1.0 + this.previousTan * tan);
            double cos = Math.cos(phi);
            double sin = Math.sin(phi);
            double sinOfDeltaLongitude = Math.sin(deltaLongitude);
            double cosOfDeltaLongitude = Math.cos(deltaLongitude);
            double y = sinOfDeltaLongitude * cos;
            double x = this.previousCos * sin - this.previousSin * cos * cosOfDeltaLongitude;
            double initialBearing = (Math.atan2(y, x) + Math.PI * 2) % (Math.PI * 2);
            double finalY = -sinOfDeltaLongitude * this.previousCos;
            double finalX = this.previousSin * cos - this.previousCos * sin * cosOfDeltaLongitude;
            double finalBearing = (Math.atan2(finalY, finalX) + Math.PI) % (Math.PI * 2);
            if (this.firstPoint) {
                this.firstInitialBearing = initialBearing;
                this.firstPoint = false;
            } else {
                this.courseDelta += (initialBearing - this.previousFinalBearing + Math.PI * 3) % (Math.PI * 2) - Math.PI;
            }
            this.courseDelta += (finalBearing - initialBearing + Math.PI * 3) % (Math.PI * 2) - Math.PI;
            this.previousFinalBearing = finalBearing;
            this.previousCos = cos;
            this.previousSin = sin;
            this.previousPhi = phi;
            this.previousTan = tan;
            this.previousLongitude = longitude;
        }

        public double computeSphericalExcess() {
            if (!this.done) {
                this.courseDelta += (this.firstInitialBearing - this.previousFinalBearing + Math.PI * 3) % (Math.PI * 2) - Math.PI;
                if (Math.abs(this.courseDelta) < 0.7853981633974483) {
                    this.sphericalExcess = Math.abs(this.sphericalExcess) - Math.PI * 2;
                }
                this.done = true;
            }
            return this.sphericalExcess;
        }
    }
}

