/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.query.scheduler;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.RateLimiter;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAccumulator;
import javax.annotation.Nullable;
import org.apache.pinot.common.exception.QueryException;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ServerGauge;
import org.apache.pinot.common.metrics.ServerMeter;
import org.apache.pinot.common.metrics.ServerMetrics;
import org.apache.pinot.common.metrics.ServerQueryPhase;
import org.apache.pinot.common.metrics.ServerTimer;
import org.apache.pinot.common.response.ProcessingException;
import org.apache.pinot.common.utils.DataTable;
import org.apache.pinot.core.common.datatable.DataTableBuilder;
import org.apache.pinot.core.query.executor.QueryExecutor;
import org.apache.pinot.core.query.request.ServerQueryRequest;
import org.apache.pinot.core.query.request.context.TimerContext;
import org.apache.pinot.core.query.scheduler.resources.ResourceManager;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class QueryScheduler {
    private static final Logger LOGGER = LoggerFactory.getLogger(QueryScheduler.class);
    private static final String INVALID_NUM_SCANNED = "-1";
    private static final String INVALID_SEGMENTS_COUNT = "-1";
    private static final String INVALID_FRESHNESS_MS = "-1";
    private static final String INVALID_NUM_RESIZES = "-1";
    private static final String INVALID_RESIZE_TIME_MS = "-1";
    private static final String QUERY_LOG_MAX_RATE_KEY = "query.log.maxRatePerSecond";
    private static final double DEFAULT_QUERY_LOG_MAX_RATE = 10000.0;
    protected final ServerMetrics _serverMetrics;
    protected final QueryExecutor _queryExecutor;
    protected final ResourceManager _resourceManager;
    protected final LongAccumulator _latestQueryTime;
    private final RateLimiter _queryLogRateLimiter;
    private final RateLimiter _numDroppedLogRateLimiter;
    private final AtomicInteger _numDroppedLogCounter;
    protected volatile boolean _isRunning = false;

    public QueryScheduler(PinotConfiguration config, QueryExecutor queryExecutor, ResourceManager resourceManager, ServerMetrics serverMetrics, LongAccumulator latestQueryTime) {
        Preconditions.checkNotNull((Object)config);
        Preconditions.checkNotNull((Object)queryExecutor);
        Preconditions.checkNotNull((Object)resourceManager);
        Preconditions.checkNotNull((Object)serverMetrics);
        Preconditions.checkNotNull((Object)latestQueryTime);
        this._serverMetrics = serverMetrics;
        this._resourceManager = resourceManager;
        this._queryExecutor = queryExecutor;
        this._latestQueryTime = latestQueryTime;
        this._queryLogRateLimiter = RateLimiter.create((double)config.getProperty(QUERY_LOG_MAX_RATE_KEY, 10000.0));
        this._numDroppedLogRateLimiter = RateLimiter.create((double)1.0);
        this._numDroppedLogCounter = new AtomicInteger(0);
        LOGGER.info("Query log max rate: {}", (Object)this._queryLogRateLimiter.getRate());
    }

    public abstract ListenableFuture<byte[]> submit(ServerQueryRequest var1);

    public abstract String name();

    public void start() {
        this._isRunning = true;
    }

    public void stop() {
        this._isRunning = false;
    }

    protected ListenableFutureTask<byte[]> createQueryFutureTask(ServerQueryRequest queryRequest, ExecutorService executorService) {
        return ListenableFutureTask.create(() -> this.processQueryAndSerialize(queryRequest, executorService));
    }

    @Nullable
    protected byte[] processQueryAndSerialize(ServerQueryRequest queryRequest, ExecutorService executorService) {
        DataTable dataTable;
        this._latestQueryTime.accumulate(System.currentTimeMillis());
        try {
            dataTable = this._queryExecutor.processQuery(queryRequest, executorService);
        }
        catch (Exception e) {
            LOGGER.error("Encountered exception while processing requestId {} from broker {}", new Object[]{queryRequest.getRequestId(), queryRequest.getBrokerId(), e});
            this._serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.UNCAUGHT_EXCEPTIONS, 1L);
            dataTable = DataTableBuilder.getEmptyDataTable();
            dataTable.addException(QueryException.getException((ProcessingException)QueryException.INTERNAL_ERROR, (Exception)e));
        }
        long requestId = queryRequest.getRequestId();
        Map dataTableMetadata = dataTable.getMetadata();
        dataTableMetadata.put(DataTable.MetadataKey.REQUEST_ID.getName(), Long.toString(requestId));
        byte[] responseBytes = this.serializeDataTable(queryRequest, dataTable);
        String tableNameWithType = queryRequest.getTableNameWithType();
        long numDocsScanned = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.NUM_DOCS_SCANNED.getName(), "-1"));
        long numEntriesScannedInFilter = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.NUM_ENTRIES_SCANNED_IN_FILTER.getName(), "-1"));
        long numEntriesScannedPostFilter = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.NUM_ENTRIES_SCANNED_POST_FILTER.getName(), "-1"));
        long numSegmentsProcessed = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.NUM_SEGMENTS_PROCESSED.getName(), "-1"));
        long numSegmentsMatched = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.NUM_SEGMENTS_MATCHED.getName(), "-1"));
        long numSegmentsConsuming = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.NUM_CONSUMING_SEGMENTS_PROCESSED.getName(), "-1"));
        long minConsumingFreshnessMs = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.MIN_CONSUMING_FRESHNESS_TIME_MS.getName(), "-1"));
        int numResizes = Integer.parseInt(dataTableMetadata.getOrDefault(DataTable.MetadataKey.NUM_RESIZES.getName(), "-1"));
        long resizeTimeMs = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.RESIZE_TIME_MS.getName(), "-1"));
        long threadCpuTimeNs = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.THREAD_CPU_TIME_NS.getName(), "0"));
        long systemActivitiesCpuTimeNs = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.SYSTEM_ACTIVITIES_CPU_TIME_NS.getName(), "0"));
        long responseSerializationCpuTimeNs = Long.parseLong(dataTableMetadata.getOrDefault(DataTable.MetadataKey.RESPONSE_SER_CPU_TIME_NS.getName(), "0"));
        long totalCpuTimeNs = threadCpuTimeNs + systemActivitiesCpuTimeNs + responseSerializationCpuTimeNs;
        if (numDocsScanned > 0L) {
            this._serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_DOCS_SCANNED, numDocsScanned);
        }
        if (numEntriesScannedInFilter > 0L) {
            this._serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_ENTRIES_SCANNED_IN_FILTER, numEntriesScannedInFilter);
        }
        if (numEntriesScannedPostFilter > 0L) {
            this._serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_ENTRIES_SCANNED_POST_FILTER, numEntriesScannedPostFilter);
        }
        if (numResizes > 0) {
            this._serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_RESIZES, (long)numResizes);
        }
        if (resizeTimeMs > 0L) {
            this._serverMetrics.addValueToTableGauge(tableNameWithType, (AbstractMetrics.Gauge)ServerGauge.RESIZE_TIME_MS, resizeTimeMs);
        }
        if (threadCpuTimeNs > 0L) {
            this._serverMetrics.addTimedTableValue(tableNameWithType, (AbstractMetrics.Timer)ServerTimer.EXECUTION_THREAD_CPU_TIME_NS, threadCpuTimeNs, TimeUnit.NANOSECONDS);
        }
        if (systemActivitiesCpuTimeNs > 0L) {
            this._serverMetrics.addTimedTableValue(tableNameWithType, (AbstractMetrics.Timer)ServerTimer.SYSTEM_ACTIVITIES_CPU_TIME_NS, systemActivitiesCpuTimeNs, TimeUnit.NANOSECONDS);
        }
        if (responseSerializationCpuTimeNs > 0L) {
            this._serverMetrics.addTimedTableValue(tableNameWithType, (AbstractMetrics.Timer)ServerTimer.RESPONSE_SER_CPU_TIME_NS, responseSerializationCpuTimeNs, TimeUnit.NANOSECONDS);
        }
        if (totalCpuTimeNs > 0L) {
            this._serverMetrics.addTimedTableValue(tableNameWithType, (AbstractMetrics.Timer)ServerTimer.TOTAL_CPU_TIME_NS, totalCpuTimeNs, TimeUnit.NANOSECONDS);
        }
        TimerContext timerContext = queryRequest.getTimerContext();
        int numSegmentsQueried = queryRequest.getSegmentsToQuery().size();
        long schedulerWaitMs = timerContext.getPhaseDurationMs(ServerQueryPhase.SCHEDULER_WAIT);
        if (this._queryLogRateLimiter.tryAcquire() || this.forceLog(schedulerWaitMs, numDocsScanned)) {
            int numDroppedLog;
            LOGGER.info("Processed requestId={},table={},segments(queried/processed/matched/consuming)={}/{}/{}/{},schedulerWaitMs={},reqDeserMs={},totalExecMs={},resSerMs={},totalTimeMs={},minConsumingFreshnessMs={},broker={},numDocsScanned={},scanInFilter={},scanPostFilter={},sched={},threadCpuTimeNs(total/thread/sysActivity/resSer)={}/{}/{}/{}", new Object[]{requestId, tableNameWithType, numSegmentsQueried, numSegmentsProcessed, numSegmentsMatched, numSegmentsConsuming, schedulerWaitMs, timerContext.getPhaseDurationMs(ServerQueryPhase.REQUEST_DESERIALIZATION), timerContext.getPhaseDurationMs(ServerQueryPhase.QUERY_PROCESSING), timerContext.getPhaseDurationMs(ServerQueryPhase.RESPONSE_SERIALIZATION), timerContext.getPhaseDurationMs(ServerQueryPhase.TOTAL_QUERY_TIME), minConsumingFreshnessMs, queryRequest.getBrokerId(), numDocsScanned, numEntriesScannedInFilter, numEntriesScannedPostFilter, this.name(), totalCpuTimeNs, threadCpuTimeNs, systemActivitiesCpuTimeNs, responseSerializationCpuTimeNs});
            if (this._numDroppedLogRateLimiter.tryAcquire() && (numDroppedLog = this._numDroppedLogCounter.get()) > 0) {
                LOGGER.info("{} logs were dropped. (log max rate per second: {})", (Object)numDroppedLog, (Object)this._queryLogRateLimiter.getRate());
                this._numDroppedLogCounter.set(0);
            }
        } else {
            this._numDroppedLogCounter.incrementAndGet();
        }
        if (minConsumingFreshnessMs > -1L) {
            this._serverMetrics.addTimedTableValue(tableNameWithType, (AbstractMetrics.Timer)ServerTimer.FRESHNESS_LAG_MS, System.currentTimeMillis() - minConsumingFreshnessMs, TimeUnit.MILLISECONDS);
        }
        this._serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_SEGMENTS_QUERIED, (long)numSegmentsQueried);
        this._serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_SEGMENTS_PROCESSED, numSegmentsProcessed);
        this._serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_SEGMENTS_MATCHED, numSegmentsMatched);
        return responseBytes;
    }

    private boolean forceLog(long schedulerWaitMs, long numDocsScanned) {
        if (schedulerWaitMs > 100L) {
            return true;
        }
        return numDocsScanned > 1000000L;
    }

    @Nullable
    private byte[] serializeDataTable(ServerQueryRequest queryRequest, DataTable dataTable) {
        TimerContext timerContext = queryRequest.getTimerContext();
        TimerContext.Timer responseSerializationTimer = timerContext.startNewPhaseTimer(ServerQueryPhase.RESPONSE_SERIALIZATION);
        byte[] responseByte = null;
        try {
            responseByte = dataTable.toBytes();
        }
        catch (Exception e) {
            this._serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.RESPONSE_SERIALIZATION_EXCEPTIONS, 1L);
            LOGGER.error("Caught exception while serializing response for requestId: {}, brokerId: {}", new Object[]{queryRequest.getRequestId(), queryRequest.getBrokerId(), e});
        }
        responseSerializationTimer.stopAndRecord();
        timerContext.startNewPhaseTimer(ServerQueryPhase.TOTAL_QUERY_TIME, timerContext.getQueryArrivalTimeMs()).stopAndRecord();
        return responseByte;
    }

    protected ListenableFuture<byte[]> immediateErrorResponse(ServerQueryRequest queryRequest, ProcessingException error) {
        DataTable result = DataTableBuilder.getEmptyDataTable();
        Map dataTableMetadata = result.getMetadata();
        dataTableMetadata.put(DataTable.MetadataKey.REQUEST_ID.getName(), Long.toString(queryRequest.getRequestId()));
        result.addException(error);
        return Futures.immediateFuture((Object)this.serializeDataTable(queryRequest, result));
    }
}

