/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.operator.scalar;

import com.facebook.presto.operator.Description;
import com.facebook.presto.operator.scalar.ScalarFunction;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.sql.analyzer.Session;
import com.facebook.presto.util.ThreadLocalCache;
import com.google.common.base.Charsets;
import com.google.common.primitives.Ints;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.nio.charset.Charset;
import java.util.concurrent.TimeUnit;
import org.joda.time.DateTimeField;
import org.joda.time.DateTimeZone;
import org.joda.time.chrono.ISOChronology;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;

public final class UnixTimeFunctions {
    private static final ThreadLocalCache<Slice, DateTimeFormatter> DATETIME_FORMATTER_CACHE = new ThreadLocalCache<Slice, DateTimeFormatter>(100){

        @Override
        protected DateTimeFormatter load(Slice format) {
            return UnixTimeFunctions.createDateTimeFormatter(format);
        }
    };
    private static final ISOChronology UTC_CHRONOLOGY = ISOChronology.getInstance((DateTimeZone)DateTimeZone.UTC);
    private static final DateTimeField SECOND_OF_MINUTE = UTC_CHRONOLOGY.secondOfMinute();
    private static final DateTimeField MINUTE_OF_HOUR = UTC_CHRONOLOGY.minuteOfHour();
    private static final DateTimeField HOUR_OF_DAY = UTC_CHRONOLOGY.hourOfDay();
    private static final DateTimeField DAY_OF_WEEK = UTC_CHRONOLOGY.dayOfWeek();
    private static final DateTimeField DAY_OF_MONTH = UTC_CHRONOLOGY.dayOfMonth();
    private static final DateTimeField DAY_OF_YEAR = UTC_CHRONOLOGY.dayOfYear();
    private static final DateTimeField WEEK_OF_YEAR = UTC_CHRONOLOGY.weekOfWeekyear();
    private static final DateTimeField MONTH_OF_YEAR = UTC_CHRONOLOGY.monthOfYear();
    private static final DateTimeField YEAR = UTC_CHRONOLOGY.year();
    private static final DateTimeField CENTURY = UTC_CHRONOLOGY.centuryOfEra();

    private UnixTimeFunctions() {
    }

    @Description(value="s/system/current/")
    @ScalarFunction(value="now")
    public static long currentTimestamp(Session session) {
        return UnixTimeFunctions.fromMillis(session.getStartTime());
    }

    @ScalarFunction(value="from_unixtime")
    public static long fromUnixTime(double unixTime) {
        return Math.round(unixTime);
    }

    @ScalarFunction(value="to_unixtime")
    public static double toUnixTime(long unixTime) {
        return unixTime;
    }

    @Description(value="add the specified amount of time to the given time")
    @ScalarFunction
    public static long dateAdd(Slice unit, long value, long unixTime) {
        return UnixTimeFunctions.fromMillis(UnixTimeFunctions.internalDateAdd(unit, value, UnixTimeFunctions.toMillis(unixTime)));
    }

    private static long internalDateAdd(Slice unit, long value, long unixTime) {
        String unitString = unit.toString(Charsets.US_ASCII).toLowerCase();
        int intValue = Ints.checkedCast((long)value);
        switch (unitString) {
            case "second": {
                return SECOND_OF_MINUTE.add(unixTime, intValue);
            }
            case "minute": {
                return MINUTE_OF_HOUR.add(unixTime, intValue);
            }
            case "hour": {
                return HOUR_OF_DAY.add(unixTime, intValue);
            }
            case "day": {
                return DAY_OF_MONTH.add(unixTime, intValue);
            }
            case "week": {
                return WEEK_OF_YEAR.add(unixTime, intValue);
            }
            case "month": {
                return MONTH_OF_YEAR.add(unixTime, intValue);
            }
            case "quarter": {
                return MONTH_OF_YEAR.add(unixTime, intValue * 3);
            }
            case "year": {
                return YEAR.add(unixTime, intValue);
            }
            case "century": {
                return CENTURY.add(unixTime, intValue);
            }
        }
        throw new IllegalArgumentException("Unsupported unit " + unitString);
    }

    @Description(value="difference of the given times in the given unit")
    @ScalarFunction
    public static long dateDiff(Slice unit, long unixTime1, long unixTime2) {
        String unitString = unit.toString(Charsets.US_ASCII).toLowerCase();
        unixTime1 = UnixTimeFunctions.toMillis(unixTime1);
        unixTime2 = UnixTimeFunctions.toMillis(unixTime2);
        switch (unitString) {
            case "second": {
                return SECOND_OF_MINUTE.getDifference(unixTime2, unixTime1);
            }
            case "minute": {
                return MINUTE_OF_HOUR.getDifference(unixTime2, unixTime1);
            }
            case "hour": {
                return HOUR_OF_DAY.getDifference(unixTime2, unixTime1);
            }
            case "day": {
                return DAY_OF_MONTH.getDifference(unixTime2, unixTime1);
            }
            case "week": {
                return WEEK_OF_YEAR.getDifference(unixTime2, unixTime1);
            }
            case "month": {
                return MONTH_OF_YEAR.getDifference(unixTime2, unixTime1);
            }
            case "quarter": {
                return MONTH_OF_YEAR.getDifference(unixTime2, unixTime1) / 4 + 1;
            }
            case "year": {
                return YEAR.getDifference(unixTime2, unixTime1);
            }
            case "century": {
                return CENTURY.getDifference(unixTime2, unixTime1);
            }
        }
        throw new IllegalArgumentException("Unsupported unit " + unitString);
    }

    @Description(value="parses the specified date/time by the given format")
    @ScalarFunction
    public static long parseDatetime(Slice datetime, Slice formatString) {
        String pattern = formatString.toString(Charsets.UTF_8);
        DateTimeFormatter formatter = DateTimeFormat.forPattern((String)pattern).withZoneUTC();
        String datetimeString = datetime.toString(Charsets.UTF_8);
        return UnixTimeFunctions.fromMillis(formatter.parseMillis(datetimeString));
    }

    @Description(value="formats the given time by the given format")
    @ScalarFunction
    public static Slice formatDatetime(long unixTime, Slice formatString) {
        String pattern = formatString.toString(Charsets.UTF_8);
        DateTimeFormatter formatter = DateTimeFormat.forPattern((String)pattern).withZoneUTC();
        String datetimeString = formatter.print(UnixTimeFunctions.toMillis(unixTime));
        return Slices.wrappedBuffer((byte[])datetimeString.getBytes(Charsets.UTF_8));
    }

    @ScalarFunction
    public static Slice dateFormat(long unixTime, Slice formatString) {
        DateTimeFormatter formatter = DATETIME_FORMATTER_CACHE.get(formatString);
        return Slices.copiedBuffer((String)formatter.print(UnixTimeFunctions.toMillis(unixTime)), (Charset)Charsets.UTF_8);
    }

    @ScalarFunction
    public static long dateParse(Slice dateTime, Slice formatString) {
        DateTimeFormatter formatter = DATETIME_FORMATTER_CACHE.get(formatString);
        return UnixTimeFunctions.fromMillis(formatter.parseMillis(dateTime.toString(Charsets.UTF_8)));
    }

    @Description(value="second of the minute of the given time")
    @ScalarFunction
    public static long second(long unixTime) {
        return SECOND_OF_MINUTE.get(UnixTimeFunctions.toMillis(unixTime));
    }

    @Description(value="minute of the hour of the given time")
    @ScalarFunction
    public static long minute(long unixTime) {
        return MINUTE_OF_HOUR.get(UnixTimeFunctions.toMillis(unixTime));
    }

    @Description(value="hour of the day of the given time")
    @ScalarFunction
    public static long hour(long unixTime) {
        return HOUR_OF_DAY.get(UnixTimeFunctions.toMillis(unixTime));
    }

    @Description(value="day of the week of the given time")
    @ScalarFunction(alias={"dow"})
    public static long dayOfWeek(long unixTime) {
        return DAY_OF_WEEK.get(UnixTimeFunctions.toMillis(unixTime));
    }

    @Description(value="day of the month of the given time")
    @ScalarFunction(alias={"day_of_month"})
    public static long day(long unixTime) {
        return DAY_OF_MONTH.get(UnixTimeFunctions.toMillis(unixTime));
    }

    @Description(value="day of the year of the given time")
    @ScalarFunction(alias={"doy"})
    public static long dayOfYear(long unixTime) {
        return DAY_OF_YEAR.get(UnixTimeFunctions.toMillis(unixTime));
    }

    @Description(value="week of the year of the given time")
    @ScalarFunction(alias={"week_of_year"})
    public static long week(long unixTime) {
        return WEEK_OF_YEAR.get(UnixTimeFunctions.toMillis(unixTime));
    }

    @Description(value="month of the year of the given time")
    @ScalarFunction
    public static long month(long unixTime) {
        return MONTH_OF_YEAR.get(UnixTimeFunctions.toMillis(unixTime));
    }

    @Description(value="quarter of the year of the given time")
    @ScalarFunction
    public static long quarter(long unixTime) {
        return MONTH_OF_YEAR.get(UnixTimeFunctions.toMillis(unixTime)) / 4 + 1;
    }

    @Description(value="year of the given time")
    @ScalarFunction
    public static long year(long unixTime) {
        return YEAR.get(UnixTimeFunctions.toMillis(unixTime));
    }

    @Description(value="century of the given time")
    @ScalarFunction
    public static long century(long unixTime) {
        return CENTURY.get(UnixTimeFunctions.toMillis(unixTime));
    }

    private static long toMillis(long seconds) {
        return TimeUnit.SECONDS.toMillis(seconds);
    }

    private static long fromMillis(long millis) {
        return TimeUnit.MILLISECONDS.toSeconds(millis);
    }

    public static DateTimeFormatter createDateTimeFormatter(Slice format) {
        DateTimeFormatterBuilder builder = new DateTimeFormatterBuilder();
        String formatString = format.toString(Charsets.UTF_8);
        boolean escaped = false;
        for (int i = 0; i < format.length(); ++i) {
            char character = formatString.charAt(i);
            if (escaped) {
                switch (character) {
                    case 'a': {
                        builder.appendDayOfWeekShortText();
                        break;
                    }
                    case 'b': {
                        builder.appendMonthOfYearShortText();
                        break;
                    }
                    case 'c': {
                        builder.appendMonthOfYear(1);
                        break;
                    }
                    case 'd': {
                        builder.appendDayOfMonth(2);
                        break;
                    }
                    case 'e': {
                        builder.appendDayOfMonth(1);
                        break;
                    }
                    case 'f': {
                        builder.appendMillisOfSecond(6);
                        break;
                    }
                    case 'H': {
                        builder.appendHourOfDay(2);
                        break;
                    }
                    case 'I': 
                    case 'h': {
                        builder.appendClockhourOfHalfday(2);
                        break;
                    }
                    case 'i': {
                        builder.appendMinuteOfHour(2);
                        break;
                    }
                    case 'j': {
                        builder.appendDayOfYear(3);
                        break;
                    }
                    case 'k': {
                        builder.appendClockhourOfDay(1);
                        break;
                    }
                    case 'l': {
                        builder.appendClockhourOfHalfday(1);
                        break;
                    }
                    case 'M': {
                        builder.appendMonthOfYearText();
                        break;
                    }
                    case 'm': {
                        builder.appendMonthOfYear(2);
                        break;
                    }
                    case 'p': {
                        builder.appendHalfdayOfDayText();
                        break;
                    }
                    case 'r': {
                        builder.appendClockhourOfHalfday(2).appendLiteral(':').appendMinuteOfHour(2).appendLiteral(':').appendSecondOfMinute(2).appendLiteral(' ').appendHalfdayOfDayText();
                        break;
                    }
                    case 'S': 
                    case 's': {
                        builder.appendSecondOfMinute(2);
                        break;
                    }
                    case 'T': {
                        builder.appendHourOfDay(2).appendLiteral(':').appendMinuteOfHour(2).appendLiteral(':').appendSecondOfMinute(2);
                        break;
                    }
                    case 'v': {
                        builder.appendWeekOfWeekyear(2);
                        break;
                    }
                    case 'W': {
                        builder.appendDayOfWeekText();
                        break;
                    }
                    case 'w': {
                        builder.appendDayOfWeek(1);
                        break;
                    }
                    case 'Y': {
                        builder.appendYear(4, 4);
                        break;
                    }
                    case 'y': {
                        builder.appendYearOfCentury(2, 2);
                        break;
                    }
                    case 'D': 
                    case 'U': 
                    case 'V': 
                    case 'X': 
                    case 'u': 
                    case 'x': {
                        throw new PrestoException(StandardErrorCode.INVALID_FUNCTION_ARGUMENT.toErrorCode(), String.format("%%%s not supported in date format string", Character.valueOf(character)));
                    }
                    case '%': {
                        builder.appendLiteral('%');
                        break;
                    }
                    default: {
                        builder.appendLiteral(character);
                    }
                }
                escaped = false;
                continue;
            }
            if (character == '%') {
                escaped = true;
                continue;
            }
            builder.appendLiteral(character);
        }
        return builder.toFormatter();
    }
}

