/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.discovery.store;

import com.google.common.base.Preconditions;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import org.weakref.jmx.Managed;

public class BatchProcessor<T> {
    private static final Logger log = Logger.get(BatchProcessor.class);
    private final BatchHandler<T> handler;
    private final int maxBatchSize;
    private final BlockingQueue<T> queue;
    private final String name;
    private ExecutorService executor;
    private volatile Future<?> future;
    private final AtomicLong processedEntries = new AtomicLong();
    private final AtomicLong droppedEntries = new AtomicLong();
    private final AtomicLong errors = new AtomicLong();

    public BatchProcessor(String name, BatchHandler<T> handler, int maxBatchSize, int queueSize) {
        Preconditions.checkNotNull((Object)name, (Object)"name is null");
        Preconditions.checkNotNull(handler, (Object)"handler is null");
        Preconditions.checkArgument((queueSize > 0 ? 1 : 0) != 0, (Object)"queue size needs to be a positive integer");
        Preconditions.checkArgument((maxBatchSize > 0 ? 1 : 0) != 0, (Object)"max batch size needs to be a positive integer");
        this.name = name;
        this.handler = handler;
        this.maxBatchSize = maxBatchSize;
        this.queue = new ArrayBlockingQueue<T>(queueSize);
    }

    @PostConstruct
    public synchronized void start() {
        if (this.future == null) {
            this.executor = Executors.newSingleThreadExecutor(Threads.threadsNamed((String)("batch-processor-" + this.name + "-%d")));
            this.future = this.executor.submit(new Runnable(){

                @Override
                public void run() {
                    while (!Thread.interrupted()) {
                        ArrayList entries = new ArrayList(BatchProcessor.this.maxBatchSize);
                        try {
                            Object first = BatchProcessor.this.queue.take();
                            entries.add(first);
                            BatchProcessor.this.queue.drainTo(entries, BatchProcessor.this.maxBatchSize - 1);
                            BatchProcessor.this.handler.processBatch(Collections.unmodifiableList(entries));
                            BatchProcessor.this.processedEntries.addAndGet(entries.size());
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        }
                        catch (Throwable t) {
                            BatchProcessor.this.errors.incrementAndGet();
                            log.warn(t, "Error handling batch");
                        }
                    }
                }
            });
        }
    }

    @Managed
    public long getProcessedEntries() {
        return this.processedEntries.get();
    }

    @Managed
    public long getDroppedEntries() {
        return this.droppedEntries.get();
    }

    @Managed
    public long getErrors() {
        return this.errors.get();
    }

    @Managed
    public long getQueueSize() {
        return this.queue.size();
    }

    @PreDestroy
    public synchronized void stop() {
        if (this.future != null) {
            this.future.cancel(true);
            this.executor.shutdownNow();
            this.future = null;
        }
    }

    public void put(T entry) {
        Preconditions.checkState((!this.future.isCancelled() ? 1 : 0) != 0, (Object)"Processor is not running");
        Preconditions.checkNotNull(entry, (Object)"entry is null");
        while (!this.queue.offer(entry)) {
            if (this.queue.poll() == null) continue;
            this.droppedEntries.incrementAndGet();
        }
    }

    public static interface BatchHandler<T> {
        public void processBatch(Collection<T> var1);
    }
}

