/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.plugin.geospatial;

import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.GeometryCursor;
import com.esri.core.geometry.GeometryException;
import com.esri.core.geometry.ListeningGeometryCursor;
import com.esri.core.geometry.NonSimpleResult;
import com.esri.core.geometry.OperatorUnion;
import com.esri.core.geometry.Polyline;
import com.esri.core.geometry.ogc.OGCConcreteGeometryCollection;
import com.esri.core.geometry.ogc.OGCGeometry;
import com.esri.core.geometry.ogc.OGCGeometryCollection;
import com.esri.core.geometry.ogc.OGCLineString;
import com.facebook.presto.geospatial.GeometryUtils;
import com.facebook.presto.geospatial.KdbTree;
import com.facebook.presto.geospatial.Rectangle;
import com.facebook.presto.geospatial.serde.EsriGeometrySerde;
import com.facebook.presto.geospatial.serde.GeometrySerializationType;
import com.facebook.presto.geospatial.serde.JtsGeometrySerde;
import com.facebook.presto.plugin.geospatial.GeometryType;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.block.Block;
import com.facebook.presto.spi.block.BlockBuilder;
import com.facebook.presto.spi.function.Description;
import com.facebook.presto.spi.function.ScalarFunction;
import com.facebook.presto.spi.function.SqlNullable;
import com.facebook.presto.spi.function.SqlType;
import com.facebook.presto.spi.type.IntegerType;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Deque;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Set;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.TopologyException;
import org.locationtech.jts.geom.impl.PackedCoordinateSequenceFactory;
import org.locationtech.jts.linearref.LengthIndexedLine;
import org.locationtech.jts.simplify.TopologyPreservingSimplifier;

public final class GeoFunctions {
    private static final Joiner OR_JOINER = Joiner.on((String)" or ");
    private static final Slice EMPTY_POLYGON = JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.createJtsEmptyPolygon());
    private static final Map<NonSimpleResult.Reason, String> NON_SIMPLE_REASONS = ImmutableMap.builder().put((Object)NonSimpleResult.Reason.DegenerateSegments, (Object)"Degenerate segments").put((Object)NonSimpleResult.Reason.Clustering, (Object)"Repeated points").put((Object)NonSimpleResult.Reason.Cracking, (Object)"Intersecting or overlapping segments").put((Object)NonSimpleResult.Reason.CrossOver, (Object)"Self-intersection").put((Object)NonSimpleResult.Reason.OGCPolylineSelfTangency, (Object)"Self-tangency").put((Object)NonSimpleResult.Reason.OGCPolygonSelfTangency, (Object)"Self-tangency").put((Object)NonSimpleResult.Reason.OGCDisconnectedInterior, (Object)"Disconnected interior").build();
    private static final int NUMBER_OF_DIMENSIONS = 3;
    private static final Block EMPTY_ARRAY_OF_INTS = IntegerType.INTEGER.createFixedSizeBlockBuilder(0).build();

    private GeoFunctions() {
    }

    @Description(value="Returns a Geometry type LineString object from Well-Known Text representation (WKT)")
    @ScalarFunction(value="ST_LineFromText")
    @SqlType(value="Geometry")
    public static Slice parseLine(@SqlType(value="varchar") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = GeometryUtils.jtsGeometryFromWkt((String)input.toStringUtf8());
        GeoFunctions.validateType("ST_LineFromText", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.LINE_STRING));
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometry);
    }

    @Description(value="Returns a LineString from an array of points")
    @ScalarFunction(value="ST_LineString")
    @SqlType(value="Geometry")
    public static Slice stLineString(@SqlType(value="array(Geometry)") Block input) {
        CoordinateSequence coordinates = GeoFunctions.readPointCoordinates(input, "ST_LineString", true);
        if (coordinates.size() < 2) {
            return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.createJtsEmptyLineString());
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.createJtsLineString((CoordinateSequence)coordinates));
    }

    @Description(value="Returns a Geometry type Point object with the given coordinate values")
    @ScalarFunction(value="ST_Point")
    @SqlType(value="Geometry")
    public static Slice stPoint(@SqlType(value="double") double x, @SqlType(value="double") double y) {
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.createJtsPoint((double)x, (double)y));
    }

    @SqlNullable
    @Description(value="Returns a multi-point geometry formed from input points")
    @ScalarFunction(value="ST_MultiPoint")
    @SqlType(value="Geometry")
    public static Slice stMultiPoint(@SqlType(value="array(Geometry)") Block input) {
        CoordinateSequence coordinates = GeoFunctions.readPointCoordinates(input, "ST_MultiPoint", false);
        if (coordinates.size() == 0) {
            return null;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.createJtsMultiPoint((CoordinateSequence)coordinates));
    }

    private static CoordinateSequence readPointCoordinates(Block input, String functionName, boolean forbidDuplicates) {
        PackedCoordinateSequenceFactory coordinateSequenceFactory = new PackedCoordinateSequenceFactory();
        double[] coordinates = new double[2 * input.getPositionCount()];
        double lastX = Double.NaN;
        double lastY = Double.NaN;
        for (int i = 0; i < input.getPositionCount(); ++i) {
            if (input.isNull(i)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to %s: null at index %s", functionName, i + 1));
            }
            BasicSliceInput slice = new BasicSliceInput(GeometryType.GEOMETRY.getSlice(input, i));
            GeometrySerializationType type = GeometrySerializationType.getForCode((int)slice.readByte());
            if (type != GeometrySerializationType.POINT) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to %s: geometry is not a point: %s at index %s", functionName, type.toString(), i + 1));
            }
            double x = slice.readDouble();
            double y = slice.readDouble();
            if (Double.isNaN(x) || Double.isNaN(x)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to %s: empty point at index %s", functionName, i + 1));
            }
            if (forbidDuplicates && x == lastX && y == lastY) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to %s: consecutive duplicate points at index %s", functionName, i + 1));
            }
            lastX = x;
            lastY = y;
            coordinates[2 * i] = x;
            coordinates[2 * i + 1] = y;
        }
        return coordinateSequenceFactory.create(coordinates, 2);
    }

    @Description(value="Returns a Geometry type Polygon object from Well-Known Text representation (WKT)")
    @ScalarFunction(value="ST_Polygon")
    @SqlType(value="Geometry")
    public static Slice stPolygon(@SqlType(value="varchar") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = GeometryUtils.jtsGeometryFromWkt((String)input.toStringUtf8());
        GeoFunctions.validateType("ST_Polygon", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.POLYGON));
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometry);
    }

    @Description(value="Returns the 2D Euclidean area of a geometry")
    @ScalarFunction(value="ST_Area")
    @SqlType(value="double")
    public static double stArea(@SqlType(value="Geometry") Slice input) {
        return JtsGeometrySerde.deserialize((Slice)input).getArea();
    }

    @Description(value="Returns a Geometry type object from Well-Known Text representation (WKT)")
    @ScalarFunction(value="ST_GeometryFromText")
    @SqlType(value="Geometry")
    public static Slice stGeometryFromText(@SqlType(value="varchar") Slice input) {
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.jtsGeometryFromWkt((String)input.toStringUtf8()));
    }

    @Description(value="Returns a Geometry type object from Well-Known Binary representation (WKB)")
    @ScalarFunction(value="ST_GeomFromBinary")
    @SqlType(value="Geometry")
    public static Slice stGeomFromBinary(@SqlType(value="varbinary") Slice input) {
        return EsriGeometrySerde.serialize((OGCGeometry)GeoFunctions.geomFromBinary(input));
    }

    @Description(value="Returns the Well-Known Text (WKT) representation of the geometry")
    @ScalarFunction(value="ST_AsText")
    @SqlType(value="varchar")
    public static Slice stAsText(@SqlType(value="Geometry") Slice input) {
        return Slices.utf8Slice((String)GeometryUtils.wktFromJtsGeometry((org.locationtech.jts.geom.Geometry)JtsGeometrySerde.deserialize((Slice)input)));
    }

    @Description(value="Returns the Well-Known Binary (WKB) representation of the geometry")
    @ScalarFunction(value="ST_AsBinary")
    @SqlType(value="varbinary")
    public static Slice stAsBinary(@SqlType(value="Geometry") Slice input) {
        try {
            return Slices.wrappedBuffer((ByteBuffer)EsriGeometrySerde.deserialize((Slice)input).asBinary());
        }
        catch (GeometryException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid geometry: " + e.getMessage(), (Throwable)e);
        }
    }

    @SqlNullable
    @Description(value="Returns the geometry that represents all points whose distance from the specified geometry is less than or equal to the specified distance")
    @ScalarFunction(value="ST_Buffer")
    @SqlType(value="Geometry")
    public static Slice stBuffer(@SqlType(value="Geometry") Slice input, @SqlType(value="double") double distance) {
        if (Double.isNaN(distance)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is NaN");
        }
        if (distance < 0.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is negative");
        }
        if (distance == 0.0) {
            return input;
        }
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometry.buffer(distance));
    }

    @Description(value="Returns the Point value that is the mathematical centroid of a Geometry")
    @ScalarFunction(value="ST_Centroid")
    @SqlType(value="Geometry")
    public static Slice stCentroid(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Centroid", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.POINT, new com.facebook.presto.geospatial.GeometryType[]{com.facebook.presto.geospatial.GeometryType.MULTI_POINT, com.facebook.presto.geospatial.GeometryType.LINE_STRING, com.facebook.presto.geospatial.GeometryType.MULTI_LINE_STRING, com.facebook.presto.geospatial.GeometryType.POLYGON, com.facebook.presto.geospatial.GeometryType.MULTI_POLYGON}));
        com.facebook.presto.geospatial.GeometryType geometryType = com.facebook.presto.geospatial.GeometryType.getForJtsGeometryType((String)geometry.getGeometryType());
        if (geometryType == com.facebook.presto.geospatial.GeometryType.POINT) {
            return input;
        }
        if (geometry.getNumPoints() == 0) {
            return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.createJtsEmptyPoint());
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometry.getCentroid());
    }

    @Description(value="Returns the minimum convex geometry that encloses all input geometries")
    @ScalarFunction(value="ST_ConvexHull")
    @SqlType(value="Geometry")
    public static Slice stConvexHull(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = EsriGeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return input;
        }
        if (com.facebook.presto.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType()) == com.facebook.presto.geospatial.GeometryType.POINT) {
            return input;
        }
        return EsriGeometrySerde.serialize((OGCGeometry)geometry.convexHull());
    }

    @Description(value="Return the coordinate dimension of the Geometry")
    @ScalarFunction(value="ST_CoordDim")
    @SqlType(value="tinyint")
    public static long stCoordinateDimension(@SqlType(value="Geometry") Slice input) {
        return EsriGeometrySerde.deserialize((Slice)input).coordinateDimension();
    }

    @Description(value="Returns the inherent dimension of this Geometry object, which must be less than or equal to the coordinate dimension")
    @ScalarFunction(value="ST_Dimension")
    @SqlType(value="tinyint")
    public static long stDimension(@SqlType(value="Geometry") Slice input) {
        return JtsGeometrySerde.deserialize((Slice)input).getDimension();
    }

    @SqlNullable
    @Description(value="Returns TRUE if the LineString or Multi-LineString's start and end points are coincident")
    @ScalarFunction(value="ST_IsClosed")
    @SqlType(value="boolean")
    public static Boolean stIsClosed(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_IsClosed", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.LINE_STRING, com.facebook.presto.geospatial.GeometryType.MULTI_LINE_STRING));
        if (geometry instanceof LineString) {
            return ((LineString)geometry).isClosed();
        }
        if (geometry instanceof MultiLineString) {
            return ((MultiLineString)geometry).isClosed();
        }
        throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid type for isClosed: %s", geometry.getGeometryType()));
    }

    @SqlNullable
    @Description(value="Returns TRUE if this Geometry is an empty geometrycollection, polygon, point etc")
    @ScalarFunction(value="ST_IsEmpty")
    @SqlType(value="boolean")
    public static Boolean stIsEmpty(@SqlType(value="Geometry") Slice input) {
        return EsriGeometrySerde.deserializeEnvelope((Slice)input).isEmpty();
    }

    @Description(value="Returns TRUE if this Geometry has no anomalous geometric points, such as self intersection or self tangency")
    @ScalarFunction(value="ST_IsSimple")
    @SqlType(value="boolean")
    public static boolean stIsSimple(@SqlType(value="Geometry") Slice input) {
        try {
            return JtsGeometrySerde.deserialize((Slice)input).isSimple();
        }
        catch (PrestoException e) {
            if (e.getCause() instanceof TopologyException) {
                return false;
            }
            throw e;
        }
    }

    @Description(value="Returns true if the input geometry is well formed")
    @ScalarFunction(value="ST_IsValid")
    @SqlType(value="boolean")
    public static boolean stIsValid(@SqlType(value="Geometry") Slice input) {
        try {
            return JtsGeometrySerde.deserialize((Slice)input).isValid();
        }
        catch (PrestoException e) {
            if (e.getCause() instanceof TopologyException) {
                return false;
            }
            throw e;
        }
    }

    @Description(value="Returns the reason for why the input geometry is not valid. Returns null if the input is valid.")
    @ScalarFunction(value="geometry_invalid_reason")
    @SqlType(value="varchar")
    @SqlNullable
    public static Slice invalidReason(@SqlType(value="Geometry") Slice input) {
        try {
            org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
            return Slices.utf8Slice((String)GeometryUtils.getGeometryInvalidReason((org.locationtech.jts.geom.Geometry)geometry));
        }
        catch (PrestoException e) {
            if (e.getCause() instanceof TopologyException) {
                return Slices.utf8Slice((String)e.getMessage());
            }
            throw e;
        }
    }

    @Description(value="Returns the length of a LineString or Multi-LineString using Euclidean measurement on a 2D plane (based on spatial ref) in projected units")
    @ScalarFunction(value="ST_Length")
    @SqlType(value="double")
    public static double stLength(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Length", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.LINE_STRING, com.facebook.presto.geospatial.GeometryType.MULTI_LINE_STRING));
        return geometry.getLength();
    }

    @SqlNullable
    @Description(value="Returns a float between 0 and 1 representing the location of the closest point on the LineString to the given Point, as a fraction of total 2d line length.")
    @ScalarFunction(value="line_locate_point")
    @SqlType(value="double")
    public static Double lineLocatePoint(@SqlType(value="Geometry") Slice lineSlice, @SqlType(value="Geometry") Slice pointSlice) {
        org.locationtech.jts.geom.Geometry line = JtsGeometrySerde.deserialize((Slice)lineSlice);
        org.locationtech.jts.geom.Geometry point = JtsGeometrySerde.deserialize((Slice)pointSlice);
        if (line.isEmpty() || point.isEmpty()) {
            return null;
        }
        com.facebook.presto.geospatial.GeometryType lineType = com.facebook.presto.geospatial.GeometryType.getForJtsGeometryType((String)line.getGeometryType());
        if (lineType != com.facebook.presto.geospatial.GeometryType.LINE_STRING && lineType != com.facebook.presto.geospatial.GeometryType.MULTI_LINE_STRING) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("First argument to line_locate_point must be a LineString or a MultiLineString. Got: %s", line.getGeometryType()));
        }
        com.facebook.presto.geospatial.GeometryType pointType = com.facebook.presto.geospatial.GeometryType.getForJtsGeometryType((String)point.getGeometryType());
        if (pointType != com.facebook.presto.geospatial.GeometryType.POINT) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Second argument to line_locate_point must be a Point. Got: %s", point.getGeometryType()));
        }
        return new LengthIndexedLine(line).indexOf(point.getCoordinate()) / line.getLength();
    }

    @Description(value="Returns the point in the line at the fractional length.")
    @ScalarFunction(value="line_interpolate_point")
    @SqlType(value="Geometry")
    public static Slice lineInterpolatePoint(@SqlType(value="Geometry") Slice lineSlice, @SqlType(value="double") double fraction) {
        if (!(0.0 <= fraction) || !(fraction <= 1.0)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("line_interpolate_point: Fraction must be between 0 and 1, but is %s", fraction));
        }
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)lineSlice);
        GeoFunctions.validateType("line_interpolate_point", geometry, (Set<com.facebook.presto.geospatial.GeometryType>)ImmutableSet.of((Object)com.facebook.presto.geospatial.GeometryType.LINE_STRING));
        LineString line = (LineString)geometry;
        if (line.isEmpty()) {
            return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.createJtsEmptyPoint());
        }
        Coordinate coordinate = new LengthIndexedLine((org.locationtech.jts.geom.Geometry)line).extractPoint(fraction * line.getLength());
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.createJtsPoint((Coordinate)coordinate));
    }

    @SqlNullable
    @Description(value="Returns X maxima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_XMax")
    @SqlType(value="double")
    public static Double stXMax(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        return envelope.getXMax();
    }

    @SqlNullable
    @Description(value="Returns Y maxima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_YMax")
    @SqlType(value="double")
    public static Double stYMax(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        return envelope.getYMax();
    }

    @SqlNullable
    @Description(value="Returns X minima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_XMin")
    @SqlType(value="double")
    public static Double stXMin(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        return envelope.getXMin();
    }

    @SqlNullable
    @Description(value="Returns Y minima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_YMin")
    @SqlType(value="double")
    public static Double stYMin(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        return envelope.getYMin();
    }

    @SqlNullable
    @Description(value="Returns the cardinality of the collection of interior rings of a polygon")
    @ScalarFunction(value="ST_NumInteriorRing")
    @SqlType(value="bigint")
    public static Long stNumInteriorRings(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_NumInteriorRing", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.POLYGON));
        if (geometry.isEmpty()) {
            return null;
        }
        return ((Polygon)geometry).getNumInteriorRing();
    }

    @SqlNullable
    @Description(value="Returns an array of interior rings of a polygon")
    @ScalarFunction(value="ST_InteriorRings")
    @SqlType(value="array(Geometry)")
    public static Block stInteriorRings(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_InteriorRings", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.POLYGON));
        if (geometry.isEmpty()) {
            return null;
        }
        Polygon polygon = (Polygon)geometry;
        BlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, polygon.getNumInteriorRing());
        for (int i = 0; i < polygon.getNumInteriorRing(); ++i) {
            GeometryType.GEOMETRY.writeSlice(blockBuilder, JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)polygon.getInteriorRingN(i)));
        }
        return blockBuilder.build();
    }

    @Description(value="Returns the cardinality of the geometry collection")
    @ScalarFunction(value="ST_NumGeometries")
    @SqlType(value="integer")
    public static long stNumGeometries(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return 0L;
        }
        return geometry.getNumGeometries();
    }

    @Description(value="Returns a geometry that represents the point set union of the input geometries.")
    @ScalarFunction(value="ST_Union")
    @SqlType(value="Geometry")
    public static Slice stUnion(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        return GeoFunctions.stUnion((Iterable<Slice>)ImmutableList.of((Object)left, (Object)right));
    }

    @Description(value="Returns a geometry that represents the point set union of the input geometries.")
    @ScalarFunction(value="geometry_union")
    @SqlType(value="Geometry")
    public static Slice geometryUnion(@SqlType(value="array(Geometry)") Block input) {
        return GeoFunctions.stUnion(GeoFunctions.getGeometrySlicesFromBlock(input));
    }

    private static Slice stUnion(Iterable<Slice> slices) {
        ListeningGeometryCursor[] cursorsByDimension = new ListeningGeometryCursor[3];
        GeometryCursor[] operatorsByDimension = new GeometryCursor[3];
        Arrays.setAll(cursorsByDimension, i -> new ListeningGeometryCursor());
        Arrays.setAll(operatorsByDimension, i -> OperatorUnion.local().execute((GeometryCursor)cursorsByDimension[i], null, null));
        Iterator<Slice> slicesIterator = slices.iterator();
        if (!slicesIterator.hasNext()) {
            return null;
        }
        while (slicesIterator.hasNext()) {
            Slice slice = slicesIterator.next();
            if (slice.getInput().available() == 0) continue;
            for (OGCGeometry geometry : GeoFunctions.flattenCollection(EsriGeometrySerde.deserialize((Slice)slice))) {
                int dimension = geometry.dimension();
                cursorsByDimension[dimension].tick(geometry.getEsriGeometry());
                operatorsByDimension[dimension].tock();
            }
        }
        ArrayList<OGCGeometry> outputs = new ArrayList<OGCGeometry>();
        for (GeometryCursor operator : operatorsByDimension) {
            OGCGeometry unionedGeometry = OGCGeometry.createFromEsriGeometry((Geometry)operator.next(), null);
            if (unionedGeometry == null) continue;
            outputs.add(unionedGeometry);
        }
        if (outputs.size() == 1) {
            return EsriGeometrySerde.serialize((OGCGeometry)((OGCGeometry)outputs.get(0)));
        }
        return EsriGeometrySerde.serialize((OGCGeometry)new OGCConcreteGeometryCollection(outputs, null).flattenAndRemoveOverlaps().reduceFromMulti());
    }

    @SqlNullable
    @Description(value="Returns the geometry element at the specified index (indices started with 1)")
    @ScalarFunction(value="ST_GeometryN")
    @SqlType(value="Geometry")
    public static Slice stGeometryN(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long index) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        com.facebook.presto.geospatial.GeometryType type = com.facebook.presto.geospatial.GeometryType.getForJtsGeometryType((String)geometry.getGeometryType());
        if (!type.isMultitype()) {
            if (index == 1L) {
                return input;
            }
            return null;
        }
        GeometryCollection geometryCollection = (GeometryCollection)geometry;
        if (index < 1L || index > (long)geometryCollection.getNumGeometries()) {
            return null;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometryCollection.getGeometryN((int)index - 1));
    }

    @SqlNullable
    @Description(value="Returns the vertex of a linestring at the specified index (indices started with 1) ")
    @ScalarFunction(value="ST_PointN")
    @SqlType(value="Geometry")
    public static Slice stPointN(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long index) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_PointN", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.LINE_STRING));
        LineString linestring = (LineString)geometry;
        if (index < 1L || index > (long)linestring.getNumPoints()) {
            return null;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)linestring.getPointN(Math.toIntExact(index) - 1));
    }

    @SqlNullable
    @Description(value="Returns an array of geometries in the specified collection")
    @ScalarFunction(value="ST_Geometries")
    @SqlType(value="array(Geometry)")
    public static Block stGeometries(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        com.facebook.presto.geospatial.GeometryType type = com.facebook.presto.geospatial.GeometryType.getForJtsGeometryType((String)geometry.getGeometryType());
        if (!type.isMultitype()) {
            BlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, 1);
            GeometryType.GEOMETRY.writeSlice(blockBuilder, JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometry));
            return blockBuilder.build();
        }
        GeometryCollection collection = (GeometryCollection)geometry;
        BlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, collection.getNumGeometries());
        for (int i = 0; i < collection.getNumGeometries(); ++i) {
            GeometryType.GEOMETRY.writeSlice(blockBuilder, JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)collection.getGeometryN(i)));
        }
        return blockBuilder.build();
    }

    @SqlNullable
    @Description(value="Returns the interior ring element at the specified index (indices start at 1)")
    @ScalarFunction(value="ST_InteriorRingN")
    @SqlType(value="Geometry")
    public static Slice stInteriorRingN(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long index) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_InteriorRingN", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.POLYGON));
        Polygon polygon = (Polygon)geometry;
        if (index < 1L || index > (long)polygon.getNumInteriorRing()) {
            return null;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)polygon.getInteriorRingN(Math.toIntExact(index) - 1));
    }

    @Description(value="Returns the number of points in a Geometry")
    @ScalarFunction(value="ST_NumPoints")
    @SqlType(value="bigint")
    public static long stNumPoints(@SqlType(value="Geometry") Slice input) {
        return GeometryUtils.getPointCount((OGCGeometry)EsriGeometrySerde.deserialize((Slice)input));
    }

    @SqlNullable
    @Description(value="Returns TRUE if and only if the line is closed and simple")
    @ScalarFunction(value="ST_IsRing")
    @SqlType(value="boolean")
    public static Boolean stIsRing(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = EsriGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_IsRing", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.LINE_STRING));
        OGCLineString line = (OGCLineString)geometry;
        return line.isClosed() && line.isSimple();
    }

    @SqlNullable
    @Description(value="Returns the first point of a LINESTRING geometry as a Point")
    @ScalarFunction(value="ST_StartPoint")
    @SqlType(value="Geometry")
    public static Slice stStartPoint(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_StartPoint", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.LINE_STRING));
        if (geometry.isEmpty()) {
            return null;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)((LineString)geometry).getStartPoint());
    }

    @Description(value="Returns a \"simplified\" version of the given geometry")
    @ScalarFunction(value="simplify_geometry")
    @SqlType(value="Geometry")
    public static Slice simplifyGeometry(@SqlType(value="Geometry") Slice input, @SqlType(value="double") double distanceTolerance) {
        if (Double.isNaN(distanceTolerance)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distanceTolerance is NaN");
        }
        if (distanceTolerance < 0.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distanceTolerance is negative");
        }
        if (distanceTolerance == 0.0) {
            return input;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)TopologyPreservingSimplifier.simplify((org.locationtech.jts.geom.Geometry)JtsGeometrySerde.deserialize((Slice)input), (double)distanceTolerance));
    }

    @SqlNullable
    @Description(value="Returns the last point of a LINESTRING geometry as a Point")
    @ScalarFunction(value="ST_EndPoint")
    @SqlType(value="Geometry")
    public static Slice stEndPoint(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_EndPoint", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.LINE_STRING));
        if (geometry.isEmpty()) {
            return null;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)((LineString)geometry).getEndPoint());
    }

    @SqlNullable
    @Description(value="Returns an array of points in a geometry")
    @ScalarFunction(value="ST_Points")
    @SqlType(value="array(Geometry)")
    public static Block stPoints(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        BlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, geometry.getNumPoints());
        GeoFunctions.buildPointsBlock(geometry, blockBuilder);
        return blockBuilder.build();
    }

    private static void buildPointsBlock(org.locationtech.jts.geom.Geometry geometry, BlockBuilder blockBuilder) {
        com.facebook.presto.geospatial.GeometryType type = com.facebook.presto.geospatial.GeometryType.getForJtsGeometryType((String)geometry.getGeometryType());
        if (type == com.facebook.presto.geospatial.GeometryType.POINT) {
            GeometryType.GEOMETRY.writeSlice(blockBuilder, JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometry));
        } else if (type == com.facebook.presto.geospatial.GeometryType.GEOMETRY_COLLECTION) {
            GeometryCollection collection = (GeometryCollection)geometry;
            for (int i = 0; i < collection.getNumGeometries(); ++i) {
                org.locationtech.jts.geom.Geometry entry = collection.getGeometryN(i);
                GeoFunctions.buildPointsBlock(entry, blockBuilder);
            }
        } else {
            Coordinate[] vertices;
            GeometryFactory geometryFactory = geometry.getFactory();
            for (Coordinate coordinate : vertices = geometry.getCoordinates()) {
                GeometryType.GEOMETRY.writeSlice(blockBuilder, JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometryFactory.createPoint(coordinate)));
            }
        }
    }

    @SqlNullable
    @Description(value="Return the X coordinate of the point")
    @ScalarFunction(value="ST_X")
    @SqlType(value="double")
    public static Double stX(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_X", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.POINT));
        if (geometry.isEmpty()) {
            return null;
        }
        return ((Point)geometry).getX();
    }

    @SqlNullable
    @Description(value="Return the Y coordinate of the point")
    @ScalarFunction(value="ST_Y")
    @SqlType(value="double")
    public static Double stY(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Y", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.POINT));
        if (geometry.isEmpty()) {
            return null;
        }
        return ((Point)geometry).getY();
    }

    @Description(value="Returns the closure of the combinatorial boundary of this Geometry")
    @ScalarFunction(value="ST_Boundary")
    @SqlType(value="Geometry")
    public static Slice stBoundary(@SqlType(value="Geometry") Slice input) {
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)JtsGeometrySerde.deserialize((Slice)input).getBoundary());
    }

    @Description(value="Returns the bounding rectangular polygon of a Geometry")
    @ScalarFunction(value="ST_Envelope")
    @SqlType(value="Geometry")
    public static Slice stEnvelope(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return EMPTY_POLYGON;
        }
        return EsriGeometrySerde.serialize((Envelope)envelope);
    }

    @SqlNullable
    @Description(value="Returns the lower left and upper right corners of bounding rectangular polygon of a Geometry")
    @ScalarFunction(value="ST_EnvelopeAsPts")
    @SqlType(value="array(Geometry)")
    public static Block stEnvelopeAsPts(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        BlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, 2);
        Point lowerLeftCorner = GeometryUtils.createJtsPoint((double)envelope.getXMin(), (double)envelope.getYMin());
        Point upperRightCorner = GeometryUtils.createJtsPoint((double)envelope.getXMax(), (double)envelope.getYMax());
        GeometryType.GEOMETRY.writeSlice(blockBuilder, JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)lowerLeftCorner));
        GeometryType.GEOMETRY.writeSlice(blockBuilder, JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)upperRightCorner));
        return blockBuilder.build();
    }

    @Description(value="Returns the bounding rectangle of a Geometry expanded by distance.")
    @ScalarFunction(value="expand_envelope")
    @SqlType(value="Geometry")
    public static Slice expandEnvelope(@SqlType(value="Geometry") Slice input, @SqlType(value="double") double distance) {
        if (Double.isNaN(distance)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "expand_envelope: distance is NaN");
        }
        if (distance < 0.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("expand_envelope: distance %s is negative", distance));
        }
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return EMPTY_POLYGON;
        }
        return EsriGeometrySerde.serialize((Envelope)new Envelope(envelope.getXMin() - distance, envelope.getYMin() - distance, envelope.getXMax() + distance, envelope.getYMax() + distance));
    }

    @Description(value="Returns the Geometry value that represents the point set difference of two geometries")
    @ScalarFunction(value="ST_Difference")
    @SqlType(value="Geometry")
    public static Slice stDifference(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return EsriGeometrySerde.serialize((OGCGeometry)leftGeometry.difference(rightGeometry));
    }

    @SqlNullable
    @Description(value="Returns the 2-dimensional cartesian minimum distance (based on spatial ref) between two geometries in projected units")
    @ScalarFunction(value="ST_Distance")
    @SqlType(value="double")
    public static Double stDistance(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.isEmpty() || rightGeometry.isEmpty() ? null : Double.valueOf(leftGeometry.distance(rightGeometry));
    }

    @SqlNullable
    @Description(value="Returns a line string representing the exterior ring of the POLYGON")
    @ScalarFunction(value="ST_ExteriorRing")
    @SqlType(value="Geometry")
    public static Slice stExteriorRing(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_ExteriorRing", geometry, EnumSet.of(com.facebook.presto.geospatial.GeometryType.POLYGON));
        if (geometry.isEmpty()) {
            return null;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)((Polygon)geometry).getExteriorRing());
    }

    @Description(value="Returns the Geometry value that represents the point set intersection of two Geometries")
    @ScalarFunction(value="ST_Intersection")
    @SqlType(value="Geometry")
    public static Slice stIntersection(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        GeometrySerializationType leftType = EsriGeometrySerde.deserializeType((Slice)left);
        GeometrySerializationType rightType = EsriGeometrySerde.deserializeType((Slice)right);
        if (leftType == GeometrySerializationType.ENVELOPE && rightType == GeometrySerializationType.ENVELOPE) {
            Envelope rightEnvelope;
            Envelope leftEnvelope = EsriGeometrySerde.deserializeEnvelope((Slice)left);
            if (!leftEnvelope.intersect(rightEnvelope = EsriGeometrySerde.deserializeEnvelope((Slice)right))) {
                return EMPTY_POLYGON;
            }
            Envelope intersection = leftEnvelope;
            if (intersection.getXMin() == intersection.getXMax()) {
                if (intersection.getYMin() == intersection.getYMax()) {
                    return EsriGeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new com.esri.core.geometry.Point(intersection.getXMin(), intersection.getXMax()), null));
                }
                return EsriGeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Polyline(new com.esri.core.geometry.Point(intersection.getXMin(), intersection.getYMin()), new com.esri.core.geometry.Point(intersection.getXMin(), intersection.getYMax())), null));
            }
            if (intersection.getYMin() == intersection.getYMax()) {
                return EsriGeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Polyline(new com.esri.core.geometry.Point(intersection.getXMin(), intersection.getYMin()), new com.esri.core.geometry.Point(intersection.getXMax(), intersection.getYMin())), null));
            }
            return EsriGeometrySerde.serialize((Envelope)intersection);
        }
        if (leftType == GeometrySerializationType.ENVELOPE && EsriGeometrySerde.deserializeEnvelope((Slice)left).contains(EsriGeometrySerde.deserializeEnvelope((Slice)right))) {
            return right;
        }
        if (rightType == GeometrySerializationType.ENVELOPE && EsriGeometrySerde.deserializeEnvelope((Slice)right).contains(EsriGeometrySerde.deserializeEnvelope((Slice)left))) {
            return left;
        }
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return EsriGeometrySerde.serialize((OGCGeometry)leftGeometry.intersection(rightGeometry));
    }

    @Description(value="Returns the Geometry value that represents the point set symmetric difference of two Geometries")
    @ScalarFunction(value="ST_SymDifference")
    @SqlType(value="Geometry")
    public static Slice stSymmetricDifference(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return EsriGeometrySerde.serialize((OGCGeometry)leftGeometry.symDifference(rightGeometry));
    }

    @SqlNullable
    @Description(value="Returns TRUE if and only if no points of right lie in the exterior of left, and at least one point of the interior of left lies in the interior of right")
    @ScalarFunction(value="ST_Contains")
    @SqlType(value="boolean")
    public static Boolean stContains(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::contains)) {
            return false;
        }
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.contains(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the supplied geometries have some, but not all, interior points in common")
    @ScalarFunction(value="ST_Crosses")
    @SqlType(value="boolean")
    public static Boolean stCrosses(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.crosses(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the Geometries do not spatially intersect - if they do not share any space together")
    @ScalarFunction(value="ST_Disjoint")
    @SqlType(value="boolean")
    public static Boolean stDisjoint(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return true;
        }
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.disjoint(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the given geometries represent the same geometry")
    @ScalarFunction(value="ST_Equals")
    @SqlType(value="boolean")
    public static Boolean stEquals(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.Equals(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the Geometries spatially intersect in 2D - (share any portion of space) and FALSE if they don't (they are Disjoint)")
    @ScalarFunction(value="ST_Intersects")
    @SqlType(value="boolean")
    public static Boolean stIntersects(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.intersects(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the Geometries share space, are of the same dimension, but are not completely contained by each other")
    @ScalarFunction(value="ST_Overlaps")
    @SqlType(value="boolean")
    public static Boolean stOverlaps(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.overlaps(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if this Geometry is spatially related to another Geometry")
    @ScalarFunction(value="ST_Relate")
    @SqlType(value="boolean")
    public static Boolean stRelate(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right, @SqlType(value="varchar") Slice relation) {
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.relate(rightGeometry, relation.toStringUtf8());
    }

    @SqlNullable
    @Description(value="Returns TRUE if the geometries have at least one point in common, but their interiors do not intersect")
    @ScalarFunction(value="ST_Touches")
    @SqlType(value="boolean")
    public static Boolean stTouches(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.touches(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the geometry A is completely inside geometry B")
    @ScalarFunction(value="ST_Within")
    @SqlType(value="boolean")
    public static Boolean stWithin(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(right, left, Envelope::contains)) {
            return false;
        }
        OGCGeometry leftGeometry = EsriGeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = EsriGeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.within(rightGeometry);
    }

    @Description(value="Returns the type of the geometry")
    @ScalarFunction(value="ST_GeometryType")
    @SqlType(value="varchar")
    public static Slice stGeometryType(@SqlType(value="Geometry") Slice input) {
        return EsriGeometrySerde.getGeometryType((Slice)input).standardName();
    }

    @ScalarFunction
    @SqlNullable
    @Description(value="Returns an array of spatial partition IDs for a given geometry")
    @SqlType(value="array(int)")
    public static Block spatialPartitions(@SqlType(value="KdbTree") Object kdbTree, @SqlType(value="Geometry") Slice geometry) {
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)geometry);
        if (envelope.isEmpty()) {
            return null;
        }
        return GeoFunctions.spatialPartitions((KdbTree)kdbTree, new Rectangle(envelope.getXMin(), envelope.getYMin(), envelope.getXMax(), envelope.getYMax()));
    }

    @ScalarFunction
    @SqlNullable
    @Description(value="Returns an array of spatial partition IDs for a geometry representing a set of points within specified distance from the input geometry")
    @SqlType(value="array(int)")
    public static Block spatialPartitions(@SqlType(value="KdbTree") Object kdbTree, @SqlType(value="Geometry") Slice geometry, @SqlType(value="double") double distance) {
        if (Double.isNaN(distance)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is NaN");
        }
        if (Double.isInfinite(distance)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is infinite");
        }
        if (distance < 0.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is negative");
        }
        Envelope envelope = EsriGeometrySerde.deserializeEnvelope((Slice)geometry);
        if (envelope.isEmpty()) {
            return null;
        }
        Rectangle expandedEnvelope2D = new Rectangle(envelope.getXMin() - distance, envelope.getYMin() - distance, envelope.getXMax() + distance, envelope.getYMax() + distance);
        return GeoFunctions.spatialPartitions((KdbTree)kdbTree, expandedEnvelope2D);
    }

    static Block spatialPartitions(KdbTree kdbTree, Rectangle envelope) {
        boolean point;
        Map partitions = kdbTree.findIntersectingLeaves(envelope);
        if (partitions.isEmpty()) {
            return EMPTY_ARRAY_OF_INTS;
        }
        boolean bl = point = envelope.getWidth() == 0.0 && envelope.getHeight() == 0.0;
        if (point) {
            for (Map.Entry partition : partitions.entrySet()) {
                if (!(envelope.getXMin() < ((Rectangle)partition.getValue()).getXMax()) || !(envelope.getYMin() < ((Rectangle)partition.getValue()).getYMax())) continue;
                BlockBuilder blockBuilder = IntegerType.INTEGER.createFixedSizeBlockBuilder(1);
                blockBuilder.writeInt(((Integer)partition.getKey()).intValue());
                return blockBuilder.build();
            }
            throw new VerifyException(String.format("Cannot find half-open partition extent for a point: (%s, %s)", envelope.getXMin(), envelope.getYMin()));
        }
        BlockBuilder blockBuilder = IntegerType.INTEGER.createFixedSizeBlockBuilder(partitions.size());
        Iterator iterator = partitions.keySet().iterator();
        while (iterator.hasNext()) {
            int id = (Integer)iterator.next();
            blockBuilder.writeInt(id);
        }
        return blockBuilder.build();
    }

    private static OGCGeometry geomFromBinary(Slice input) {
        OGCGeometry geometry;
        Objects.requireNonNull(input, "input is null");
        try {
            geometry = OGCGeometry.fromBinary((ByteBuffer)input.toByteBuffer().slice());
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid WKB", (Throwable)e);
        }
        geometry.setSpatialReference(null);
        return geometry;
    }

    private static void validateType(String function, OGCGeometry geometry, Set<com.facebook.presto.geospatial.GeometryType> validTypes) {
        com.facebook.presto.geospatial.GeometryType type = com.facebook.presto.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (!validTypes.contains(type)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("%s only applies to %s. Input type is: %s", function, OR_JOINER.join(validTypes), type));
        }
    }

    private static void validateType(String function, org.locationtech.jts.geom.Geometry geometry, Set<com.facebook.presto.geospatial.GeometryType> validTypes) {
        com.facebook.presto.geospatial.GeometryType type = com.facebook.presto.geospatial.GeometryType.getForJtsGeometryType((String)geometry.getGeometryType());
        if (!validTypes.contains(type)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("%s only applies to %s. Input type is: %s", function, OR_JOINER.join(validTypes), type));
        }
    }

    private static void verifySameSpatialReference(OGCGeometry leftGeometry, OGCGeometry rightGeometry) {
        Preconditions.checkArgument((boolean)Objects.equals(leftGeometry.getEsriSpatialReference(), rightGeometry.getEsriSpatialReference()), (Object)"Input geometries must have the same spatial reference");
    }

    private static boolean envelopes(Slice left, Slice right, EnvelopesPredicate predicate) {
        Envelope leftEnvelope = EsriGeometrySerde.deserializeEnvelope((Slice)left);
        Envelope rightEnvelope = EsriGeometrySerde.deserializeEnvelope((Slice)right);
        if (leftEnvelope.isEmpty() || rightEnvelope.isEmpty()) {
            return false;
        }
        return predicate.apply(leftEnvelope, rightEnvelope);
    }

    private static Iterable<Slice> getGeometrySlicesFromBlock(final Block block) {
        Objects.requireNonNull(block, "block is null");
        return () -> new Iterator<Slice>(){
            private int iteratorPosition;

            @Override
            public boolean hasNext() {
                return this.iteratorPosition != block.getPositionCount();
            }

            @Override
            public Slice next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException("Slices have been consumed");
                }
                return GeometryType.GEOMETRY.getSlice(block, this.iteratorPosition++);
            }
        };
    }

    private static Iterable<OGCGeometry> flattenCollection(OGCGeometry geometry) {
        if (geometry == null) {
            return ImmutableList.of();
        }
        if (!(geometry instanceof OGCConcreteGeometryCollection)) {
            return ImmutableList.of((Object)geometry);
        }
        if (((OGCConcreteGeometryCollection)geometry).numGeometries() == 0) {
            return ImmutableList.of();
        }
        return () -> new GeometryCollectionIterator(geometry);
    }

    private static class GeometryCollectionIterator
    implements Iterator<OGCGeometry> {
        private final Deque<OGCGeometry> geometriesDeque = new ArrayDeque<OGCGeometry>();

        GeometryCollectionIterator(OGCGeometry geometries) {
            this.geometriesDeque.push(Objects.requireNonNull(geometries, "geometries is null"));
        }

        @Override
        public boolean hasNext() {
            if (this.geometriesDeque.isEmpty()) {
                return false;
            }
            while (this.geometriesDeque.peek() instanceof OGCConcreteGeometryCollection) {
                OGCGeometryCollection collection = (OGCGeometryCollection)this.geometriesDeque.pop();
                for (int i = 0; i < collection.numGeometries(); ++i) {
                    this.geometriesDeque.push(collection.geometryN(i));
                }
            }
            return !this.geometriesDeque.isEmpty();
        }

        @Override
        public OGCGeometry next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException("Geometries have been consumed");
            }
            return this.geometriesDeque.pop();
        }
    }

    private static interface EnvelopesPredicate {
        public boolean apply(Envelope var1, Envelope var2);
    }
}

