/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.spi.page;

import com.facebook.presto.common.Page;
import com.facebook.presto.common.block.BlockEncodingSerde;
import com.facebook.presto.spi.page.PageCodecMarker;
import com.facebook.presto.spi.page.PageCompressor;
import com.facebook.presto.spi.page.PageDecompressor;
import com.facebook.presto.spi.page.PagesSerdeUtil;
import com.facebook.presto.spi.page.SerializedPage;
import com.facebook.presto.spi.spiller.SpillCipher;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.concurrent.NotThreadSafe;
import sun.misc.Unsafe;

@NotThreadSafe
public class PagesSerde {
    private static final double MINIMUM_COMPRESSION_RATIO = 0.9;
    private final BlockEncodingSerde blockEncodingSerde;
    private final Optional<PageCompressor> compressor;
    private final Optional<PageDecompressor> decompressor;
    private final Optional<SpillCipher> spillCipher;
    private final boolean checksumEnabled;
    private byte[] compressionBuffer;

    public PagesSerde(BlockEncodingSerde blockEncodingSerde, Optional<PageCompressor> compressor, Optional<PageDecompressor> decompressor, Optional<SpillCipher> spillCipher) {
        this(blockEncodingSerde, compressor, decompressor, spillCipher, false);
    }

    public PagesSerde(BlockEncodingSerde blockEncodingSerde, Optional<PageCompressor> compressor, Optional<PageDecompressor> decompressor, Optional<SpillCipher> spillCipher, boolean checksumEnabled) {
        this.blockEncodingSerde = Objects.requireNonNull(blockEncodingSerde, "blockEncodingSerde is null");
        PagesSerde.checkArgument(compressor.isPresent() == decompressor.isPresent(), "compressor and decompressor must both be present or both be absent");
        this.compressor = Objects.requireNonNull(compressor, "compressor is null");
        this.decompressor = Objects.requireNonNull(decompressor, "decompressor is null");
        this.spillCipher = Objects.requireNonNull(spillCipher, "spillCipher is null");
        PagesSerde.checkState(!spillCipher.isPresent() || !spillCipher.get().isDestroyed(), "spillCipher is already destroyed");
        this.checksumEnabled = checksumEnabled;
    }

    public SerializedPage serialize(Page page) {
        DynamicSliceOutput serializationBuffer = new DynamicSliceOutput(Math.toIntExact(page.getSizeInBytes() + 4L));
        PagesSerdeUtil.writeRawPage(page, (SliceOutput)serializationBuffer, this.blockEncodingSerde);
        return this.wrapSlice(serializationBuffer.slice(), page.getPositionCount());
    }

    public SerializedPage serialize(Slice slice, int positionCount) {
        PagesSerde.checkArgument(slice.isCompact(), "slice is not compact");
        return this.wrapSlice(slice, positionCount);
    }

    public Page deserialize(SerializedPage serializedPage) {
        PagesSerde.checkArgument(serializedPage != null, "serializedPage is null");
        Slice slice = serializedPage.getSlice();
        if (PageCodecMarker.ENCRYPTED.isSet(serializedPage.getPageCodecMarkers())) {
            PagesSerde.checkState(this.spillCipher.isPresent(), "Page is encrypted, but spill cipher is missing");
            slice = Slices.wrappedBuffer((ByteBuffer)this.spillCipher.get().decrypt(slice.toByteBuffer()));
        }
        if (PageCodecMarker.COMPRESSED.isSet(serializedPage.getPageCodecMarkers())) {
            PagesSerde.checkState(this.decompressor.isPresent(), "Page is compressed, but decompressor is missing");
            int uncompressedSize = serializedPage.getUncompressedSizeInBytes();
            ByteBuffer decompressionBuffer = ByteBuffer.allocate(uncompressedSize);
            this.decompressor.get().decompress(slice.toByteBuffer(), decompressionBuffer);
            ((Buffer)decompressionBuffer).flip();
            PagesSerde.checkState(decompressionBuffer.remaining() == uncompressedSize, "page size changed after decompression into decompressionBuffer");
            slice = Slices.wrappedBuffer((ByteBuffer)decompressionBuffer);
        }
        return PagesSerdeUtil.readRawPage(serializedPage.getPositionCount(), (SliceInput)slice.getInput(), this.blockEncodingSerde);
    }

    public long getSizeInBytes() {
        return this.compressionBuffer == null ? 0L : (long)this.compressionBuffer.length;
    }

    public long getRetainedSizeInBytes() {
        return SizeOf.sizeOf((byte[])this.compressionBuffer);
    }

    private SerializedPage wrapSlice(Slice slice, int positionCount) {
        int uncompressedSize = slice.length();
        byte markers = PageCodecMarker.none();
        if (this.compressor.isPresent()) {
            int maxCompressedSize = this.compressor.get().maxCompressedLength(uncompressedSize);
            this.compressionBuffer = PagesSerde.ensureCapacity(this.compressionBuffer, maxCompressedSize);
            int compressedSize = this.compressor.get().compress((byte[])slice.getBase(), (int)(slice.getAddress() - (long)Unsafe.ARRAY_BYTE_BASE_OFFSET), uncompressedSize, this.compressionBuffer, 0, maxCompressedSize);
            if ((double)compressedSize / (double)uncompressedSize <= 0.9) {
                slice = Slices.copyOf((Slice)Slices.wrappedBuffer((byte[])this.compressionBuffer, (int)0, (int)compressedSize));
                markers = PageCodecMarker.COMPRESSED.set(markers);
            }
        }
        if (this.spillCipher.isPresent()) {
            slice = Slices.wrappedBuffer((ByteBuffer)this.spillCipher.get().encrypt(slice.toByteBuffer()));
            markers = PageCodecMarker.ENCRYPTED.set(markers);
        } else if (!slice.isCompact()) {
            slice = Slices.copyOf((Slice)slice);
        }
        long checksum = 0L;
        if (this.checksumEnabled) {
            markers = PageCodecMarker.CHECKSUMMED.set(markers);
            checksum = PagesSerdeUtil.computeSerializedPageChecksum(slice, markers, positionCount, uncompressedSize);
        }
        return new SerializedPage(slice, markers, positionCount, uncompressedSize, checksum);
    }

    private static void checkArgument(boolean condition, String message) {
        if (!condition) {
            throw new IllegalArgumentException(message);
        }
    }

    private static void checkState(boolean test, String message) {
        if (!test) {
            throw new IllegalStateException(message);
        }
    }

    private static byte[] ensureCapacity(byte[] buffer, int capacity) {
        if (buffer == null || buffer.length < capacity) {
            return new byte[capacity];
        }
        return buffer;
    }
}

