/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.verifier.framework;

import com.facebook.airlift.event.client.EventClient;
import com.facebook.airlift.log.Logger;
import com.facebook.presto.sql.parser.ParsingException;
import com.facebook.presto.sql.parser.SqlParser;
import com.facebook.presto.sql.tree.AstVisitor;
import com.facebook.presto.sql.tree.CreateTable;
import com.facebook.presto.sql.tree.Except;
import com.facebook.presto.sql.tree.Intersect;
import com.facebook.presto.sql.tree.Join;
import com.facebook.presto.sql.tree.Literal;
import com.facebook.presto.sql.tree.Node;
import com.facebook.presto.sql.tree.Offset;
import com.facebook.presto.sql.tree.OrderBy;
import com.facebook.presto.sql.tree.Query;
import com.facebook.presto.sql.tree.QuerySpecification;
import com.facebook.presto.sql.tree.Statement;
import com.facebook.presto.sql.tree.SubqueryExpression;
import com.facebook.presto.sql.tree.Table;
import com.facebook.presto.sql.tree.TableSubquery;
import com.facebook.presto.sql.tree.Union;
import com.facebook.presto.sql.tree.Unnest;
import com.facebook.presto.sql.tree.Values;
import com.facebook.presto.sql.tree.WithQuery;
import com.facebook.presto.verifier.annotation.ForControl;
import com.facebook.presto.verifier.annotation.ForTest;
import com.facebook.presto.verifier.event.VerifierQueryEvent;
import com.facebook.presto.verifier.framework.ClusterType;
import com.facebook.presto.verifier.framework.QueryConfigurationOverrides;
import com.facebook.presto.verifier.framework.QueryType;
import com.facebook.presto.verifier.framework.SkippedReason;
import com.facebook.presto.verifier.framework.SourceQuery;
import com.facebook.presto.verifier.framework.Verification;
import com.facebook.presto.verifier.framework.VerificationContext;
import com.facebook.presto.verifier.framework.VerificationFactory;
import com.facebook.presto.verifier.framework.VerificationResult;
import com.facebook.presto.verifier.framework.VerifierConfig;
import com.facebook.presto.verifier.framework.VerifierUtil;
import com.facebook.presto.verifier.source.SourceQuerySupplier;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;

public class VerificationManager {
    private static final Logger log = Logger.get(VerificationManager.class);
    private final SourceQuerySupplier sourceQuerySupplier;
    private final VerificationFactory verificationFactory;
    private final SqlParser sqlParser;
    private final Set<EventClient> eventClients;
    private final List<Predicate<SourceQuery>> customQueryFilters;
    private final QueryConfigurationOverrides controlOverrides;
    private final QueryConfigurationOverrides testOverrides;
    private final String testId;
    private final Optional<Set<String>> whitelist;
    private final Optional<Set<String>> blacklist;
    private final int maxConcurrency;
    private final int suiteRepetitions;
    private final int queryRepetitions;
    private final int verificationResubmissionLimit;
    private final boolean explain;
    private final boolean skipControl;
    private final ExecutorService executor;
    private final CompletionService<VerificationResult> completionService;
    private final AtomicInteger queriesSubmitted = new AtomicInteger();

    @Inject
    public VerificationManager(SourceQuerySupplier sourceQuerySupplier, VerificationFactory verificationFactory, SqlParser sqlParser, Set<EventClient> eventClients, List<Predicate<SourceQuery>> customQueryFilters, @ForControl QueryConfigurationOverrides controlOverrides, @ForTest QueryConfigurationOverrides testOverrides, VerifierConfig config) {
        this.sourceQuerySupplier = Objects.requireNonNull(sourceQuerySupplier, "sourceQuerySupplier is null");
        this.verificationFactory = Objects.requireNonNull(verificationFactory, "verificationFactory is null");
        this.sqlParser = Objects.requireNonNull(sqlParser, "sqlParser is null");
        this.eventClients = ImmutableSet.copyOf(eventClients);
        this.customQueryFilters = Objects.requireNonNull(customQueryFilters, "customQueryFilters is null");
        this.controlOverrides = Objects.requireNonNull(controlOverrides, "controlOverrides is null");
        this.testOverrides = Objects.requireNonNull(testOverrides, "testOverride is null");
        this.testId = Objects.requireNonNull(config.getTestId(), "testId is null");
        this.whitelist = Objects.requireNonNull(config.getWhitelist(), "whitelist is null");
        this.blacklist = Objects.requireNonNull(config.getBlacklist(), "blacklist is null");
        this.maxConcurrency = config.getMaxConcurrency();
        this.suiteRepetitions = config.getSuiteRepetitions();
        this.queryRepetitions = config.getQueryRepetitions();
        this.verificationResubmissionLimit = config.getVerificationResubmissionLimit();
        this.skipControl = config.isSkipControl();
        this.explain = config.isExplain();
        this.executor = Executors.newFixedThreadPool(this.maxConcurrency);
        this.completionService = new ExecutorCompletionService<VerificationResult>(this.executor);
    }

    @PostConstruct
    public void start() {
        List<SourceQuery> sourceQueries = (List<SourceQuery>)this.sourceQuerySupplier.get();
        log.info("Total Queries: %s", new Object[]{sourceQueries.size()});
        sourceQueries = this.applyOverrides(sourceQueries);
        sourceQueries = this.applyWhitelist(sourceQueries);
        sourceQueries = this.applyBlacklist(sourceQueries);
        sourceQueries = this.filterQueryType(sourceQueries);
        sourceQueries = this.applyCustomFilters(sourceQueries);
        this.submit(sourceQueries);
        this.reportProgressUntilFinished();
    }

    @PreDestroy
    public void close() {
        for (EventClient eventClient : this.eventClients) {
            if (!(eventClient instanceof Closeable)) continue;
            try {
                ((Closeable)eventClient).close();
            }
            catch (IOException e) {
                log.error((Throwable)e);
            }
        }
        this.executor.shutdownNow();
    }

    private void resubmit(Verification verification) {
        SourceQuery sourceQuery = verification.getSourceQuery();
        VerificationContext newContext = verification.getVerificationContext();
        Verification newVerification = this.verificationFactory.get(sourceQuery, Optional.of(newContext));
        this.completionService.submit(newVerification::run);
        this.queriesSubmitted.addAndGet(1);
        log.info("Verification %s failed, resubmitted for verification (%s/%s)", new Object[]{sourceQuery.getName(), newContext.getResubmissionCount(), this.verificationResubmissionLimit});
    }

    @VisibleForTesting
    AtomicInteger getQueriesSubmitted() {
        return this.queriesSubmitted;
    }

    private List<SourceQuery> applyOverrides(List<SourceQuery> sourceQueries) {
        return (List)sourceQueries.stream().map(sourceQuery -> new SourceQuery(sourceQuery.getSuite(), sourceQuery.getName(), sourceQuery.getQuery(ClusterType.CONTROL), sourceQuery.getQuery(ClusterType.TEST), sourceQuery.getControlConfiguration().applyOverrides(this.controlOverrides), sourceQuery.getTestConfiguration().applyOverrides(this.testOverrides))).collect(ImmutableList.toImmutableList());
    }

    private List<SourceQuery> applyWhitelist(List<SourceQuery> sourceQueries) {
        if (!this.whitelist.isPresent()) {
            return sourceQueries;
        }
        List selected = (List)sourceQueries.stream().filter(sourceQuery -> this.whitelist.get().contains(sourceQuery.getName())).collect(ImmutableList.toImmutableList());
        log.info("Applying whitelist... Remaining queries: %s", new Object[]{selected.size()});
        return selected;
    }

    private List<SourceQuery> applyBlacklist(List<SourceQuery> sourceQueries) {
        if (!this.blacklist.isPresent()) {
            return sourceQueries;
        }
        List selected = (List)sourceQueries.stream().filter(sourceQuery -> !this.blacklist.get().contains(sourceQuery.getName())).collect(ImmutableList.toImmutableList());
        log.info("Applying blacklist... Remaining queries: %s", new Object[]{selected.size()});
        return selected;
    }

    private List<SourceQuery> filterQueryType(List<SourceQuery> sourceQueries) {
        if (this.explain) {
            return sourceQueries;
        }
        ImmutableList.Builder selected = ImmutableList.builder();
        for (SourceQuery sourceQuery : sourceQueries) {
            try {
                Statement controlStatement = this.sqlParser.createStatement(sourceQuery.getQuery(ClusterType.CONTROL), VerifierUtil.PARSING_OPTIONS);
                QueryType controlQueryType = QueryType.of(controlStatement);
                QueryType testQueryType = QueryType.of(this.sqlParser.createStatement(sourceQuery.getQuery(ClusterType.TEST), VerifierUtil.PARSING_OPTIONS));
                if (controlQueryType == QueryType.UNSUPPORTED || testQueryType == QueryType.UNSUPPORTED) {
                    this.postEvent(VerifierQueryEvent.skipped(sourceQuery.getSuite(), this.testId, sourceQuery, SkippedReason.UNSUPPORTED_QUERY_TYPE, this.skipControl));
                    continue;
                }
                if (controlQueryType != testQueryType) {
                    this.postEvent(VerifierQueryEvent.skipped(sourceQuery.getSuite(), this.testId, sourceQuery, SkippedReason.MISMATCHED_QUERY_TYPE, this.skipControl));
                    continue;
                }
                if (this.isLimitWithoutOrderBy(controlStatement, sourceQuery.getName()).booleanValue()) {
                    log.debug("LimitWithoutOrderByChecker Skipped %s", new Object[]{sourceQuery.getName()});
                    this.postEvent(VerifierQueryEvent.skipped(sourceQuery.getSuite(), this.testId, sourceQuery, SkippedReason.NON_DETERMINISTIC, this.skipControl));
                    continue;
                }
                selected.add((Object)sourceQuery);
            }
            catch (ParsingException e) {
                log.warn("Failed to parse query: %s", new Object[]{sourceQuery.getName()});
                this.postEvent(VerifierQueryEvent.skipped(sourceQuery.getSuite(), this.testId, sourceQuery, SkippedReason.SYNTAX_ERROR, this.skipControl));
            }
        }
        ImmutableList selectQueries = selected.build();
        log.info("Filtering query type... Remaining queries: %s", new Object[]{selectQueries.size()});
        return selectQueries;
    }

    private Boolean isLimitWithoutOrderBy(Statement controlStatement, String queryId) {
        try {
            Boolean process = (Boolean)new LimitWithoutOrderByChecker().process((Node)controlStatement, 0);
            if (process == null) {
                log.warn("LimitWithoutOrderByChecker Check failed for %s ", new Object[]{queryId});
                return false;
            }
            return process;
        }
        catch (Exception e) {
            log.warn("LimitWithoutOrderByChecker Check failed for %s ", new Object[]{queryId});
            log.error((Throwable)e);
            return false;
        }
    }

    private List<SourceQuery> applyCustomFilters(List<SourceQuery> sourceQueries) {
        if (this.customQueryFilters.isEmpty()) {
            return sourceQueries;
        }
        log.info("Applying custom query filters");
        sourceQueries = new ArrayList<SourceQuery>(sourceQueries);
        for (Predicate<SourceQuery> filter : this.customQueryFilters) {
            Iterator<SourceQuery> iterator = sourceQueries.iterator();
            while (iterator.hasNext()) {
                SourceQuery sourceQuery = iterator.next();
                if (filter.test(sourceQuery)) continue;
                iterator.remove();
                this.postEvent(VerifierQueryEvent.skipped(sourceQuery.getSuite(), this.testId, sourceQuery, SkippedReason.CUSTOM_FILTER, this.skipControl));
            }
            log.info("Applying custom filter %s... Remaining queries: %s", new Object[]{filter.getClass().getSimpleName(), sourceQueries.size()});
        }
        return sourceQueries;
    }

    private void submit(List<SourceQuery> sourceQueries) {
        for (int i = 0; i < this.suiteRepetitions; ++i) {
            for (SourceQuery sourceQuery : sourceQueries) {
                for (int j = 0; j < this.queryRepetitions; ++j) {
                    Verification verification = this.verificationFactory.get(sourceQuery, Optional.empty());
                    this.completionService.submit(verification::run);
                }
            }
        }
        int queriesSubmitted = sourceQueries.size() * this.suiteRepetitions * this.queryRepetitions;
        log.info("Queries submitted: %s", new Object[]{queriesSubmitted});
        this.queriesSubmitted.addAndGet(queriesSubmitted);
    }

    private void reportProgressUntilFinished() {
        int completed = 0;
        double lastProgress = 0.0;
        EnumMap<VerifierQueryEvent.EventStatus, Integer> statusCount = new EnumMap<VerifierQueryEvent.EventStatus, Integer>(VerifierQueryEvent.EventStatus.class);
        while (completed < this.queriesSubmitted.get()) {
            try {
                double progress;
                VerificationResult result = this.completionService.take().get();
                Optional<VerifierQueryEvent> event = result.getEvent();
                ++completed;
                if (!event.isPresent()) {
                    statusCount.compute(VerifierQueryEvent.EventStatus.SKIPPED, (status, count) -> count == null ? 1 : count + 1);
                } else {
                    statusCount.compute(VerifierQueryEvent.EventStatus.valueOf(event.get().getStatus()), (status, count) -> count == null ? 1 : count + 1);
                    this.postEvent(event.get());
                }
                if (result.shouldResubmit()) {
                    this.resubmit(result.getVerification());
                }
                if (!((progress = (double)completed / (double)this.queriesSubmitted.get() * 100.0) - lastProgress > 0.5) && completed != this.queriesSubmitted.get()) continue;
                log.info("Progress: %s succeeded, %s skipped, %s resolved, %s failed, %.2f%% done", new Object[]{statusCount.getOrDefault((Object)VerifierQueryEvent.EventStatus.SUCCEEDED, 0), statusCount.getOrDefault((Object)VerifierQueryEvent.EventStatus.SKIPPED, 0), statusCount.getOrDefault((Object)VerifierQueryEvent.EventStatus.FAILED_RESOLVED, 0), statusCount.getOrDefault((Object)VerifierQueryEvent.EventStatus.FAILED, 0), progress});
                lastProgress = progress;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
            catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private void postEvent(VerifierQueryEvent event) {
        for (EventClient eventClient : this.eventClients) {
            eventClient.post((Object[])new VerifierQueryEvent[]{event});
        }
    }

    static class LimitWithoutOrderByChecker
    extends AstVisitor<Boolean, Integer> {
        LimitWithoutOrderByChecker() {
        }

        protected Boolean visitQuerySpecification(QuerySpecification node, Integer level) {
            if (node.getChildren().stream().anyMatch(n -> (Boolean)this.process((Node)n, level))) {
                return true;
            }
            return level != 0 && node.getLimit().isPresent() && !node.getOrderBy().isPresent();
        }

        protected Boolean visitQuery(Query node, Integer level) {
            if (node.getChildren().stream().anyMatch(n -> (Boolean)this.process((Node)n, level))) {
                return true;
            }
            return level != 0 && node.getLimit().isPresent() && !node.getOrderBy().isPresent();
        }

        protected Boolean visitWithQuery(WithQuery node, Integer level) {
            return (Boolean)this.process((Node)node.getQuery(), level + 1);
        }

        protected Boolean visitTableSubquery(TableSubquery node, Integer level) {
            return (Boolean)this.process((Node)node.getQuery(), level + 1);
        }

        protected Boolean visitSubqueryExpression(SubqueryExpression node, Integer level) {
            return (Boolean)this.process((Node)node.getQuery(), level + 1);
        }

        protected Boolean visitNode(Node node, Integer level) {
            return node.getChildren().stream().anyMatch(n -> (Boolean)this.process((Node)n, level));
        }

        protected Boolean visitUnion(Union node, Integer level) {
            return node.getChildren().stream().anyMatch(n -> (Boolean)this.process((Node)n, level));
        }

        protected Boolean visitIntersect(Intersect node, Integer level) {
            return node.getChildren().stream().anyMatch(n -> (Boolean)this.process((Node)n, level));
        }

        protected Boolean visitExcept(Except node, Integer level) {
            return node.getChildren().stream().anyMatch(n -> (Boolean)this.process((Node)n, level));
        }

        protected Boolean visitJoin(Join node, Integer level) {
            return node.getChildren().stream().anyMatch(n -> (Boolean)this.process((Node)n, level));
        }

        protected Boolean visitUnnest(Unnest node, Integer level) {
            return false;
        }

        protected Boolean visitTable(Table node, Integer level) {
            return false;
        }

        protected Boolean visitCreateTable(CreateTable node, Integer level) {
            return false;
        }

        protected Boolean visitValues(Values node, Integer level) {
            return false;
        }

        protected Boolean visitLiteral(Literal node, Integer context) {
            return false;
        }

        protected Boolean visitOrderBy(OrderBy node, Integer context) {
            return false;
        }

        protected Boolean visitOffset(Offset node, Integer context) {
            return false;
        }
    }
}

