/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.react.devsupport;

import androidx.annotation.Nullable;
import com.facebook.common.logging.FLog;
import com.facebook.infer.annotation.Assertions;
import com.facebook.react.common.DebugServerException;
import com.facebook.react.devsupport.MultipartStreamReader;
import com.facebook.react.devsupport.interfaces.DevBundleDownloadListener;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Headers;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okio.Buffer;
import okio.BufferedSource;
import okio.Okio;
import okio.Sink;
import okio.Source;
import org.json.JSONException;
import org.json.JSONObject;

public class BundleDownloader {
    private static final String TAG = "BundleDownloader";
    private static final int FILES_CHANGED_COUNT_NOT_BUILT_BY_BUNDLER = -2;
    private final OkHttpClient mClient;
    @Nullable
    private Call mDownloadBundleFromURLCall;

    public BundleDownloader(OkHttpClient client) {
        this.mClient = client;
    }

    public void downloadBundleFromURL(DevBundleDownloadListener callback, File outputFile, String bundleURL, @Nullable BundleInfo bundleInfo) {
        this.downloadBundleFromURL(callback, outputFile, bundleURL, bundleInfo, new Request.Builder());
    }

    public void downloadBundleFromURL(final DevBundleDownloadListener callback, final File outputFile, String bundleURL, final @Nullable BundleInfo bundleInfo, Request.Builder requestBuilder) {
        Request request = requestBuilder.url(bundleURL).addHeader("Accept", "multipart/mixed").build();
        this.mDownloadBundleFromURLCall = (Call)Assertions.assertNotNull((Object)this.mClient.newCall(request));
        this.mDownloadBundleFromURLCall.enqueue(new Callback(){

            public void onFailure(Call call, IOException e) {
                if (BundleDownloader.this.mDownloadBundleFromURLCall == null || BundleDownloader.this.mDownloadBundleFromURLCall.isCanceled()) {
                    BundleDownloader.this.mDownloadBundleFromURLCall = null;
                    return;
                }
                BundleDownloader.this.mDownloadBundleFromURLCall = null;
                String url = call.request().url().toString();
                callback.onFailure(DebugServerException.makeGeneric(url, "Could not connect to development server.", "URL: " + url, e));
            }

            public void onResponse(Call call, Response response) throws IOException {
                if (BundleDownloader.this.mDownloadBundleFromURLCall == null || BundleDownloader.this.mDownloadBundleFromURLCall.isCanceled()) {
                    BundleDownloader.this.mDownloadBundleFromURLCall = null;
                    return;
                }
                BundleDownloader.this.mDownloadBundleFromURLCall = null;
                String url = response.request().url().toString();
                String contentType = response.header("content-type");
                Pattern regex = Pattern.compile("multipart/mixed;.*boundary=\"([^\"]+)\"");
                Matcher match = regex.matcher(contentType);
                try (Response r = response;){
                    if (match.find()) {
                        BundleDownloader.this.processMultipartResponse(url, r, match.group(1), outputFile, bundleInfo, callback);
                    } else {
                        BundleDownloader.this.processBundleResult(url, r.code(), r.headers(), Okio.buffer((Source)r.body().source()), outputFile, bundleInfo, callback);
                    }
                }
            }
        });
    }

    private void processMultipartResponse(final String url, final Response response, String boundary, final File outputFile, final @Nullable BundleInfo bundleInfo, final DevBundleDownloadListener callback) throws IOException {
        MultipartStreamReader bodyReader = new MultipartStreamReader(response.body().source(), boundary);
        boolean completed = bodyReader.readAllParts(new MultipartStreamReader.ChunkListener(){

            @Override
            public void onChunkComplete(Map<String, String> headers, Buffer body, boolean isLastChunk) throws IOException {
                if (isLastChunk) {
                    int status = response.code();
                    if (headers.containsKey("X-Http-Status")) {
                        status = Integer.parseInt(headers.get("X-Http-Status"));
                    }
                    BundleDownloader.this.processBundleResult(url, status, Headers.of(headers), (BufferedSource)body, outputFile, bundleInfo, callback);
                } else {
                    if (!headers.containsKey("Content-Type") || !headers.get("Content-Type").equals("application/json")) {
                        return;
                    }
                    try {
                        JSONObject progress = new JSONObject(body.readUtf8());
                        String status = progress.has("status") ? progress.getString("status") : "Bundling";
                        Integer done = null;
                        if (progress.has("done")) {
                            done = progress.getInt("done");
                        }
                        Integer total = null;
                        if (progress.has("total")) {
                            total = progress.getInt("total");
                        }
                        callback.onProgress(status, done, total);
                    }
                    catch (JSONException e) {
                        FLog.e((String)"ReactNative", (String)("Error parsing progress JSON. " + e.toString()));
                    }
                }
            }

            @Override
            public void onChunkProgress(Map<String, String> headers, long loaded, long total) {
                if ("application/javascript".equals(headers.get("Content-Type"))) {
                    callback.onProgress("Downloading", (int)(loaded / 1024L), (int)(total / 1024L));
                }
            }
        });
        if (!completed) {
            callback.onFailure(new DebugServerException("Error while reading multipart response.\n\nResponse code: " + response.code() + "\n\nURL: " + url.toString() + "\n\n"));
        }
    }

    private void processBundleResult(String url, int statusCode, Headers headers, BufferedSource body, File outputFile, BundleInfo bundleInfo, DevBundleDownloadListener callback) throws IOException {
        File tmpFile;
        if (statusCode != 200) {
            String bodyString = body.readUtf8();
            DebugServerException debugServerException = DebugServerException.parse(url, bodyString);
            if (debugServerException != null) {
                callback.onFailure(debugServerException);
            } else {
                StringBuilder sb = new StringBuilder();
                sb.append("The development server returned response error code: ").append(statusCode).append("\n\n").append("URL: ").append(url).append("\n\n").append("Body:\n").append(bodyString);
                callback.onFailure(new DebugServerException(sb.toString()));
            }
            return;
        }
        if (bundleInfo != null) {
            BundleDownloader.populateBundleInfo(url, headers, bundleInfo);
        }
        if (BundleDownloader.storePlainJSInFile(body, tmpFile = new File(outputFile.getPath() + ".tmp")) && !tmpFile.renameTo(outputFile)) {
            throw new IOException("Couldn't rename " + tmpFile + " to " + outputFile);
        }
        callback.onSuccess();
    }

    private static boolean storePlainJSInFile(BufferedSource body, File outputFile) throws IOException {
        try (Sink output = null;){
            output = Okio.sink((File)outputFile);
            body.readAll(output);
        }
        return true;
    }

    private static void populateBundleInfo(String url, Headers headers, BundleInfo bundleInfo) {
        bundleInfo.mUrl = url;
        String filesChangedCountStr = headers.get("X-Metro-Files-Changed-Count");
        if (filesChangedCountStr != null) {
            try {
                bundleInfo.mFilesChangedCount = Integer.parseInt(filesChangedCountStr);
            }
            catch (NumberFormatException e) {
                bundleInfo.mFilesChangedCount = -2;
            }
        }
    }

    public static class BundleInfo {
        @Nullable
        private String mUrl;
        private int mFilesChangedCount;

        @Nullable
        public static BundleInfo fromJSONString(String jsonStr) {
            if (jsonStr == null) {
                return null;
            }
            BundleInfo info = new BundleInfo();
            try {
                JSONObject obj = new JSONObject(jsonStr);
                info.mUrl = obj.getString("url");
                info.mFilesChangedCount = obj.getInt("filesChangedCount");
            }
            catch (JSONException e) {
                FLog.e((String)BundleDownloader.TAG, (String)"Invalid bundle info: ", (Throwable)e);
                return null;
            }
            return info;
        }

        @Nullable
        public String toJSONString() {
            JSONObject obj = new JSONObject();
            try {
                obj.put("url", (Object)this.mUrl);
                obj.put("filesChangedCount", this.mFilesChangedCount);
            }
            catch (JSONException e) {
                FLog.e((String)BundleDownloader.TAG, (String)"Can't serialize bundle info: ", (Throwable)e);
                return null;
            }
            return obj.toString();
        }

        public String getUrl() {
            return this.mUrl != null ? this.mUrl : "unknown";
        }

        public int getFilesChangedCount() {
            return this.mFilesChangedCount;
        }
    }
}

