package com.feingto.cloud.cache;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * 缓存接口
 *
 * @author longfei
 */
public interface ICache {
    /**
     * 获取全部值
     *
     * @return T
     */
    <T> Map<String, T> get();

    /**
     * 获取指定键的值
     *
     * @param key 键
     * @return T
     */
    <T> T get(String key);

    /**
     * 获取满足pattern表达式条件的所有元素
     *
     * @param pattern 表达式
     * @return Map<String, T>
     */
    <T> Map<String, T> match(String pattern);

    /**
     * 设置键值
     *
     * @param key   键
     * @param value 值
     */
    void put(String key, Object value);

    /**
     * 设置键值有效期
     *
     * @param key        键
     * @param value      值
     * @param expireTime 过期时间，单位毫秒
     */
    void put(String key, Object value, long expireTime);

    /**
     * 设置键值有效期
     *
     * @param key        键
     * @param value      值
     * @param expireTime 过期时间
     * @param timeUnit   过期时间单位
     */
    void put(String key, Object value, long expireTime, TimeUnit timeUnit);

    /**
     * 获取所有缓存键
     *
     * @return Set<String>
     */
    Set<String> keys();

    /**
     * 获取满足pattern表达式条件的所有缓存键
     *
     * @param pattern 表达式
     * @return Set<String>
     */
    Set<String> scan(String pattern);

    /**
     * 是否存在
     *
     * @param key 键
     * @return boolean
     */
    boolean has(String key);

    /**
     * 移除
     *
     * @param key 键
     */
    void remove(String key);

    /**
     * 根据前缀移除
     *
     * @param prefix 键前缀
     */
    void removeByPrefix(String prefix);

    /**
     * 清空缓存
     */
    void clear();
}
