package com.feingto.cloud.cache;

import java.util.Collection;
import java.util.List;

/**
 * List 缓存接口
 *
 * @author longfei
 */
public interface IListCache<T> {
    /**
     * 获取指定索引位置的值
     *
     * @param key   键
     * @param index 下标
     * @return T
     */
    T get(String key, long index);

    /**
     * 获取范围值
     *
     * @param key   键
     * @param start 开始位置
     * @param end   结束位置
     * @return List<T>
     */
    List<T> range(String key, long start, long end);

    /**
     * 进栈
     *
     * @param key   键
     * @param value 值
     */
    void leftPush(String key, T value);

    /**
     * 进栈
     *
     * @param key    键
     * @param values 集合值
     */
    void leftPush(String key, Collection<T> values);

    /**
     * 入队列
     *
     * @param key   键
     * @param value 值
     */
    void push(String key, T value);

    /**
     * 入队列
     *
     * @param key    键
     * @param values 集合值
     */
    void push(String key, Collection<T> values);

    /**
     * 出栈
     *
     * @param key 键
     */
    void pop(String key);

    /**
     * 出队列
     *
     * @param key 键
     */
    void rightPop(String key);

    /**
     * 设置列表中指定下标的值
     *
     * @param key   键
     * @param index 下标
     * @param value 值
     */
    void set(String key, long index, T value);

    /**
     * 删除列表中值为 value 的所有元素
     *
     * @param key   键
     * @param value 值
     */
    void remove(String key, T value);

    /**
     * 删除列表中值为 value 的元素，总共删除 count 次
     *
     * @param key   键
     * @param count 数量
     * @param value 值
     */
    void remove(String key, long count, T value);

    /**
     * 截取列表，保留 [start, end] 之间的值
     *
     * @param key   键
     * @param start 开始位置
     * @param end   结束位置
     */
    void trim(String key, long start, long end);

    /**
     * 返回列表的长度
     *
     * @param key 键
     * @return Long
     */
    Long size(String key);

    /**
     * 清空列表
     */
    void clear(String key);
}
