package com.feingto.cloud.cache;

import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * Set 缓存接口
 *
 * @author longfei
 */
public interface ISetCache<T> {
    /**
     * 添加
     *
     * @param key   键
     * @param value 值
     */
    void add(String key, T value);

    /**
     * 添加多个
     *
     * @param key    键
     * @param values 集合值
     */
    void add(String key, Collection<T> values);

    /**
     * 删除值为 value 的所有元素
     *
     * @param key   键
     * @param value 值
     */
    void remove(String key, T value);

    /**
     * 删除值在 values 中的所有元素，返回删除元素的个数
     *
     * @param key    键
     * @param values 集合值
     * @return Long
     */
    Long remove(String key, Collection<T> values);

    /**
     * 从 key 中随机删除一个元素并返回删除的元素
     *
     * @param key 键
     * @return T
     */
    T pop(String key);

    /**
     * 从 key 中随机删除 count 个元素并返回删除的元素
     *
     * @param key   键
     * @param count 数量
     * @return List<T>
     */
    List<T> pop(String key, long count);

    /**
     * 从 key 集合中移动 value 到 destKey 集合
     *
     * @param key     键
     * @param value   值
     * @param destKey 目标键
     * @return boolean
     */
    Boolean move(String key, T value, String destKey);

    /**
     * 返回集合的长度
     *
     * @param key 键
     * @return Long
     */
    Long size(String key);

    /**
     * 判断是否存在
     *
     * @param key   键
     * @param value 值
     * @return boolean
     */
    Boolean contains(String key, T value);

    /**
     * 获取 key 的全部值
     *
     * @param key 键
     * @return Set<T>
     */
    Set<T> values(String key);

    /**
     * 获取 key 的随机值
     *
     * @param key 键
     * @return T
     */
    T randomValue(String key);

    /**
     * 获取 count 个 key 的随机值
     *
     * @param key   键
     * @param count 数量
     * @return List<T>
     */
    List<T> randomValues(String key, long count);

    /**
     * 获取 count 个 key 的不重复随机值
     *
     * @param key   键
     * @param count 数量
     * @return Set<T>
     */
    Set<T> distinctRandomValues(String key, long count);

    /**
     * 获取 key 和 otherKey 值的交集
     *
     * @param key      键
     * @param otherKey 其他键
     * @return Set<T>
     */
    Set<T> intersect(String key, String otherKey);

    /**
     * 获取 key 和 otherKeys 值的交集
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @return Set<T>
     */
    Set<T> intersect(String key, Collection<String> otherKeys);

    /**
     * 获取 keys 值的交集
     *
     * @param keys 键集合
     * @return Set<T>
     */
    Set<T> intersect(Collection<String> keys);

    /**
     * 获取 key 和 otherKeys 值的交集，并存在 destKey 键下
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @param destKey   目标键
     * @return Long
     */
    Long intersectAndStore(String key, Collection<String> otherKeys, String destKey);

    /**
     * 获取 keys 值的交集，并存在 destKey 键下
     *
     * @param keys    键集合
     * @param destKey 目标键
     * @return Long
     */
    Long intersectAndStore(Collection<String> keys, String destKey);

    /**
     * 获取 key 和 otherKey 值的并集
     *
     * @param key      键
     * @param otherKey 其他键
     * @return Set<T>
     */
    Set<T> union(String key, String otherKey);

    /**
     * 获取 key 和 otherKeys 值的并集
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @return Set<T>
     */
    Set<T> union(String key, Collection<String> otherKeys);

    /**
     * 获取 keys 值的并集
     *
     * @param keys 键集合
     * @return Set<T>
     */
    Set<T> union(Collection<String> keys);

    /**
     * 获取 key 和 otherKeys 值的并集，并存在 destKey 键下
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @param destKey   目标键
     * @return Long
     */
    Long unionAndStore(String key, Collection<String> otherKeys, String destKey);

    /**
     * 获取 keys 值的并集，并存在 destKey 键下
     *
     * @param keys    键集合
     * @param destKey 目标键
     * @return Long
     */
    Long unionAndStore(Collection<String> keys, String destKey);

    /**
     * 获取 key 中存在，otherKey 中不存在的值
     *
     * @param key      键
     * @param otherKey 其他键
     * @return Set<T>
     */
    Set<T> difference(String key, String otherKey);

    /**
     * 获取 key 中存在，otherKeys 中不存在的值
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @return Set<T>
     */
    Set<T> difference(String key, Collection<String> otherKeys);

    /**
     * 获取 keys 中不同的值
     *
     * @param keys 键集合
     * @return Set<T>
     */
    Set<T> difference(Collection<String> keys);

    /**
     * 获取 key 中存在，otherKey 中不存在的值，并存在 destKey 键下
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @param destKey   目标键
     * @return Long
     */
    Long differenceAndStore(String key, Collection<String> otherKeys, String destKey);

    /**
     * 获取 keys 中不同的值，并存在 destKey 键下
     *
     * @param keys    键集合
     * @param destKey 目标键
     * @return Long
     */
    Long differenceAndStore(Collection<String> keys, String destKey);

    /**
     * 获取满足pattern表达式条件的所有元素
     *
     * @param key     键
     * @param pattern 表达式
     * @return Set<T>
     */
    Set<T> scan(String key, String pattern);

    /**
     * 清空 key 集合
     */
    void clear(String key);
}
