package com.feingto.cloud.cache;

import org.springframework.data.redis.connection.RedisZSetCommands;
import org.springframework.data.redis.core.ZSetOperations;

import java.util.Collection;
import java.util.Set;

/**
 * ZSet 缓存接口
 *
 * @author longfei
 */
public interface IZSetCache<T> {
    /**
     * 添加
     *
     * @param key   键
     * @param value 值
     * @param score 分数
     */
    void add(String key, T value, double score);

    /**
     * 添加多个
     *
     * @param key    键
     * @param values 集合值
     * @param score  分数
     */
    void add(String key, Collection<T> values, double score);

    /**
     * 添加，分数递增
     *
     * @param key   键
     * @param value 值
     * @param delta 递增值
     * @return Double
     */
    Double incrementScore(String key, T value, double delta);

    /**
     * 删除值为 value 的所有元素
     *
     * @param key   键
     * @param value 值
     */
    void remove(String key, T value);

    /**
     * 删除值在 values 中的所有元素，返回删除元素的个数
     *
     * @param key    键
     * @param values 集合值
     * @return Long
     */
    Long remove(String key, Collection<T> values);

    /**
     * 获取排名
     *
     * @param key   键
     * @param value 值
     * @return Long
     */
    Long rank(String key, T value);

    /**
     * 获取倒序排名
     *
     * @param key   键
     * @param value 值
     * @return Long
     */
    Long reverseRank(String key, T value);

    /**
     * 获取范围值
     *
     * @param key   键
     * @param start 开始位置
     * @param end   结束位置
     * @return Set<T>
     */
    Set<T> range(String key, long start, long end);

    /**
     * 获取范围值和分数
     *
     * @param key   键
     * @param start 开始位置
     * @param end   结束位置
     * @return Set<ZSetOperations.TypedTuple < Object>>
     */
    Set<ZSetOperations.TypedTuple<Object>> rangeWithScores(String key, long start, long end);

    /**
     * 获取分数在 [min, max] 之间的值
     *
     * @param key 键
     * @param min 最小分
     * @param max 最大分
     * @return Set<T>
     */
    Set<T> rangeByScore(String key, double min, double max);

    /**
     * 获取分数在 [min, max] 之间的值和分数
     *
     * @param key 键
     * @param min 最小分
     * @param max 最大分
     * @return Set<ZSetOperations.TypedTuple < Object>>
     */
    Set<ZSetOperations.TypedTuple<Object>> rangeByScoreWithScores(String key, double min, double max);

    /**
     * 获取 count 个 分数在 [min, max] 之间的值
     *
     * @param key    键
     * @param min    最小分
     * @param max    最大分
     * @param offset offset
     * @param count  数量
     * @return Set<T>
     */
    Set<T> rangeByScore(String key, double min, double max, long offset, long count);

    /**
     * 获取 count 个 分数在 [min, max] 之间的值和分数
     *
     * @param key    键
     * @param min    最小分
     * @param max    最大分
     * @param offset offset
     * @param count  数量
     * @return Set<ZSetOperations.TypedTuple < Object>>
     */
    Set<ZSetOperations.TypedTuple<Object>> rangeByScoreWithScores(String key, double min, double max, long offset, long count);

    /**
     * 获取范围值，倒序排名
     *
     * @param key   键
     * @param start 开始位置
     * @param end   结束位置
     * @return Set<T>
     */
    Set<T> reverseRange(String key, long start, long end);

    /**
     * 获取范围值和分数，倒序排名
     *
     * @param key   键
     * @param start 开始位置
     * @param end   结束位置
     * @return Set<ZSetOperations.TypedTuple < Object>>
     */
    Set<ZSetOperations.TypedTuple<Object>> reverseRangeWithScores(String key, long start, long end);

    /**
     * 获取分数在 [min, max] 之间的值，倒序排名
     *
     * @param key 键
     * @param min 最小分
     * @param max 最大分
     * @return Set<T>
     */
    Set<T> reverseRangeByScore(String key, double min, double max);

    /**
     * 获取分数在 [min, max] 之间的值和分数，倒序排名
     *
     * @param key 键
     * @param min 最小分
     * @param max 最大分
     * @return Set<ZSetOperations.TypedTuple < Object>>
     */
    Set<ZSetOperations.TypedTuple<Object>> reverseRangeByScoreWithScores(String key, double min, double max);

    /**
     * 获取 count 个 分数在 [min, max] 之间的值，倒序排名
     *
     * @param key    键
     * @param min    最小分
     * @param max    最大分
     * @param offset offset
     * @param count  数量
     * @return Set<T>
     */
    Set<T> reverseRangeByScore(String key, double min, double max, long offset, long count);

    /**
     * 获取 count 个 分数在 [min, max] 之间的值和分数，倒序排名
     *
     * @param key    键
     * @param min    最小分
     * @param max    最大分
     * @param offset offset
     * @param count  数量
     * @return Set<ZSetOperations.TypedTuple < Object>>
     */
    Set<ZSetOperations.TypedTuple<Object>> reverseRangeByScoreWithScores(String key, double min, double max, long offset, long count);

    /**
     * 返回集合的长度
     *
     * @param key 键
     * @return Long
     */
    Long size(String key);

    /**
     * 获取集合 value 的分数
     *
     * @param key   键
     * @param value 值
     * @return Double
     */
    Double score(String key, T value);

    /**
     * 删除 [start, end] 之间的值
     *
     * @param key   键
     * @param start 开始位置
     * @param end   结束位置
     * @return Long
     */
    Long removeRange(String key, long start, long end);

    /**
     * 删除分数在 [min, max] 之间的值
     *
     * @param key 键
     * @param min 最小分
     * @param max 最大分
     * @return Long
     */
    Long removeRangeByScore(String key, double min, double max);

    /**
     * 获取 key 和 otherKey 值的并集，并存在 destKey 键下
     *
     * @param key      键
     * @param otherKey 其他键
     * @param destKey  目标键
     * @return Long
     */
    Long unionAndStore(String key, String otherKey, String destKey);

    /**
     * 获取 key 和 otherKeys 值的并集，并存在 destKey 键下
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @param destKey   目标键
     * @return Long
     */
    Long unionAndStore(String key, Collection<String> otherKeys, String destKey);

    /**
     * 获取 key 和 otherKeys 值的并集并联合排序，并存在 destKey 键下
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @param destKey   目标键
     * @param aggregate RedisZSetCommands.Aggregate
     * @return Long
     */
    Long unionAndStore(String key, Collection<String> otherKeys, String destKey, RedisZSetCommands.Aggregate aggregate);

    /**
     * 获取 key 和 otherKeys 值的并集并联合排序，并存在 destKey 键下
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @param destKey   目标键
     * @param aggregate RedisZSetCommands.Aggregate
     * @param weights   RedisZSetCommands.Weights
     * @return Long
     */
    Long unionAndStore(String key, Collection<String> otherKeys, String destKey, RedisZSetCommands.Aggregate aggregate, RedisZSetCommands.Weights weights);

    /**
     * 获取 key 和 otherKey 值的交集，并存在 destKey 键下
     *
     * @param key      键
     * @param otherKey 其他键
     * @param destKey  目标键
     * @return Long
     */
    Long intersectAndStore(String key, String otherKey, String destKey);

    /**
     * 获取 key 和 otherKeys 值的交集，并存在 destKey 键下
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @param destKey   目标键
     * @return Long
     */
    Long intersectAndStore(String key, Collection<String> otherKeys, String destKey);

    /**
     * 获取 key 和 otherKeys 值的交集并联合排序，并存在 destKey 键下
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @param destKey   目标键
     * @param aggregate RedisZSetCommands.Aggregate
     * @return Long
     */
    Long intersectAndStore(String key, Collection<String> otherKeys, String destKey, RedisZSetCommands.Aggregate aggregate);

    /**
     * 获取 key 和 otherKeys 值的交集并联合排序，并存在 destKey 键下
     *
     * @param key       键
     * @param otherKeys 其他键集合
     * @param destKey   目标键
     * @param aggregate RedisZSetCommands.Aggregate
     * @param weights   RedisZSetCommands.Weights
     * @return Long
     */
    Long intersectAndStore(String key, Collection<String> otherKeys, String destKey, RedisZSetCommands.Aggregate aggregate, RedisZSetCommands.Weights weights);

    /**
     * 获取满足pattern表达式条件的所有元素
     *
     * @param key     键
     * @param pattern 表达式
     * @return Set<T>
     */
    Set<T> scan(String key, String pattern);

    /**
     * 清空 key 集合
     */
    void clear(String key);
}
