package com.feingto.cloud.cache.provider;

import com.feingto.cloud.cache.ICache;
import com.feingto.cloud.cache.RedisManager;
import org.springframework.data.redis.connection.DataType;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;

import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Redis 字符串缓存
 *
 * @author longfei
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public class RedisProvider implements ICache {
    private static RedisManager redisManager;

    public RedisProvider(RedisTemplate template) {
        redisManager = new RedisManager().setTemplate(template);
    }

    @Override
    public Map<String, Object> get() {
        return match("*");
    }

    @Override
    public <T> T get(String key) {
        return (T) redisManager.getValueStore().get(key);
    }

    @Override
    public <T> Map<String, T> match(String pattern) {
        return scan(pattern).stream()
                .collect(Collectors.toMap(k -> k, this::get));
    }

    @Override
    public void put(String key, Object value) {
        redisManager.getValueStore().set(key, value);
    }

    @Override
    public void put(String key, Object value, long expireTime) {
        this.put(key, value, expireTime, TimeUnit.MILLISECONDS);
    }

    @Override
    public void put(String key, Object value, long expireTime, TimeUnit timeUnit) {
        redisManager.getValueStore().set(key, value, expireTime, timeUnit);
    }

    @Override
    public Set<String> keys() {
        return scan("*");
    }

    @Override
    public Set<String> scan(String pattern) {
        Set<String> keys = new HashSet<>();
        redisManager.scan(pattern, item -> {
            String key = new String(item, StandardCharsets.UTF_8);
            if (DataType.STRING.equals(redisManager.getTemplate().type(key))) {
                keys.add(key);
            }
        });
        return keys;
    }

    @Override
    public boolean has(String key) {
        return redisManager.getTemplate().hasKey(key);
    }

    @Override
    public void remove(String key) {
        redisManager.getTemplate().delete(key);
    }

    @Override
    public void removeByPrefix(String prefix) {
        Optional.ofNullable(scan(prefix.concat("*")))
                .ifPresent(keys -> keys.forEach(this::remove));
    }

    @Override
    public void clear() {
        redisManager.getTemplate().execute((RedisCallback<Object>) connection -> {
            connection.flushDb();
            return true;
        });
    }
}
