package com.feingto.cloud.cache.support;

import com.fasterxml.jackson.databind.JsonNode;
import com.feingto.cloud.kit.json.JSON;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.lang.Nullable;
import org.springframework.web.client.DefaultResponseErrorHandler;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.client.UnknownHttpStatusCodeException;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * 自定义 RestTemplate 异常
 *
 * @author longfei
 */
public class RestResponseErrorHandler extends DefaultResponseErrorHandler {
    @Override
    public void handleError(ClientHttpResponse response) throws IOException {
        HttpStatus statusCode = HttpStatus.resolve(response.getRawStatusCode());
        if (statusCode == null) {
            byte[] body = getResponseBody(response);
            String message = getErrorMessage(body, getCharset(response));
            throw new UnknownHttpStatusCodeException(message,
                    response.getRawStatusCode(), response.getStatusText(),
                    response.getHeaders(), body, getCharset(response));
        }
        handleError(response, statusCode);
    }

    private String getErrorMessage(@Nullable byte[] responseBody, @Nullable Charset charset) {
        charset = charset == null ? StandardCharsets.UTF_8 : charset;
        int maxChars = 200;

        if (responseBody.length < maxChars * 2) {
            return getJsonErrorMessage(new String(responseBody, charset));
        }

        try {
            Reader reader = new InputStreamReader(new ByteArrayInputStream(responseBody), charset);
            CharBuffer buffer = CharBuffer.allocate(maxChars);
            reader.read(buffer);
            reader.close();
            buffer.flip();
            return getJsonErrorMessage(buffer.toString());
        } catch (IOException ex) {
            throw new IllegalStateException(ex);
        }
    }

    private String getJsonErrorMessage(String message) {
        JsonNode jsonNode = JSON.read(message);
        return jsonNode.has("message")
                ? jsonNode.get("message").asText()
                : message;
    }

    /**
     * Handle the error based on the resolved status code
     */
    protected void handleError(ClientHttpResponse response, HttpStatus statusCode) throws IOException {
        String statusText = response.getStatusText();
        HttpHeaders headers = response.getHeaders();
        byte[] body = getResponseBody(response);
        Charset charset = getCharset(response);
        String message = getErrorMessage(body, charset);

        switch (statusCode.series()) {
            case CLIENT_ERROR:
                throw HttpClientErrorException.create(message, statusCode, statusText, headers, body, charset);
            case SERVER_ERROR:
                throw HttpServerErrorException.create(message, statusCode, statusText, headers, body, charset);
            default:
                throw new UnknownHttpStatusCodeException(message, statusCode.value(), statusText, headers, body, charset);
        }
    }
}
