package com.feingto.iot.server.bootstrap;

import com.feingto.iot.common.handler.MessageHandler;
import com.feingto.iot.server.handler.BaseMessageHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Configuration;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 处理器上下文
 *
 * @author longfei
 */
@Slf4j
@Configuration
@ConditionalOnBean(MessageHandler.class)
public class HandlerContext implements ApplicationContextAware {
    private static BaseMessageHandler handler;

    public static BaseMessageHandler getHandler() {
        return handler;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Optional.of(applicationContext.getBeansOfType(MessageHandler.class))
                .ifPresent((Map<String, MessageHandler> map) -> initHandler(map.values()));
    }

    private static void initHandler(Collection<MessageHandler> beans) {
        log.info("{} handlers have been detected", beans.size());
        AtomicInteger latch = new AtomicInteger(0);
        beans.stream()
                .filter(bean -> BaseMessageHandler.class.isAssignableFrom(bean.getClass()))
                .forEach(bean -> {
                    log.info("Registering handler with {}", bean.getClass().getSimpleName());
                    try {
                        BaseMessageHandler baseMessageHandler = (BaseMessageHandler) bean;
                        if (latch.get() == 0) {
                            handler = baseMessageHandler;
                        }
                        handler.filter(baseMessageHandler);
                        latch.getAndIncrement();
                    } catch (Exception e) {
                        log.error(e.getMessage(), e);
                    }
                });
    }
}
