package com.feingto.iot.server.cache;

import com.feingto.iot.common.model.mqtt.SendMessage;
import lombok.extern.slf4j.Slf4j;
import org.apache.ignite.IgniteCache;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Mqtt 消息队列缓存
 *
 * @author longfei
 */
@Slf4j
public class MessageCache {
    private static MessageCache instance;
    /**
     * 消息队列缓存 [clientId, [messageId, SendMessage]]
     */
    private static IgniteCache<String, ConcurrentHashMap<Integer, SendMessage>> igniteCache;

    public static MessageCache getInstance(IgniteCache<String, ConcurrentHashMap<Integer, SendMessage>> cache) {
        if (instance == null) {
            instance = new MessageCache();
            igniteCache = cache;
        }
        return instance;
    }

    /**
     * 根据clientId获取消息
     */
    public List<SendMessage> findBylientId(String clientId) {
        List<SendMessage> messages = new ArrayList<>();
        igniteCache.forEach(entry -> {
            if (clientId.equals(entry.getKey())) {
                messages.addAll(entry.getValue().values());
            }
        });
        log.debug(">>> client: [{}] match {} messages", clientId, messages.size());
        return messages;
    }

    /**
     * 根据messageId获取客户端消息
     */
    public List<Map<String, SendMessage>> findByMessageId(Integer messageId) {
        List<Map<String, SendMessage>> messages = new ArrayList<>();
        igniteCache.forEach(entry -> {
            if (entry.getValue().containsKey(messageId)) {
                Map<String, SendMessage> map = new HashMap<>();
                map.put(entry.getKey(), entry.getValue().get(messageId));
                messages.add(map);
            }
        });
        log.debug(">>> message id: [{}] match {} messages", messageId, messages.size());
        return messages;
    }

    /**
     * 根据clientId和messageId获取消息
     */
    public SendMessage get(String clientId, Integer messageId) {
        return findBylientId(clientId).stream()
                .filter(message -> messageId.equals(message.id()))
                .findFirst()
                .orElse(null);
    }

    /**
     * 保存消息
     */
    public void put(String clientId, SendMessage message) {
        Assert.notNull(clientId, "The clientId parameter cannot be empty");
        ConcurrentHashMap<Integer, SendMessage> map = igniteCache.containsKey(clientId) ?
                igniteCache.get(clientId) : new ConcurrentHashMap<>();
        // 保存一个messageId的pubrel消息, 之后再次收到同一个messageId的消息, 不做处理
        log.debug(">>> client: [{}] save message with id: [{}]", clientId, message.id());
        map.putIfAbsent(message.id(), message);
        igniteCache.put(clientId, map);
    }

    /**
     * 移除clientId消息
     */
    public void remove(String clientId) {
        Assert.notNull(clientId, "The clientId parameter cannot be empty");
        igniteCache.forEach(entry -> {
            if (clientId.equals(entry.getKey())) {
                igniteCache.remove(clientId);
            }
        });
    }

    /**
     * 移除clientId的messageId消息
     */
    public void remove(String clientId, Integer messageId) {
        Assert.notNull(clientId, "The clientId parameter cannot be empty");
        Assert.notNull(messageId, "The messageId parameter cannot be empty");
        Optional.ofNullable(igniteCache.get(clientId))
                .ifPresent(map -> {
                    map.remove(messageId);
                    if (map.size() > 0) {
                        igniteCache.put(clientId, map);
                    } else {
                        igniteCache.remove(clientId);
                    }
                });
    }
}
