package com.feingto.iot.server.config;

import com.feingto.iot.server.config.properties.ElasticSearchProperties;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsRequest;
import org.elasticsearch.client.Requests;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.transport.client.PreBuiltTransportClient;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.io.IOException;
import java.net.InetAddress;

import static org.elasticsearch.common.xcontent.XContentFactory.jsonBuilder;

/**
 * ElasticSearch Client 自动注入配置
 */
@Slf4j
@Configuration
public class ElasticSearchAutoConfiguration {
    /**
     * 生成消息当天的indexName
     */
    @Bean
    public String spanIndexName() {
        return getSpanIndexName();
    }

    @Bean("elasticSearchProperties")
    public ElasticSearchProperties config() {
        return new ElasticSearchProperties();
    }

    @Bean
    public TransportClient transportClient() {
        TransportClient client = new PreBuiltTransportClient(Settings.builder()
                .put("cluster.name", config().getClusterName())
                .build());
        log.info("Elasticsearch cluster nodes: [{}]", config().getClusterNodes());
        String[] clusterNodes = config().getClusterNodes().split(",", -1);
        String[] node;
        try {
            for (String str : clusterNodes) {
                node = str.split(":");
                client.addTransportAddress(new TransportAddress(InetAddress.getByName(node[0]), Integer.valueOf(node[1])));
            }

            if (!client.admin().indices()
                    .exists(new IndicesExistsRequest(spanIndexName())).actionGet()
                    .isExists()) {
                // 创建索引结构
                client.admin().indices()
                        .prepareCreate(spanIndexName())
                        .execute().actionGet();
                client.admin().indices()
                        .putMapping(Requests.putMappingRequest(spanIndexName()).type("mqtt").source(builder()))
                        .actionGet();
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return client;
    }

    public static String getSpanIndexName() {
        return "im:span:" + new DateTime(DateTimeZone.UTC).toString("yyyy-MM-dd");
    }

    public static XContentBuilder builder() throws IOException {
        // @formatter:off
        return jsonBuilder()
                .startObject()
                    .startObject("mqtt")
                        .startObject("properties")
                            .startObject("from")
                                .field("type", "keyword")
                            .endObject()
                            .startObject("to")
                                .field("type", "keyword")
                            .endObject()
                            .startObject("topic")
                                .field("type", "keyword")
                            .endObject()
                            .startObject("payload")
                                .field("type", "keyword")
                            .endObject()
                            .startObject("status")
                                .field("type", "keyword")
                            .endObject()
                            .startObject("timestamp")
                                .field("type", "date")
                            .endObject()
                        .endObject()
                    .endObject()
                .endObject();
        // @formatter:on
    }
}
