package com.feingto.iot.server.config;

import com.feingto.iot.server.config.properties.IgniteProperties;
import org.apache.ignite.Ignite;
import org.apache.ignite.IgniteCache;
import org.apache.ignite.IgniteMessaging;
import org.apache.ignite.Ignition;
import org.apache.ignite.cache.CacheMode;
import org.apache.ignite.configuration.CacheConfiguration;
import org.apache.ignite.configuration.DataRegionConfiguration;
import org.apache.ignite.configuration.DataStorageConfiguration;
import org.apache.ignite.configuration.IgniteConfiguration;
import org.apache.ignite.logger.slf4j.Slf4jLogger;
import org.apache.ignite.spi.checkpoint.cache.CacheCheckpointSpi;
import org.apache.ignite.spi.discovery.tcp.TcpDiscoverySpi;
import org.apache.ignite.spi.discovery.tcp.ipfinder.multicast.TcpDiscoveryMulticastIpFinder;
import org.apache.ignite.spi.discovery.tcp.ipfinder.vm.TcpDiscoveryVmIpFinder;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.annotation.Resource;
import java.util.Arrays;

/**
 * Ignite 自动注入配置
 *
 * @author longfei
 */
@Configuration
@EnableConfigurationProperties(IgniteProperties.class)
public class IgniteAutoConfiguration {
    private static final String DEFAULT_CACHE_NAME = "iot-cache-checkpoint";
    private static final String CACHE_REGION = "iot-cache-region";
    private static final String DATA_REGION = "iot-data-region";
    private static final String CACHE_SUBSCRIBE = "iot-cache-subscribe";
    private static final String CACHE_RETAINED = "iot-cache-retained";
    private static final String CACHE_MESSAGE = "iot-cache-message";

    @Resource
    private IgniteProperties properties;

    @Bean("igniteCache")
    public Ignite ignite() {
        IgniteProperties.Broker brokerConfig = properties.getBroker();
        CacheCheckpointSpi spi = new CacheCheckpointSpi();
        spi.setCacheName(DEFAULT_CACHE_NAME);

        // Non-persistent data regions
        DataRegionConfiguration notPersistenceRegion = new DataRegionConfiguration()
                .setPersistenceEnabled(false)
                .setName(CACHE_REGION);
        if (properties.getNotPersistenceInitialSize() > 0) {
            notPersistenceRegion.setInitialSize(properties.getNotPersistenceInitialSize() * 1024 * 1024);
        }
        if (properties.getNotPersistenceMaxSize() > 0) {
            notPersistenceRegion.setMaxSize(properties.getNotPersistenceMaxSize() * 1024 * 1024);
        }

        // Persistent data regions
        DataRegionConfiguration persistenceRegion = new DataRegionConfiguration()
                .setPersistenceEnabled(true)
                .setName(DATA_REGION);
        if (properties.getPersistenceInitialSize() > 0) {
            persistenceRegion.setInitialSize(properties.getPersistenceInitialSize() * 1024 * 1024);
        }
        if (properties.getPersistenceMaxSize() > 0) {
            persistenceRegion.setMaxSize(properties.getPersistenceMaxSize() * 1024 * 1024);
        }

        TcpDiscoverySpi discoverySpi = new TcpDiscoverySpi().setLocalPortRange(5);
        if (brokerConfig.isEnableMulticastGroup()) {
            // 组播
            discoverySpi.setIpFinder(new TcpDiscoveryMulticastIpFinder()
                    .setMulticastGroup(brokerConfig.getMulticastGroup()));
        } else {
            // 静态IP
            discoverySpi.setIpFinder(new TcpDiscoveryVmIpFinder()
                    .setAddresses(Arrays.asList(brokerConfig.getStaticIpAddresses())));
        }

        Ignite ignite = Ignition.start(new IgniteConfiguration()
                .setCheckpointSpi(spi)
                .setIgniteInstanceName(brokerConfig.getName())
                .setGridLogger(new Slf4jLogger(LoggerFactory.getLogger("org.apache.ignite")))
                .setSystemThreadPoolSize(properties.getThreadPoolSize())
                // Persistent data disk storage path
                .setDataStorageConfiguration(new DataStorageConfiguration()
                        .setDefaultDataRegionConfiguration(notPersistenceRegion)
                        .setDataRegionConfigurations(persistenceRegion)
                        .setStoragePath(properties.getPersistenceStorePath()))
                .setDiscoverySpi(discoverySpi));
        ignite.cluster().active(true);

        return ignite;
    }

    @Bean("igniteMessaging")
    public IgniteMessaging igniteMessaging(Ignite ignite) {
        return ignite.message(ignite.cluster().forRemotes());
    }

    /**
     * 缓存配置: REPLICATED: 复制模式，写入慢，读取快; PARTITIONED: 分区模式，写入快，读取慢;
     */
    @Bean("igniteSubscribe")
    public IgniteCache igniteSubscribe(Ignite ignite) {
        return ignite.getOrCreateCache(new CacheConfiguration<String, Object>()
                .setDataRegionName(DATA_REGION)
                .setCacheMode(CacheMode.PARTITIONED)
                .setName(CACHE_SUBSCRIBE));
    }

    @Bean("igniteRetained")
    public IgniteCache igniteRetained(Ignite ignite) {
        return ignite.getOrCreateCache(new CacheConfiguration<String, Object>()
                .setDataRegionName(DATA_REGION)
                .setCacheMode(CacheMode.PARTITIONED)
                .setName(CACHE_RETAINED));
    }

    @Bean("igniteMessage")
    public IgniteCache igniteMessage(Ignite ignite) {
        return ignite.getOrCreateCache(new CacheConfiguration<String, Object>()
                .setDataRegionName(DATA_REGION)
                .setCacheMode(CacheMode.PARTITIONED)
                .setName(CACHE_MESSAGE));
    }
}
