package com.feingto.iot.server.handler;

import com.feingto.iot.common.handler.DefaultSimpleChannelHandler;
import com.feingto.iot.common.model.mqtt.SendMessage;
import com.feingto.iot.common.model.mqtt.SessionStore;
import com.feingto.iot.server.bootstrap.HandlerContext;
import com.feingto.iot.server.cache.SessionCache;
import com.feingto.iot.server.service.PushService;
import io.netty.channel.ChannelHandler.Sharable;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.timeout.IdleState;
import io.netty.handler.timeout.IdleStateEvent;
import io.netty.util.AttributeKey;
import lombok.extern.slf4j.Slf4j;

import java.util.Optional;

/**
 * Mqtt 消息处理器
 *
 * @author longfei
 */
@Slf4j
@Sharable
public class MqttMessageHandler extends DefaultSimpleChannelHandler<MqttMessage> {
    private final PushService pushService;

    public MqttMessageHandler(PushService pushService) {
        this.pushService = pushService;
    }

    @Override
    public void handleMessage(ChannelHandlerContext ctx, MqttMessage msg) {
        MqttFixedHeader header = msg.fixedHeader();
        if (header == null) {
            return;
        }

        HandlerContext.getHandler().proceed(header.messageType(), ctx.channel(), msg);
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        if (evt instanceof IdleStateEvent) {
            IdleStateEvent idleStateEvent = (IdleStateEvent) evt;
            if (idleStateEvent.state() == IdleState.ALL_IDLE) {
                String clientId = (String) ctx.channel().attr(AttributeKey.valueOf("clientId")).get();
                // 发布遗嘱消息
                Optional.ofNullable(SessionCache.getInstance().get(clientId))
                        .filter(sess -> sess.willMessage() != null)
                        .map(SessionStore::willMessage)
                        .ifPresent(msg -> {
                            log.debug(">>> publish will message to {}", clientId);
                            pushService.internalSend(SendMessage.newInstance(msg));
                        });
                ctx.close();
            }
        } else {
            super.userEventTriggered(ctx, evt);
        }
    }
}
