package com.feingto.iot.server.handler.mqtt;

import com.feingto.cloud.kit.IdGenerator;
import com.feingto.iot.common.Constants;
import com.feingto.iot.common.model.mqtt.ImSpan;
import com.feingto.iot.common.model.mqtt.SendMessage;
import com.feingto.iot.common.model.mqtt.SubscribeMessage;
import com.feingto.iot.common.service.mqtt.MessageResponse;
import com.feingto.iot.server.cache.MessageCache;
import com.feingto.iot.server.cache.RetainedCache;
import com.feingto.iot.server.cache.SubscribeCache;
import com.feingto.iot.server.handler.BaseMessageHandler;
import com.feingto.iot.server.serialize.JSON;
import com.feingto.iot.server.service.PushService;
import io.netty.channel.Channel;
import io.netty.handler.codec.mqtt.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.ignite.IgniteCache;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.xcontent.XContentType;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 消息发布处理器
 *
 * @author longfei
 */
@Slf4j
@Component
public class PublishHandler extends BaseMessageHandler {
    @Resource(name = "igniteSubscribe")
    private IgniteCache<String, ConcurrentHashMap<String, SubscribeMessage>> igniteSubscribe;

    @Resource(name = "igniteMessage")
    private IgniteCache<String, ConcurrentHashMap<Integer, SendMessage>> igniteMessage;

    @Resource(name = "igniteRetained")
    private IgniteCache<String, SendMessage> igniteRetained;

    @Resource
    private PushService pushService;

    @Resource
    private TransportClient client;

    @Resource
    private String spanIndexName;

    @Resource
    private IdGenerator idGenerator;

    public PublishHandler() {
        super(MqttMessageType.PUBLISH);
    }

    @Override
    public void handle(Channel channel, Object object) {
        MqttPublishMessage msg = (MqttPublishMessage) object;
        MqttFixedHeader fixedHeader = msg.fixedHeader();
        MqttPublishVariableHeader variableHeader = msg.variableHeader();
        SendMessage sendMessage = SendMessage.newInstance(msg)
                .from(channel.attr(Constants.KEY_CLIENT_ID).get());

        // 存储每个Topic的最后一条保留消息及其Qos
        if (sendMessage.retain()) {
            log.debug(">>> save retain message on {}", sendMessage.topic());
            RetainedCache.getInstance(igniteRetained).put(sendMessage.topic(), sendMessage);
        }
        sendMessage.retain(false);

        // 云端持久化
        this.store(sendMessage);

        switch (fixedHeader.qosLevel()) {
            case AT_MOST_ONCE:
                // 发布消息到所有订阅者
                pushService.internalSend(sendMessage);
                break;
            case AT_LEAST_ONCE:
                // 持久化消息
                this.storeMessage(sendMessage, MqttMessageType.PUBLISH);

                // 发布消息到所有订阅者
                pushService.internalSend(sendMessage);

                // 返回puback消息
                MessageResponse.puback(channel, MqttMessageType.PUBACK, MqttQoS.AT_MOST_ONCE, variableHeader.packetId());
                break;
            case EXACTLY_ONCE:
                // 持久化消息
                this.storeMessage(sendMessage, MqttMessageType.PUBREL);

                // 返回pubrec消息
                MessageResponse.puback(channel, MqttMessageType.PUBREC, MqttQoS.AT_LEAST_ONCE, variableHeader.packetId());
                break;
            case FAILURE:
                break;
        }
    }

    /**
     * 云端持久化
     */
    private void store(SendMessage message) {
        SubscribeCache.getInstance(igniteSubscribe).findByTopic(message.topic())
                .forEach(subscribe -> {
                    log.debug(">>> store on cloud with topic {}", message.topic());
                    this.storeOnEs(subscribe.clientId(), message);
                });
    }

    /**
     * ElasticSearch 持久化
     */
    private void storeOnEs(String toClientId, SendMessage message) {
        if (toClientId.equals(message.from())) {
            return;
        }

        client.prepareIndex(spanIndexName, "mqtt")
                .setSource(JSON.getInstance().obj2json(new ImSpan()
                        .id(idGenerator.nextId())
                        .from(message.from())
                        .to(toClientId)
                        .topic(message.topic())
                        .payload(new String(message.payload(), StandardCharsets.UTF_8))), XContentType.JSON)
                .get();
    }

    /**
     * 持久化消息
     */
    private void storeMessage(SendMessage message, MqttMessageType type) {
        SubscribeCache.getInstance((igniteSubscribe)).findByTopic(message.topic())
                .forEach(subscribe -> MessageCache.getInstance(igniteMessage)
                        .put(subscribe.clientId(), message.type(type)));
    }
}
