package com.feingto.iot.server.handler.mqtt;

import com.feingto.iot.common.Constants;
import com.feingto.iot.common.model.mqtt.SendMessage;
import com.feingto.iot.common.model.mqtt.SubscribeMessage;
import com.feingto.iot.common.service.mqtt.MessageResponse;
import com.feingto.iot.server.cache.RetainedCache;
import com.feingto.iot.server.cache.SubscribeCache;
import com.feingto.iot.server.handler.BaseMessageHandler;
import com.feingto.iot.server.service.PushService;
import io.netty.channel.Channel;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttSubscribeMessage;
import lombok.extern.slf4j.Slf4j;
import org.apache.ignite.IgniteCache;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * Mqtt 消息订阅处理器
 *
 * @author longfei
 */
@Slf4j
@Component
public class SubscribeHandler extends BaseMessageHandler {
    @Resource(name = "igniteSubscribe")
    private IgniteCache<String, ConcurrentHashMap<String, SubscribeMessage>> igniteSubscribe;

    @Resource(name = "igniteRetained")
    private IgniteCache<String, SendMessage> igniteRetained;

    @Resource
    private PushService pushService;

    public SubscribeHandler() {
        super(MqttMessageType.SUBSCRIBE);
    }

    @Override
    public void handle(Channel channel, Object object) {
        String clientId = channel.attr(Constants.KEY_CLIENT_ID).get();
        MqttSubscribeMessage msg = (MqttSubscribeMessage) object;
        msg.payload().topicSubscriptions().forEach(subscription ->
                SubscribeCache.getInstance(igniteSubscribe)
                        .put(subscription.topicName(), new SubscribeMessage()
                                .clientId(clientId)
                                .topicName(subscription.topicName())
                                .mqttQoS(subscription.qualityOfService().value())));

        // 返回suback消息
        MessageResponse.suback(channel, msg.payload().topicSubscriptions().stream()
                .map(subscription -> subscription.qualityOfService().value())
                .collect(Collectors.toList()), msg.variableHeader().messageId());

        // 返回保留消息
        SubscribeCache.getInstance(igniteSubscribe).findByClientId(clientId).forEach(subscribe ->
                Optional.ofNullable(RetainedCache.getInstance(igniteRetained).get(subscribe.topicName()))
                        .ifPresent(sendMessage -> {
                            log.debug(">>> publish retained message to {}", clientId);
                            pushService.internalSend(sendMessage);
                        }));
    }
}
