package com.feingto.iot.server.serialize;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.ser.FilterProvider;
import com.fasterxml.jackson.databind.ser.impl.SimpleBeanPropertyFilter;
import com.fasterxml.jackson.databind.ser.impl.SimpleFilterProvider;
import lombok.SneakyThrows;
import org.apache.commons.lang3.StringUtils;

import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

/**
 * json字符串、对象、List转换工具类
 *
 * @author longfei
 */
@SuppressWarnings("unchecked")
public class JSON {
    private static JSON instance;
    private JSONObjectMapper objectMapper = new JSONObjectMapper();

    public static JSON getInstance() {
        if (null == instance) {
            instance = new JSON();
        }
        return instance;
    }

    public JSON objectMapper(JSONObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
        return this;
    }

    /**
     * 过滤属性，配置@JsonFilter(filterName)使用
     */
    public JSON filter(String filterName, String... properties) {
        FilterProvider filterProvider = new SimpleFilterProvider().addFilter(filterName,
                SimpleBeanPropertyFilter.serializeAllExcept(properties));
        objectMapper.setFilterProvider(filterProvider);
        return this;
    }

    public ObjectNode JSONObject() {
        return objectMapper.createObjectNode();
    }

    public ArrayNode JSONArray() {
        return objectMapper.createArrayNode();
    }

    @SneakyThrows
    public JsonNode read(String json) {
        return StringUtils.isNotBlank(json) ? objectMapper.readTree(json) : JSONObject();
    }

    /**
     * byte[] read to JsonNode
     */
    @SneakyThrows
    public JsonNode read(byte[] bytes) {
        return objectMapper.readTree(bytes);
    }

    /**
     * inputStream read to JsonNode
     */
    @SneakyThrows
    public JsonNode read(InputStream inputStream) {
        return objectMapper.readTree(inputStream);
    }

    /**
     * pojo convert to json string
     */
    @SneakyThrows
    public String obj2json(Object pojo) {
        return pojo != null ? objectMapper.writeValueAsString(pojo) : "";
    }

    /**
     * json string convert to javaBean
     */
    @SneakyThrows
    public <T> T json2pojo(String json, Class<T> clazz) {
        return StringUtils.isNotBlank(json) ? objectMapper.readValue(json, clazz) : null;
    }

    /**
     * json string convert to map with javaBean
     */
    public <T> Map<String, T> json2map(String json, Class<T> clazz)
            throws Exception {
        Map<String, Map<String, Object>> map = objectMapper.readValue(json,
                new TypeReference<Map<String, T>>() {
                });
        Map<String, T> result = new HashMap<>();
        map.forEach((key, value) -> result.put(key, object2pojo(value, clazz)));
        return result;
    }

    /**
     * json string convert to list with javaBean
     */
    @SneakyThrows
    public <T> List<T> json2list(String json, Class<T> clazz) {
        return StringUtils.isNotBlank(json) ? (List<T>) objectMapper.readValue(json, getCollectionType(clazz)) : new ArrayList<>();
    }

    /**
     * object convert to javaBean
     */
    public <T> T object2pojo(Object object, Class<T> clazz) {
        return objectMapper.convertValue(object, clazz);
    }

    /**
     * object convert to list with javaBean
     */
    public Map object2Map(Object object) {
        return ((Map) objectMapper.convertValue(object,
                new TypeReference<LinkedHashMap>() {
                }));
    }

    /**
     * object convert to list with javaBean
     */
    public <T> List<T> object2list(Object object, Class<T> clazz) {
        return ((List<LinkedHashMap>) objectMapper.convertValue(object,
                new TypeReference<List<LinkedHashMap>>() {
                })).stream()
                .map(map -> object2pojo(map, clazz))
                .collect(Collectors.toList());
    }

    /**
     * format javaBean
     */
    @SneakyThrows
    public <T> String format(T entity) {
        return entity != null ? objectMapper.writerWithDefaultPrettyPrinter().writeValueAsString(entity) : "";
    }

    /**
     * 结果集合集
     */
    public String reduce(List<JsonNode> list) {
        return list.stream().reduce((l, r) -> {
            ObjectNode node = JSONObject();
            l.fields().forEachRemaining(it -> node.set(it.getKey(), it.getValue()));
            r.fields().forEachRemaining(it -> node.set(it.getKey(), it.getValue()));
            return node;
        }).orElse(JSONObject()).toString();
    }

    /**
     * 获取泛型的Collection Type
     */
    private JavaType getCollectionType(Class<?>... elementClasses) {
        return objectMapper.getTypeFactory().constructParametricType(ArrayList.class, elementClasses);
    }
}
